% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/run_schedule.R
\name{run_schedule}
\alias{run_schedule}
\title{Run a schedule}
\usage{
run_schedule(
  schedule,
  orch_frequency = "1 day",
  check_datetime = lubridate::now(tzone = "UTC"),
  resources = list(),
  run_all = FALSE,
  n_show_next = 5,
  cores = 1,
  log_file_max_bytes = 1e+06,
  quiet = FALSE,
  log_to_console = FALSE,
  log_to_file = FALSE
)
}
\arguments{
\item{schedule}{object of type MaestroSchedule created using \code{build_schedule()}}

\item{orch_frequency}{of the orchestrator, a single string formatted like "1 day", "2 weeks", "hourly", etc.}

\item{check_datetime}{datetime against which to check the running of pipelines (default is current system time in UTC)}

\item{resources}{named list of shared resources made available to pipelines as needed}

\item{run_all}{run all pipelines regardless of the schedule (default is \code{FALSE}) - useful for testing.
Does not apply to pipes with a \code{maestroSkip} tag.}

\item{n_show_next}{show the next n scheduled pipes}

\item{cores}{number of cpu cores to run if running in parallel. If > 1, \code{furrr} is used and a multisession plan must be executed in the orchestrator (see details)}

\item{log_file_max_bytes}{numeric specifying the maximum number of bytes allowed in the log file before purging the log (within a margin of error)}

\item{quiet}{silence metrics to the console (default = \code{FALSE}). Note this does not affect messages generated from pipelines when \code{log_to_console = TRUE}.}

\item{log_to_console}{whether or not to include pipeline messages, warnings, errors to the console (default = \code{FALSE}) (see Logging & Console Output section)}

\item{log_to_file}{either a boolean to indicate whether to create and append to a \code{maestro.log} or a character path to a specific log file. If \code{FALSE} or \code{NULL} it will not log to a file.}
}
\value{
MaestroSchedule object
}
\description{
Given a schedule in a \code{maestro} project, runs the pipelines that are scheduled to execute
based on the current time.
}
\details{
\subsection{Pipeline schedule logic}{

The function \code{run_schedule()} examines each pipeline in the schedule table and determines
whether it is scheduled to run at the current time using some simple time arithmetic. We assume
\code{run_schedule(schedule, check_datetime = Sys.time())}, but this need not be the case.
}

\subsection{Output}{

\code{run_schedule()} returns the same MaestroSchedule object with modified attributes. Use \code{get_status()}
to examine the status of each pipeline and use \code{get_artifacts()} to get any return values from the
pipelines as a list.
}

\subsection{Pipelines with arguments (resources)}{

If a pipeline takes an argument that doesn't include a default value, these can be supplied
in the orchestrator via \code{run_schedule(resources = list(arg1 = val))}. The name of the argument
used by the pipeline must match the name of the argument in the list. Currently, each named
resource must refer to a single object. In other words, you can't have two pipes using
the same argument but requiring different values.
}

\subsection{Running in parallel}{

Pipelines can be run in parallel using the \code{cores} argument. First, you must run \code{future::plan(future::multisession)}
in the orchestrator. Then, supply the desired number of cores to the \code{cores} argument. Note that
console output appears different in multicore mode.
}

\subsection{Logging & Console Output}{

By default, \code{maestro} suppresses pipeline messages, warnings, and errors from appearing in the console, but
messages coming from \code{print()} and other console logging packages like \code{cli} and \code{logger} are not suppressed
and will be interwoven into the output generated from \code{run_schedule()}. Messages from \code{cat()} and related functions are always suppressed
due to the nature of how those functions operate with standard output.

Users are advised to make use of R's \code{message()}, \code{warning()}, and \code{stop()} functions in their pipelines
for managing conditions. Use \code{log_to_console = TRUE} to print these to the console.

Maestro can generate a log file that is appended to each time the orchestrator is run. Use \code{log_to_file = TRUE} or \code{log_to_file = '[path-to-file]'} and
maestro will create/append to a file in the project directory.
This log file will be appended to until it exceeds the byte size defined in \code{log_file_max_bytes} argument after which
the log file is deleted.
}
}
\examples{

if (interactive()) {
  pipeline_dir <- tempdir()
  create_pipeline("my_new_pipeline", pipeline_dir, open = FALSE)
  schedule <- build_schedule(pipeline_dir = pipeline_dir)

  # Runs the schedule every 1 day
  run_schedule(
    schedule,
    orch_frequency = "1 day",
    quiet = TRUE
  )

  # Runs the schedule every 15 minutes
  run_schedule(
    schedule,
    orch_frequency = "15 minutes",
    quiet = TRUE
  )
}
}
