% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/feature-space.R
\name{layer_feature_space}
\alias{layer_feature_space}
\alias{feature_cross}
\alias{feature_custom}
\alias{feature_float}
\alias{feature_float_rescaled}
\alias{feature_float_normalized}
\alias{feature_float_discretized}
\alias{feature_integer_categorical}
\alias{feature_string_categorical}
\alias{feature_string_hashed}
\alias{feature_integer_hashed}
\title{One-stop utility for preprocessing and encoding structured data.}
\usage{
layer_feature_space(
  object,
  features,
  output_mode = "concat",
  crosses = NULL,
  crossing_dim = 32L,
  hashing_dim = 32L,
  num_discretization_bins = 32L,
  name = NULL,
  feature_names = NULL
)

feature_cross(feature_names, crossing_dim, output_mode = "one_hot")

feature_custom(dtype, preprocessor, output_mode)

feature_float(name = NULL)

feature_float_rescaled(scale = 1, offset = 0, name = NULL)

feature_float_normalized(name = NULL)

feature_float_discretized(
  num_bins,
  bin_boundaries = NULL,
  output_mode = "one_hot",
  name = NULL
)

feature_integer_categorical(
  max_tokens = NULL,
  num_oov_indices = 1,
  output_mode = "one_hot",
  name = NULL
)

feature_string_categorical(
  max_tokens = NULL,
  num_oov_indices = 1,
  output_mode = "one_hot",
  name = NULL
)

feature_string_hashed(num_bins, output_mode = "one_hot", name = NULL)

feature_integer_hashed(num_bins, output_mode = "one_hot", name = NULL)
}
\arguments{
\item{object}{see description}

\item{features}{see description}

\item{output_mode}{A string.
\itemize{
\item For \code{layer_feature_space()}, one of \code{"concat"} or \code{"dict"}. In concat mode, all
features get concatenated together into a single vector.
In dict mode, the \code{FeatureSpace} returns a named list of individually
encoded features (with the same names as the input list names).
\item For the \verb{feature_*} functions, one of: \code{"int"} \code{"one_hot"} or \code{"float"}.
}}

\item{crosses}{List of features to be crossed together, e.g.
\code{crosses=list(c("feature_1", "feature_2"))}. The features will be
"crossed" by hashing their combined value into
a fixed-length vector.}

\item{crossing_dim}{Default vector size for hashing crossed features.
Defaults to \code{32}.}

\item{hashing_dim}{Default vector size for hashing features of type
\code{"integer_hashed"} and \code{"string_hashed"}. Defaults to \code{32}.}

\item{num_discretization_bins}{Default number of bins to be used for
discretizing features of type \code{"float_discretized"}.
Defaults to \code{32}.}

\item{name}{String, name for the object}

\item{feature_names}{Named list mapping the names of your features to their
type specification, e.g. \code{list(my_feature = "integer_categorical")}
or \code{list(my_feature = feature_integer_categorical())}.
For a complete list of all supported types, see
"Available feature types" paragraph below.}

\item{dtype}{string, the output dtype of the feature. E.g., "float32".}

\item{preprocessor}{A callable.}

\item{scale, offset}{Passed on to \code{\link[=layer_rescaling]{layer_rescaling()}}}

\item{num_bins, bin_boundaries}{Passed on to \code{\link[=layer_discretization]{layer_discretization()}}}

\item{max_tokens, num_oov_indices}{Passed on to \code{\link[=layer_integer_lookup]{layer_integer_lookup()}} by \code{feature_integer_categorical()} or to \code{\link[=layer_string_lookup]{layer_string_lookup()}} by \code{feature_string_categorical()}.}
}
\value{
The return value depends on the value provided for the first argument.
If  \code{object} is:
\itemize{
\item a \code{keras_model_sequential()}, then the layer is added to the sequential model
(which is modified in place). To enable piping, the sequential model is also
returned, invisibly.
\item a \code{keras_input()}, then the output tensor from calling \code{layer(input)} is returned.
\item \code{NULL} or missing, then a \code{Layer} instance is returned.
}
}
\description{
\strong{Available feature types:}

Note that all features can be referred to by their string name,
e.g. \code{"integer_categorical"}. When using the string name, the default
argument values are used.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{# Plain float values.
feature_float(name = NULL)

# Float values to be preprocessed via featurewise standardization
# (i.e. via a `layer_normalization()` layer).
feature_float_normalized(name = NULL)

# Float values to be preprocessed via linear rescaling
# (i.e. via a `layer_rescaling` layer).
feature_float_rescaled(scale = 1., offset = 0., name = NULL)

# Float values to be discretized. By default, the discrete
# representation will then be one-hot encoded.
feature_float_discretized(
  num_bins,
  bin_boundaries = NULL,
  output_mode = "one_hot",
  name = NULL
)

# Integer values to be indexed. By default, the discrete
# representation will then be one-hot encoded.
feature_integer_categorical(
  max_tokens = NULL,
  num_oov_indices = 1,
  output_mode = "one_hot",
  name = NULL
)

# String values to be indexed. By default, the discrete
# representation will then be one-hot encoded.
feature_string_categorical(
  max_tokens = NULL,
  num_oov_indices = 1,
  output_mode = "one_hot",
  name = NULL
)

# Integer values to be hashed into a fixed number of bins.
# By default, the discrete representation will then be one-hot encoded.
feature_integer_hashed(num_bins, output_mode = "one_hot", name = NULL)

# String values to be hashed into a fixed number of bins.
# By default, the discrete representation will then be one-hot encoded.
feature_string_hashed(num_bins, output_mode = "one_hot", name = NULL)
}\if{html}{\out{</div>}}
}
\section{Examples}{
\strong{Basic usage with a named list of input data:}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{raw_data <- list(
  float_values = c(0.0, 0.1, 0.2, 0.3),
  string_values = c("zero", "one", "two", "three"),
  int_values = as.integer(c(0, 1, 2, 3))
)

dataset <- tfdatasets::tensor_slices_dataset(raw_data)

feature_space <- layer_feature_space(
  features = list(
    float_values = "float_normalized",
    string_values = "string_categorical",
    int_values = "integer_categorical"
  ),
  crosses = list(c("string_values", "int_values")),
  output_mode = "concat"
)

# Before you start using the feature_space(),
# you must `adapt()` it on some data.
feature_space |> adapt(dataset)

# You can call the feature_space() on a named list of
# data (batched or unbatched).
output_vector <- feature_space(raw_data)
}\if{html}{\out{</div>}}

\strong{Basic usage with \code{tf.data}:}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{library(tfdatasets)
# Unlabeled data
preprocessed_ds <- unlabeled_dataset |>
  dataset_map(feature_space)

# Labeled data
preprocessed_ds <- labeled_dataset |>
  dataset_map(function(x, y) tuple(feature_space(x), y))
}\if{html}{\out{</div>}}

\strong{Basic usage with the Keras Functional API:}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{# Retrieve a named list of Keras layer_input() objects
(inputs <- feature_space$get_inputs())
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## $float_values
## <KerasTensor shape=(None, 1), dtype=float32, sparse=None, name=float_values>
##
## $string_values
## <KerasTensor shape=(None, 1), dtype=string, sparse=None, name=string_values>
##
## $int_values
## <KerasTensor shape=(None, 1), dtype=int32, sparse=None, name=int_values>

}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{# Retrieve the corresponding encoded Keras tensors
(encoded_features <- feature_space$get_encoded_features())
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## <KerasTensor shape=(None, 43), dtype=float32, sparse=False, name=keras_tensor_7>

}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{# Build a Functional model
outputs <- encoded_features |> layer_dense(1, activation = "sigmoid")
model <- keras_model(inputs, outputs)
}\if{html}{\out{</div>}}

\strong{Customizing each feature or feature cross:}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{feature_space <- layer_feature_space(
  features = list(
    float_values = feature_float_normalized(),
    string_values = feature_string_categorical(max_tokens = 10),
    int_values = feature_integer_categorical(max_tokens = 10)
  ),
  crosses = list(
    feature_cross(c("string_values", "int_values"), crossing_dim = 32)
  ),
  output_mode = "concat"
)
}\if{html}{\out{</div>}}

\strong{Returning a dict (a named list) of integer-encoded features:}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{feature_space <- layer_feature_space(
  features = list(
    "string_values" = feature_string_categorical(output_mode = "int"),
    "int_values" = feature_integer_categorical(output_mode = "int")
  ),
  crosses = list(
    feature_cross(
      feature_names = c("string_values", "int_values"),
      crossing_dim = 32,
      output_mode = "int"
    )
  ),
  output_mode = "dict"
)
}\if{html}{\out{</div>}}

\strong{Specifying your own Keras preprocessing layer:}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{# Let's say that one of the features is a short text paragraph that
# we want to encode as a vector (one vector per paragraph) via TF-IDF.
data <- list(text = c("1st string", "2nd string", "3rd string"))

# There's a Keras layer for this: layer_text_vectorization()
custom_layer <- layer_text_vectorization(output_mode = "tf_idf")

# We can use feature_custom() to create a custom feature
# that will use our preprocessing layer.
feature_space <- layer_feature_space(
  features = list(
    text = feature_custom(preprocessor = custom_layer,
                          dtype = "string",
                          output_mode = "float"
    )
  ),
  output_mode = "concat"
)
feature_space |> adapt(tfdatasets::tensor_slices_dataset(data))
output_vector <- feature_space(data)
}\if{html}{\out{</div>}}

\strong{Retrieving the underlying Keras preprocessing layers:}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{# The preprocessing layer of each feature is available in `$preprocessors`.
preprocessing_layer <- feature_space$preprocessors$feature1

# The crossing layer of each feature cross is available in `$crossers`.
# It's an instance of layer_hashed_crossing()
crossing_layer <- feature_space$crossers[["feature1_X_feature2"]]
}\if{html}{\out{</div>}}

\strong{Saving and reloading a FeatureSpace:}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{feature_space$save("featurespace.keras")
reloaded_feature_space <- keras$models$load_model("featurespace.keras")
}\if{html}{\out{</div>}}
}

\seealso{
\itemize{
\item \url{https://keras.io/api/utils/feature_space#featurespace-class}
}

Other preprocessing layers: \cr
\code{\link{layer_category_encoding}()} \cr
\code{\link{layer_center_crop}()} \cr
\code{\link{layer_discretization}()} \cr
\code{\link{layer_hashed_crossing}()} \cr
\code{\link{layer_hashing}()} \cr
\code{\link{layer_integer_lookup}()} \cr
\code{\link{layer_mel_spectrogram}()} \cr
\code{\link{layer_normalization}()} \cr
\code{\link{layer_random_brightness}()} \cr
\code{\link{layer_random_contrast}()} \cr
\code{\link{layer_random_crop}()} \cr
\code{\link{layer_random_flip}()} \cr
\code{\link{layer_random_rotation}()} \cr
\code{\link{layer_random_translation}()} \cr
\code{\link{layer_random_zoom}()} \cr
\code{\link{layer_rescaling}()} \cr
\code{\link{layer_resizing}()} \cr
\code{\link{layer_string_lookup}()} \cr
\code{\link{layer_text_vectorization}()} \cr

Other layers: \cr
\code{\link{Layer}()} \cr
\code{\link{layer_activation}()} \cr
\code{\link{layer_activation_elu}()} \cr
\code{\link{layer_activation_leaky_relu}()} \cr
\code{\link{layer_activation_parametric_relu}()} \cr
\code{\link{layer_activation_relu}()} \cr
\code{\link{layer_activation_softmax}()} \cr
\code{\link{layer_activity_regularization}()} \cr
\code{\link{layer_add}()} \cr
\code{\link{layer_additive_attention}()} \cr
\code{\link{layer_alpha_dropout}()} \cr
\code{\link{layer_attention}()} \cr
\code{\link{layer_average}()} \cr
\code{\link{layer_average_pooling_1d}()} \cr
\code{\link{layer_average_pooling_2d}()} \cr
\code{\link{layer_average_pooling_3d}()} \cr
\code{\link{layer_batch_normalization}()} \cr
\code{\link{layer_bidirectional}()} \cr
\code{\link{layer_category_encoding}()} \cr
\code{\link{layer_center_crop}()} \cr
\code{\link{layer_concatenate}()} \cr
\code{\link{layer_conv_1d}()} \cr
\code{\link{layer_conv_1d_transpose}()} \cr
\code{\link{layer_conv_2d}()} \cr
\code{\link{layer_conv_2d_transpose}()} \cr
\code{\link{layer_conv_3d}()} \cr
\code{\link{layer_conv_3d_transpose}()} \cr
\code{\link{layer_conv_lstm_1d}()} \cr
\code{\link{layer_conv_lstm_2d}()} \cr
\code{\link{layer_conv_lstm_3d}()} \cr
\code{\link{layer_cropping_1d}()} \cr
\code{\link{layer_cropping_2d}()} \cr
\code{\link{layer_cropping_3d}()} \cr
\code{\link{layer_dense}()} \cr
\code{\link{layer_depthwise_conv_1d}()} \cr
\code{\link{layer_depthwise_conv_2d}()} \cr
\code{\link{layer_discretization}()} \cr
\code{\link{layer_dot}()} \cr
\code{\link{layer_dropout}()} \cr
\code{\link{layer_einsum_dense}()} \cr
\code{\link{layer_embedding}()} \cr
\code{\link{layer_flatten}()} \cr
\code{\link{layer_flax_module_wrapper}()} \cr
\code{\link{layer_gaussian_dropout}()} \cr
\code{\link{layer_gaussian_noise}()} \cr
\code{\link{layer_global_average_pooling_1d}()} \cr
\code{\link{layer_global_average_pooling_2d}()} \cr
\code{\link{layer_global_average_pooling_3d}()} \cr
\code{\link{layer_global_max_pooling_1d}()} \cr
\code{\link{layer_global_max_pooling_2d}()} \cr
\code{\link{layer_global_max_pooling_3d}()} \cr
\code{\link{layer_group_normalization}()} \cr
\code{\link{layer_group_query_attention}()} \cr
\code{\link{layer_gru}()} \cr
\code{\link{layer_hashed_crossing}()} \cr
\code{\link{layer_hashing}()} \cr
\code{\link{layer_identity}()} \cr
\code{\link{layer_integer_lookup}()} \cr
\code{\link{layer_jax_model_wrapper}()} \cr
\code{\link{layer_lambda}()} \cr
\code{\link{layer_layer_normalization}()} \cr
\code{\link{layer_lstm}()} \cr
\code{\link{layer_masking}()} \cr
\code{\link{layer_max_pooling_1d}()} \cr
\code{\link{layer_max_pooling_2d}()} \cr
\code{\link{layer_max_pooling_3d}()} \cr
\code{\link{layer_maximum}()} \cr
\code{\link{layer_mel_spectrogram}()} \cr
\code{\link{layer_minimum}()} \cr
\code{\link{layer_multi_head_attention}()} \cr
\code{\link{layer_multiply}()} \cr
\code{\link{layer_normalization}()} \cr
\code{\link{layer_permute}()} \cr
\code{\link{layer_random_brightness}()} \cr
\code{\link{layer_random_contrast}()} \cr
\code{\link{layer_random_crop}()} \cr
\code{\link{layer_random_flip}()} \cr
\code{\link{layer_random_rotation}()} \cr
\code{\link{layer_random_translation}()} \cr
\code{\link{layer_random_zoom}()} \cr
\code{\link{layer_repeat_vector}()} \cr
\code{\link{layer_rescaling}()} \cr
\code{\link{layer_reshape}()} \cr
\code{\link{layer_resizing}()} \cr
\code{\link{layer_rnn}()} \cr
\code{\link{layer_separable_conv_1d}()} \cr
\code{\link{layer_separable_conv_2d}()} \cr
\code{\link{layer_simple_rnn}()} \cr
\code{\link{layer_spatial_dropout_1d}()} \cr
\code{\link{layer_spatial_dropout_2d}()} \cr
\code{\link{layer_spatial_dropout_3d}()} \cr
\code{\link{layer_spectral_normalization}()} \cr
\code{\link{layer_string_lookup}()} \cr
\code{\link{layer_subtract}()} \cr
\code{\link{layer_text_vectorization}()} \cr
\code{\link{layer_tfsm}()} \cr
\code{\link{layer_time_distributed}()} \cr
\code{\link{layer_torch_module_wrapper}()} \cr
\code{\link{layer_unit_normalization}()} \cr
\code{\link{layer_upsampling_1d}()} \cr
\code{\link{layer_upsampling_2d}()} \cr
\code{\link{layer_upsampling_3d}()} \cr
\code{\link{layer_zero_padding_1d}()} \cr
\code{\link{layer_zero_padding_2d}()} \cr
\code{\link{layer_zero_padding_3d}()} \cr
\code{\link{rnn_cell_gru}()} \cr
\code{\link{rnn_cell_lstm}()} \cr
\code{\link{rnn_cell_simple}()} \cr
\code{\link{rnn_cells_stack}()} \cr

Other utils: \cr
\code{\link{audio_dataset_from_directory}()} \cr
\code{\link{clear_session}()} \cr
\code{\link{config_disable_interactive_logging}()} \cr
\code{\link{config_disable_traceback_filtering}()} \cr
\code{\link{config_enable_interactive_logging}()} \cr
\code{\link{config_enable_traceback_filtering}()} \cr
\code{\link{config_is_interactive_logging_enabled}()} \cr
\code{\link{config_is_traceback_filtering_enabled}()} \cr
\code{\link{get_file}()} \cr
\code{\link{get_source_inputs}()} \cr
\code{\link{image_array_save}()} \cr
\code{\link{image_dataset_from_directory}()} \cr
\code{\link{image_from_array}()} \cr
\code{\link{image_load}()} \cr
\code{\link{image_smart_resize}()} \cr
\code{\link{image_to_array}()} \cr
\code{\link{normalize}()} \cr
\code{\link{pad_sequences}()} \cr
\code{\link{set_random_seed}()} \cr
\code{\link{split_dataset}()} \cr
\code{\link{text_dataset_from_directory}()} \cr
\code{\link{timeseries_dataset_from_array}()} \cr
\code{\link{to_categorical}()} \cr
\code{\link{zip_lists}()} \cr
}
\concept{layers}
\concept{preprocessing layers}
\concept{utils}
