% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/posterior-samples.R
\name{derivative_samples}
\alias{derivative_samples}
\alias{derivative_samples.default}
\alias{derivative_samples.gamm}
\alias{derivative_samples.gam}
\title{Posterior expectations of derivatives from an estimated model}
\usage{
derivative_samples(object, ...)

\method{derivative_samples}{default}(object, ...)

\method{derivative_samples}{gamm}(object, ...)

\method{derivative_samples}{gam}(
  object,
  focal = NULL,
  data = NULL,
  order = 1L,
  type = c("forward", "backward", "central"),
  scale = c("response", "linear_predictor"),
  method = c("gaussian", "mh", "inla", "user"),
  n = 100,
  eps = 1e-07,
  n_sim = 10000,
  level = lifecycle::deprecated(),
  seed = NULL,
  envir = environment(formula(object)),
  draws = NULL,
  mvn_method = c("mvnfast", "mgcv"),
  ...
)
}
\arguments{
\item{object}{an R object to compute derivatives for}

\item{...}{arguments passed to other methods and on to \code{fitted_samples()}}

\item{focal}{character; name of the focal variable. The response derivative
of the response with respect to this variable will be returned.
All other variables involved in the model will be held at constant values.
This can be missing if supplying \code{data}, in which case, the focal variable
will be identified as the one variable that is not constant.}

\item{data}{a data frame containing the values of the model covariates
at which to evaluate the first derivatives of the smooths. If supplied,
all but one variable must be held at a constant value.}

\item{order}{numeric; the order of derivative.}

\item{type}{character; the type of finite difference used. One of
\code{"forward"}, \code{"backward"}, or \code{"central"}.}

\item{scale}{character; should the derivative be estimated on the response
or the linear predictor (link) scale? One of \code{"response"} (the default),
or \code{"linear predictor"}.}

\item{method}{character; which method should be used to draw samples from
the posterior distribution. \code{"gaussian"} uses a Gaussian (Laplace)
approximation to the posterior. \code{"mh"} uses a Metropolis Hastings sample
that alternates t proposals with proposals based on a shrunken version of
the posterior covariance matrix. \code{"inla"} uses a variant of Integrated
Nested Laplace Approximation due to Wood (2019), (currently not
implemented). \code{"user"} allows for user-supplied posterior draws
(currently not implemented).}

\item{n}{numeric; the number of points to evaluate the derivative at (if
\code{data} is not supplied).}

\item{eps}{numeric; the finite difference.}

\item{n_sim}{integer; the number of simulations used in computing the
simultaneous intervals.}

\item{level}{\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#deprecated}{\figure{lifecycle-deprecated.svg}{options: alt='[Deprecated]'}}}{\strong{[Deprecated]}}}

\item{seed}{numeric; a random seed for the simulations.}

\item{envir}{the environment within which to recreate the data used to fit
\code{object}.}

\item{draws}{matrix; user supplied posterior draws to be used when
\code{method = "user"}.}

\item{mvn_method}{character; one of \code{"mvnfast"} or \code{"mgcv"}. The default is
uses \code{mvnfast::rmvn()}, which can be considerably faster at generate large
numbers of MVN random values than \code{mgcv::rmvn()}, but which might not work
for some marginal fits, such as those where the covariance matrix is close
to singular.}
}
\value{
A tibble, currently with the following variables:
\itemize{
\item \code{.derivative}: the estimated partial derivative,
\item additional columns containing the covariate values at which the derivative
was evaluated.
}
}
\description{
Posterior expectations of derivatives from an estimated model
}
\examples{

load_mgcv()
df <- data_sim("eg1", dist = "negbin", scale = 0.25, seed = 42)

# fit the GAM (note: for execution time reasons using bam())
m <- bam(y ~ s(x0) + s(x1) + s(x2) + s(x3),
  data = df, family = nb(), method = "fREML")

# data slice through data along x2 - all other covariates will be set to
# typical values (value closest to median)
ds <- data_slice(m, x2 = evenly(x2, n = 200))

# samples from posterior of derivatives
fd_samp <- derivative_samples(m,
  data = ds, type = "central",
  focal = "x2", eps = 0.01, seed = 21, n_sim = 100
)

# plot the first 20 posterior draws
if (requireNamespace("ggplot2") && requireNamespace("dplyr")) {
  library("ggplot2")
  fd_samp |>
    dplyr::filter(.draw <= 20) |>
    ggplot(aes(x = x2, y = .derivative, group = .draw)) +
    geom_line(alpha = 0.5)
}
}
\author{
Gavin L. Simpson
}
