% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/corridor.R
\name{corridor}
\alias{corridor}
\alias{corridor,goc-method}
\title{Visualize corridors between two points using a grains of connectivity (GOC)}
\usage{
corridor(x, ...)

\S4method{corridor}{goc}(x, whichThresh, coords, weight = "meanWeight", ...)
}
\arguments{
\item{x}{A \code{goc} object created by \code{\link{GOC}}.}

\item{...}{Additional arguments (not used).}

\item{whichThresh}{Integer giving the index of the threshold to visualize.}

\item{coords}{A two column matrix or a \code{\link{SpatialPoints}} object
giving coordinates at the end points of the corridor.}

\item{weight}{The GOC graph link weight to use in calculating the distance.
Please see details in \code{\link{distance}}.}
}
\value{
An object of class \code{\linkS4class{corridor}}.
}
\description{
Given a series of GOC models built at different scales, visualize the corridor
(or shortest path) between two points using one of the tessellations
(i.e., scales) in these models.
}
\examples{
## Load raster landscape
tiny <- raster::raster(system.file("extdata/tiny.asc", package = "grainscape"))

## Create a resistance surface from a raster using an is-becomes reclassification
tinyCost <- raster::reclassify(tiny, rcl = cbind(c(1, 2, 3, 4), c(1, 5, 10, 12)))

## Produce a patch-based MPG where patches are resistance features=1
tinyPatchMPG <- MPG(cost = tinyCost, patch = tinyCost == 1)

## Extract a representative subset of 5 grains of connectivity
tinyPatchGOC <- GOC(tinyPatchMPG, nThresh = 5)

## Quick visualization of a corridor
corridorStartEnd <- rbind(c(10, 10), c(90, 90))
tinyPatchCorridor <- corridor(tinyPatchGOC, whichThresh = 3, coords = corridorStartEnd)
if (interactive())
  plot(tinyPatchCorridor)

## More control over a corridor visualization
if (interactive()) {
  plot(tinyPatchCorridor@voronoi, col = "lightgrey", lwd = 2)
  plot(tinyPatchCorridor@linksSP, col = "darkred", lty = "dashed", add = TRUE)
  plot(tinyPatchCorridor@nodesSP, col = "darkred", pch = 21, bg = "white", add = TRUE)
  plot(tinyPatchCorridor@shortestLinksSP, col = "darkred", lty = "solid", lwd = 2, add = TRUE)
  plot(tinyPatchCorridor@shortestNodesSP, col = "darkred", pch = 21, bg = "darkred", add = TRUE)
  mtext(paste("Corridor shortest path length:",
              round(tinyPatchCorridor@corridorLength, 2),
              "resistance units"), side = 1)
}
}
\references{
Fall, A., M.-J. Fortin, M. Manseau, D. O'Brien. (2007) Spatial graphs:
Principles and applications for habitat connectivity. Ecosystems 10:448:461.

Galpern, P., M. Manseau. (2013a) Finding the functional grain: comparing methods
for scaling resistance surfaces. Landscape Ecology 28:1269-1291.

Galpern, P., M. Manseau. (2013b) Modelling the influence of landscape connectivity
on animal distribution: a functional grain approach. Ecography 36:1004-1016.

Galpern, P., M. Manseau, A. Fall. (2011) Patch-based graphs of landscape connectivity:
a guide to construction, analysis, and application for conservation.
Biological Conservation 144:44-55.

Galpern, P., M. Manseau, P.J. Wilson. (2012) Grains of connectivity: analysis
at multiple spatial scales in landscape genetics. Molecular Ecology 21:3996-4009.
}
\seealso{
\code{\link{GOC}}, \code{\link{visualize}}
}
\author{
Paul Galpern and Alex Chubaty
}
