/*
 * Decompiled with CFR 0.152.
 */
package org.apache.commons.math.util;

import java.math.BigDecimal;

public final class MathUtils {
    private static final byte NB = -1;
    private static final short NS = -1;
    private static final byte PB = 1;
    private static final short PS = 1;
    private static final byte ZB = 0;
    private static final short ZS = 0;
    private static final double TWO_PI = Math.PI * 2;

    private MathUtils() {
    }

    public static int addAndCheck(int x, int y) {
        long s = (long)x + (long)y;
        if (s < Integer.MIN_VALUE || s > Integer.MAX_VALUE) {
            throw new ArithmeticException("overflow: add");
        }
        return (int)s;
    }

    public static long addAndCheck(long a, long b) {
        return MathUtils.addAndCheck(a, b, "overflow: add");
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    private static long addAndCheck(long a, long b, String msg) {
        if (a > b) {
            return MathUtils.addAndCheck(b, a, msg);
        }
        if (a < 0L) {
            if (b >= 0L) return a + b;
            if (Long.MIN_VALUE - b > a) throw new ArithmeticException(msg);
            return a + b;
        }
        if (a > Long.MAX_VALUE - b) throw new ArithmeticException(msg);
        return a + b;
    }

    public static long binomialCoefficient(int n, int k) {
        if (n < k) {
            throw new IllegalArgumentException("must have n >= k for binomial coefficient (n,k)");
        }
        if (n < 0) {
            throw new IllegalArgumentException("must have n >= 0 for binomial coefficient (n,k)");
        }
        if (n == k || k == 0) {
            return 1L;
        }
        if (k == 1 || k == n - 1) {
            return n;
        }
        long result = Math.round(MathUtils.binomialCoefficientDouble(n, k));
        if (result == Long.MAX_VALUE) {
            throw new ArithmeticException("result too large to represent in a long integer");
        }
        return result;
    }

    public static double binomialCoefficientDouble(int n, int k) {
        return Math.floor(Math.exp(MathUtils.binomialCoefficientLog(n, k)) + 0.5);
    }

    public static double binomialCoefficientLog(int n, int k) {
        int i;
        if (n < k) {
            throw new IllegalArgumentException("must have n >= k for binomial coefficient (n,k)");
        }
        if (n < 0) {
            throw new IllegalArgumentException("must have n >= 0 for binomial coefficient (n,k)");
        }
        if (n == k || k == 0) {
            return 0.0;
        }
        if (k == 1 || k == n - 1) {
            return Math.log(n);
        }
        double logSum = 0.0;
        for (i = k + 1; i <= n; ++i) {
            logSum += Math.log(i);
        }
        for (i = 2; i <= n - k; ++i) {
            logSum -= Math.log(i);
        }
        return logSum;
    }

    public static double cosh(double x) {
        return (Math.exp(x) + Math.exp(-x)) / 2.0;
    }

    public static boolean equals(double x, double y) {
        return Double.isNaN(x) && Double.isNaN(y) || x == y;
    }

    public static boolean equals(double[] x, double[] y) {
        if (x == null || y == null) {
            return !(x == null ^ y == null);
        }
        if (x.length != y.length) {
            return false;
        }
        for (int i = 0; i < x.length; ++i) {
            if (MathUtils.equals(x[i], y[i])) continue;
            return false;
        }
        return true;
    }

    public static long factorial(int n) {
        long result = Math.round(MathUtils.factorialDouble(n));
        if (result == Long.MAX_VALUE) {
            throw new ArithmeticException("result too large to represent in a long integer");
        }
        return result;
    }

    public static double factorialDouble(int n) {
        if (n < 0) {
            throw new IllegalArgumentException("must have n >= 0 for n!");
        }
        return Math.floor(Math.exp(MathUtils.factorialLog(n)) + 0.5);
    }

    public static double factorialLog(int n) {
        if (n < 0) {
            throw new IllegalArgumentException("must have n > 0 for n!");
        }
        double logSum = 0.0;
        for (int i = 2; i <= n; ++i) {
            logSum += Math.log(i);
        }
        return logSum;
    }

    public static int gcd(int u, int v) {
        int t;
        int k;
        if (u * v == 0) {
            return Math.abs(u) + Math.abs(v);
        }
        if (u > 0) {
            u = -u;
        }
        if (v > 0) {
            v = -v;
        }
        for (k = 0; (u & 1) == 0 && (v & 1) == 0 && k < 31; ++k) {
            u /= 2;
            v /= 2;
        }
        if (k == 31) {
            throw new ArithmeticException("overflow: gcd is 2^31");
        }
        int n = t = (u & 1) == 1 ? v : -(u / 2);
        while (true) {
            if ((t & 1) == 0) {
                t /= 2;
                continue;
            }
            if (t > 0) {
                u = -t;
            } else {
                v = t;
            }
            if ((t = (v - u) / 2) == 0) break;
        }
        return -u * (1 << k);
    }

    public static int hash(double value) {
        long bits = Double.doubleToLongBits(value);
        return (int)(bits ^ bits >>> 32);
    }

    public static int hash(double[] value) {
        if (value == null) {
            return 0;
        }
        int result = value.length;
        for (int i = 0; i < value.length; ++i) {
            result = result * 31 + MathUtils.hash(value[i]);
        }
        return result;
    }

    public static byte indicator(byte x) {
        return x >= 0 ? (byte)1 : -1;
    }

    public static double indicator(double x) {
        if (Double.isNaN(x)) {
            return Double.NaN;
        }
        return x >= 0.0 ? 1.0 : -1.0;
    }

    public static float indicator(float x) {
        if (Float.isNaN(x)) {
            return Float.NaN;
        }
        return x >= 0.0f ? 1.0f : -1.0f;
    }

    public static int indicator(int x) {
        return x >= 0 ? 1 : -1;
    }

    public static long indicator(long x) {
        return x >= 0L ? 1L : -1L;
    }

    public static short indicator(short x) {
        return x >= 0 ? (short)1 : -1;
    }

    public static int lcm(int a, int b) {
        return Math.abs(MathUtils.mulAndCheck(a / MathUtils.gcd(a, b), b));
    }

    public static double log(double base, double x) {
        return Math.log(x) / Math.log(base);
    }

    public static int mulAndCheck(int x, int y) {
        long m = (long)x * (long)y;
        if (m < Integer.MIN_VALUE || m > Integer.MAX_VALUE) {
            throw new ArithmeticException("overflow: mul");
        }
        return (int)m;
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    public static long mulAndCheck(long a, long b) {
        String msg = "overflow: multiply";
        if (a > b) {
            return MathUtils.mulAndCheck(b, a);
        }
        if (a < 0L) {
            if (b < 0L) {
                if (a < Long.MAX_VALUE / b) throw new ArithmeticException(msg);
                return a * b;
            }
            if (b <= 0L) return 0L;
            if (Long.MIN_VALUE / b > a) throw new ArithmeticException(msg);
            return a * b;
        }
        if (a <= 0L) return 0L;
        if (a > Long.MAX_VALUE / b) throw new ArithmeticException(msg);
        return a * b;
    }

    public static double nextAfter(double d, double direction) {
        if (Double.isNaN(d) || Double.isInfinite(d)) {
            return d;
        }
        if (d == 0.0) {
            return direction < 0.0 ? -4.9E-324 : Double.MIN_VALUE;
        }
        long bits = Double.doubleToLongBits(d);
        long sign = bits & Long.MIN_VALUE;
        long exponent = bits & 0x7FF0000000000000L;
        long mantissa = bits & 0xFFFFFFFFFFFFFL;
        if (d * (direction - d) >= 0.0) {
            if (mantissa == 0xFFFFFFFFFFFFFL) {
                return Double.longBitsToDouble(sign | exponent + 0x10000000000000L);
            }
            return Double.longBitsToDouble(sign | exponent | mantissa + 1L);
        }
        if (mantissa == 0L) {
            return Double.longBitsToDouble(sign | exponent - 0x10000000000000L | 0xFFFFFFFFFFFFFL);
        }
        return Double.longBitsToDouble(sign | exponent | mantissa - 1L);
    }

    public static double normalizeAngle(double a, double center) {
        return a - Math.PI * 2 * Math.floor((a + Math.PI - center) / (Math.PI * 2));
    }

    public static double round(double x, int scale) {
        return MathUtils.round(x, scale, 4);
    }

    public static double round(double x, int scale, int roundingMethod) {
        try {
            return new BigDecimal(Double.toString(x)).setScale(scale, roundingMethod).doubleValue();
        }
        catch (NumberFormatException ex) {
            if (Double.isInfinite(x)) {
                return x;
            }
            return Double.NaN;
        }
    }

    public static float round(float x, int scale) {
        return MathUtils.round(x, scale, 4);
    }

    public static float round(float x, int scale, int roundingMethod) {
        float sign = MathUtils.indicator(x);
        float factor = (float)Math.pow(10.0, scale) * sign;
        return (float)MathUtils.roundUnscaled(x * factor, sign, roundingMethod) / factor;
    }

    private static double roundUnscaled(double unscaled, double sign, int roundingMethod) {
        switch (roundingMethod) {
            case 2: {
                if (sign == -1.0) {
                    unscaled = Math.floor(MathUtils.nextAfter(unscaled, Double.NEGATIVE_INFINITY));
                    break;
                }
                unscaled = Math.ceil(MathUtils.nextAfter(unscaled, Double.POSITIVE_INFINITY));
                break;
            }
            case 1: {
                unscaled = Math.floor(MathUtils.nextAfter(unscaled, Double.NEGATIVE_INFINITY));
                break;
            }
            case 3: {
                if (sign == -1.0) {
                    unscaled = Math.ceil(MathUtils.nextAfter(unscaled, Double.POSITIVE_INFINITY));
                    break;
                }
                unscaled = Math.floor(MathUtils.nextAfter(unscaled, Double.NEGATIVE_INFINITY));
                break;
            }
            case 5: {
                unscaled = MathUtils.nextAfter(unscaled, Double.NEGATIVE_INFINITY);
                double fraction = unscaled - Math.floor(unscaled);
                if (fraction > 0.5) {
                    unscaled = Math.ceil(unscaled);
                    break;
                }
                unscaled = Math.floor(unscaled);
                break;
            }
            case 6: {
                double fraction = unscaled - Math.floor(unscaled);
                if (fraction > 0.5) {
                    unscaled = Math.ceil(unscaled);
                    break;
                }
                if (fraction < 0.5) {
                    unscaled = Math.floor(unscaled);
                    break;
                }
                if (Math.floor(unscaled) / 2.0 == Math.floor(Math.floor(unscaled) / 2.0)) {
                    unscaled = Math.floor(unscaled);
                    break;
                }
                unscaled = Math.ceil(unscaled);
                break;
            }
            case 4: {
                unscaled = MathUtils.nextAfter(unscaled, Double.POSITIVE_INFINITY);
                double fraction = unscaled - Math.floor(unscaled);
                if (fraction >= 0.5) {
                    unscaled = Math.ceil(unscaled);
                    break;
                }
                unscaled = Math.floor(unscaled);
                break;
            }
            case 7: {
                if (unscaled == Math.floor(unscaled)) break;
                throw new ArithmeticException("Inexact result from rounding");
            }
            case 0: {
                unscaled = Math.ceil(MathUtils.nextAfter(unscaled, Double.POSITIVE_INFINITY));
                break;
            }
            default: {
                throw new IllegalArgumentException("Invalid rounding method.");
            }
        }
        return unscaled;
    }

    public static byte sign(byte x) {
        return (byte)(x == 0 ? 0 : (x > 0 ? 1 : -1));
    }

    public static double sign(double x) {
        if (Double.isNaN(x)) {
            return Double.NaN;
        }
        return x == 0.0 ? 0.0 : (x > 0.0 ? 1.0 : -1.0);
    }

    public static float sign(float x) {
        if (Float.isNaN(x)) {
            return Float.NaN;
        }
        return x == 0.0f ? 0.0f : (x > 0.0f ? 1.0f : -1.0f);
    }

    public static int sign(int x) {
        return x == 0 ? 0 : (x > 0 ? 1 : -1);
    }

    public static long sign(long x) {
        return x == 0L ? 0L : (x > 0L ? 1L : -1L);
    }

    public static short sign(short x) {
        return (short)(x == 0 ? 0 : (x > 0 ? 1 : -1));
    }

    public static double sinh(double x) {
        return (Math.exp(x) - Math.exp(-x)) / 2.0;
    }

    public static int subAndCheck(int x, int y) {
        long s = (long)x - (long)y;
        if (s < Integer.MIN_VALUE || s > Integer.MAX_VALUE) {
            throw new ArithmeticException("overflow: subtract");
        }
        return (int)s;
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    public static long subAndCheck(long a, long b) {
        String msg = "overflow: subtract";
        if (b != Long.MIN_VALUE) return MathUtils.addAndCheck(a, -b, msg);
        if (a >= 0L) throw new ArithmeticException(msg);
        return a - b;
    }
}

