#' Compute shortest distance between origin and destination nodes.
#' 
#' @param Graph  An object generated by makegraph(), cpp_simplify() or cpp_contract() function.
#' @param from A vector of one or more vertices from which distances are calculated (origin).
#' @param to A vector of one or more vertices (destination).
#' @param algorithm character. "Dijkstra" for uni-directional Dijkstra, "bi" for bi-directional Dijkstra, "A*" for A star unidirectional search or "NBA" for New bi-directional A star .Default to "Dijkstra"
#' @param constant numeric. Constant to maintain the heuristic function admissible in A* and NBA algorithms. 
#' Default to 1, when cost is expressed in the same unit than coordinates. See details
#' @param allcores Logical. If TRUE, all cores are used.
#' @return Vector of shortest distances.
#' @note 'from' and 'to' must be the same length.
#' @details If graph is not contracted, the user has the choice between : \itemize{
#'   \item unidirectional Dijkstra (Dijkstra) 
#'   \item A star (A*) : projected coordinates should be provided
#'   \item bidirectional Dijkstra (bi)
#'   \item New bi-directional A star (NBA) : projected coordinates should be provided
#' }
#' 
#' If the input graph has been contracted by cpp_contract() function, the algorithm is a modified bidirectional search.  
#'   
#' In A* and New Bidirectional A star algorithms, euclidean distance is used as heuristic function.
#' To understand how A star algorithm work, see \url{https://en.wikipedia.org/wiki/A*_search_algorithm}.
#' To understand the importance of constant parameter, see the package description : \url{https://github.com/vlarmet/cppRouting/blob/master/README.md}
#' @examples 
#' #Data describing edges of the graph
#' edges<-data.frame(from_vertex=c(0,0,1,1,2,2,3,4,4), 
#'                   to_vertex=c(1,3,2,4,4,5,1,3,5), 
#'                   cost=c(9,2,11,3,5,12,4,1,6))
#' 
#' #Get all nodes
#' nodes<-unique(c(edges$from_vertex,edges$to_vertex))
#'  
#' #Construct directed and undirected graph 
#' directed_graph<-makegraph(edges,directed=TRUE)
#' non_directed<-makegraph(edges,directed=FALSE)
#' 
#' #Sampling origin and destination nodes
#' origin<-sample(nodes,10,replace=TRUE)
#' destination<-sample(nodes,10,replace=TRUE)
#' 
#' #Get distance between origin and destination in the two graphs
#' dir_dist<-get_distance_pair(Graph=directed_graph, from=origin, to=destination, allcores=FALSE)
#' non_dir_dist<-get_distance_pair(Graph=non_directed, from=origin, to=destination, allcores=FALSE)
#' print(dir_dist)
#' print(non_dir_dist)

get_distance_pair<-function(Graph,from,to,algorithm="bi",constant=1,allcores=FALSE){
  
  if (length(from)!=length(to)) stop("From and to have not the same length")
  
  if (any(is.na(cbind(from,to)))) stop("NAs are not allowed in origin/destination nodes")

  from<-as.character(from)
  
  to<-as.character(to)
  allnodes<-c(from,to)
  if (sum(allnodes %in% Graph$dict$ref)<length(allnodes)) stop("Some nodes are not in the graph")
  
  from_id<-Graph$dict$id[match(from,Graph$dict$ref)]
  to_id<-Graph$dict$id[match(to,Graph$dict$ref)]
  
  
  
  if (length(Graph)==4) {
    if (!is.null(Graph$coords)){
      if (algorithm %in% c("NBA","A*","bi")){
        if (algorithm=="A*"){
          message("Running A* ...")
          if (allcores==FALSE) res<-Astar(from_id,to_id,Graph$data$from,Graph$data$to,Graph$data$dist,Graph$nbnode,Graph$coords[,2],Graph$coords[,3],constant)
          else res<-Astar_par(from_id,to_id,Graph$data$from,Graph$data$to,Graph$data$dist,Graph$nbnode,Graph$coords[,2],Graph$coords[,3],constant)
        }
        if (algorithm=="NBA"){
          message("Running NBA* ...")
          if (allcores==FALSE) res<-NBA(from_id,to_id,Graph$data$from,Graph$data$to,Graph$data$dist,Graph$nbnode,Graph$coords[,2],Graph$coords[,3],constant)
          else res<-NBA_par(from_id,to_id,Graph$data$from,Graph$data$to,Graph$data$dist,Graph$nbnode,Graph$coords[,2],Graph$coords[,3],constant)
        }
        
        
        if (algorithm=="bi"){
          message("Running bidirectional Dijkstra...")
          if (allcores==FALSE) res<-Bidir(from_id,to_id,Graph$data$from,Graph$data$to,Graph$data$dist,Graph$nbnode)
          else res<-Bidir_par(from_id,to_id,Graph$data$from,Graph$data$to,Graph$data$dist,Graph$nbnode)
        }
      }
      
      else {
        message("Running Dijkstra ...")
        if (allcores==FALSE) res<-Dijkstra_early_stop(from_id,to_id,Graph$data$from,Graph$data$to,Graph$data$dist,Graph$nbnode)
        else res<-Dijkstra_par(from_id,to_id,Graph$data$from,Graph$data$to,Graph$data$dist,Graph$nbnode)
      }
      
    }
    else {
      if (algorithm=="bi"){
        message("Running bidirectional Dijkstra...")
        if (allcores==FALSE) res<-Bidir(from_id,to_id,Graph$data$from,Graph$data$to,Graph$data$dist,Graph$nbnode)
        else res<-Bidir_par(from_id,to_id,Graph$data$from,Graph$data$to,Graph$data$dist,Graph$nbnode)
      }
      else {
        message("Running Dijkstra ...")
        if (allcores==FALSE) res<-Dijkstra_early_stop(from_id,to_id,Graph$data$from,Graph$data$to,Graph$data$dist,Graph$nbnode)
        else res<-Dijkstra_par(from_id,to_id,Graph$data$from,Graph$data$to,Graph$data$dist,Graph$nbnode)
      }
      
    } 
  }
  
  if (length(Graph)==5){
    message("Running CH Dijkstra...")
    if (allcores==FALSE) res<-Bidir_mod3(from_id,to_id,Graph$data$from,Graph$data$to,Graph$data$dist,Graph$nbnode,Graph$rank)
    else res<-par_Bidir_mod2(from_id,to_id,Graph$data$from,Graph$data$to,Graph$data$dist,Graph$nbnode,Graph$rank)
  }
  
 
  
  
  return(res)
  
  
}
