\name{plsda}
\alias{plsda.default}
\alias{predict.plsda}
\alias{plsda}
\alias{splsda.default}
\alias{predict.splsda}
\alias{splsda}
\title{Partial Least Squares and Sparse Partial Least Squares Discriminant Analysis}
\description{
\code{plsda} is used to fit standard PLS models for classification while \code{splsda} performs sparse PLS that embeds feature selection and regularization for the same purpose.
}
\usage{
plsda(x, ...)

\method{plsda}{default}(x,  y, ncomp = 2, probMethod = "softmax", prior = NULL, ...)

\method{predict}{plsda}(object, newdata = NULL, ncomp = NULL, type = "class", ...)

splsda(x, ...)

\method{splsda}{default}(x, y, probMethod = "softmax", prior = NULL, ...)

\method{predict}{splsda}(object, newdata = NULL, type = "class", ...)
}

\arguments{
  \item{x}{a matrix or data frame of predictors}
  \item{y}{a factor or indicator matrix for the discrete outcome. If a matrix, the entries must be either 0 or 1 and rows must sum to one}
  \item{ncomp}{the number of components to include in the model. Predictions can be made for models with values less than \code{ncomp}. }
  \item{probMethod}{either "softmax" or "Bayes" (see Details)}
  \item{prior}{a vector or prior probabilities for the classes (only used for \code{probeMethod = "Bayes"}}
  \item{\dots}{arguments to pass to \code{\link[pls:mvr]{plsr}} or \code{\link[spls]{spls}}. For \code{splsda}, this is the method for passing tuning parameters specifications (e.g. \code{K}, \code{eta} or \code{kappa})}
  \item{object}{an object produced by \code{plsda}}
  \item{newdata}{a matrix or data frame of predictors}
  \item{type}{either \code{"class"}, \code{"prob"} or \code{"raw"} to produce the predicted class, class probabilities  or the raw model scores, respectively.}  
}
\details{
If a factor is supplied, the appropriate indicator matrix is created.

A multivariate PLS model is fit to the indicator matrix using the \code{\link[pls:mvr]{plsr}} or  \code{\link[spls]{spls}} function. 

Two prediciton methods can be used.

  The \bold{softmax function} transforms the model predictions to "probability-like" values (e.g. on  [0, 1] and sum to 1). The class with the largest class probability is the predicted class.

   Also, \bold{Bayes rule} can be applied to the model predictions to form posterior probabilities. Here, the model predictions for the training set are used along with the training set outcomes to create conditional distributions for each class. When new samples are predicted, the raw model predictions are run through these conditional distributions to produce a posterior probability for each class (along with the prior). This process is repeated \code{ncomp} times for every possible PLS model. The \code{\link[klaR]{NaiveBayes}} function is used with \code{usekernel = TRUE} for the posterior probability calculations.

}
\value{
For \code{plsda}, an object of class "plsda" and "mvr". For \code{splsda}, an object of class \code{splsda}. 

The predict
methods produce either a vector, matrix or three-dimensional array,
depending on the values of \code{type} of \code{ncomp}. For example,
specifying more than one value of \code{ncomp} with \code{type =
  "class"} with produce a three dimensional array but the default
specification would produce a factor vector.
}

\seealso{\code{\link[pls:mvr]{plsr}}, \code{\link[spls]{spls}}}
\examples{
data(mdrr)
set.seed(1)
inTrain <- sample(seq(along = mdrrClass), 450)
 
nzv <- nearZeroVar(mdrrDescr)
filteredDescr <- mdrrDescr[, -nzv]

training <- filteredDescr[inTrain,]
test <- filteredDescr[-inTrain,]
trainMDRR <- mdrrClass[inTrain]
testMDRR <- mdrrClass[-inTrain]
 
preProcValues <- preProcess(training)

trainDescr <- predict(preProcValues, training)
testDescr <- predict(preProcValues, test)

useBayes   <- plsda(trainDescr, trainMDRR, ncomp = 5,
                    probMethod = "Bayes")
useSoftmax <- plsda(trainDescr, trainMDRR, ncomp = 5)

confusionMatrix(
                predict(useBayes, testDescr),
                testMDRR)

confusionMatrix(
                predict(useSoftmax, testDescr),
                testMDRR)

histogram(
          ~predict(useBayes, testDescr, type = "prob")[,"Active"]
          | testMDRR, xlab = "Active Prob", xlim = c(-.1,1.1))
histogram(
          ~predict(useSoftmax, testDescr, type = "prob")[,"Active",]
          | testMDRR, xlab = "Active Prob", xlim = c(-.1,1.1))


## different sized objects are returned
length(predict(useBayes, testDescr))
dim(predict(useBayes, testDescr, ncomp = 1:3))
dim(predict(useBayes, testDescr, type = "prob"))
dim(predict(useBayes, testDescr, type = "prob", ncomp = 1:3))

## Using spls:
## (As of 11/09, the spls package now has a similar funciton with
## the same mane. To avoid conflicts, use caret:::splsda to 
## get this version)

splsFit <- caret:::splsda(trainDescr, trainMDRR, 
                          K = 5, eta = .9,
                          probMethod = "Bayes")

confusionMatrix(
                caret:::predict.splsda(splsFit, testDescr),
                testMDRR)
}


\keyword{models}

