% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fit_dynamic_growth.R
\name{fit_MCMC_growth}
\alias{fit_MCMC_growth}
\title{Fit growth models using MCMC}
\usage{
fit_MCMC_growth(
  fit_data,
  env_conditions,
  starting_point,
  known_pars,
  sec_model_names,
  niter,
  ...
)
}
\arguments{
\item{fit_data}{Tibble with the data to use for model fit. It must
contain a column named 'time' with the storage time and another named
'logN' with the observed microbial count.}

\item{env_conditions}{Tibble with the (dynamic) environmental conditions
during the experiment. It must have one column named 'time' with the
storage time and as many columns as required with the environmental conditions.
Note that only those defined in 'sec_model_names' will be considered for
the model fit.}

\item{starting_point}{A named vector of starting values for the model parameters.
Parameters for the primary model must be named in the usual way. Parameters for the
secondary model are named as \code{env_factor}+'_'+\code{parameter}. For instance,
the maximum growth temperature shall be named 'temperature_xmax'.}

\item{known_pars}{A named vectors of known model parameters (i.e. not fitted). They
must be named using the same convention as for \code{starting_point}.}

\item{sec_model_names}{A named character vector defining the secondary model for each
environmental factor. The names define the factor and the value the type of model.
Names must match columns in \code{fit_data} and \code{env_conditions}.}

\item{niter}{number of iterations of the MCMC algorithm.}

\item{...}{Additional arguments passed to modFit.}
}
\value{
A list of class 'FitDynamicGrowthMCMC' with the following items:
     \itemize{
         \item fit_results: the object returned by \code{modMCMC}.
         \item best_prediction: the model prediction for the fitted parameters.
         \item data: data used for the fit.
         \item starting: starting values for model fitting
         \item known: parameter values set as known.
         \item sec_models: a named vector with the secondary model
         for each environmental factor
         }
}
\description{
Fits a growth model to a data obtained under dynamic conditions
using the one-step approach (MCMC algorithm).
}
\examples{
\donttest{
## We use the example data included in the package

data("example_dynamic_growth")
data("example_env_conditions")

## Definition of the secondary models
sec_model_names <- c(temperature = "CPM", aw= "CPM")

## Any model parameter can be fixed
known_pars <- list(Nmax = 1e4,  # Primary model
    N0 = 1e0, Q0 = 1e-3,  # Initial values of the primary model
    mu_opt = 4, # mu_opt of the gamma model
    temperature_n = 1,  # Secondary model for temperature
    aw_xmax = 1, aw_xmin = .9, aw_n = 1  # Secondary model for water activity
    )

## We need starting values for the remaining parameters

my_start <- list(temperature_xmin = 25, temperature_xopt = 35,
    temperature_xmax = 40,
    aw_xopt = .95)

## We can now call the fitting function

set.seed(12124) # Setting seed for repeatability

my_MCMC_fit <- fit_MCMC_growth(example_dynamic_growth, example_env_conditions,
    my_start, known_pars, sec_model_names, niter = 3000)

## Always check the MCMC chain!!

plot(my_MCMC_fit$fit_results)

## We can compare data against fitted curve

plot(my_MCMC_fit)

## Any environmental factor can be included using add_factor

plot(my_MCMC_fit, add_factor = "temperature",
    label_y1 = "Count (log CFU/ml)", label_y2 = "Temperature (C)")

}

}
