% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/biglasso.R
\name{big_spLinReg}
\alias{big_spLinReg}
\title{Sparse linear regression}
\usage{
big_spLinReg(
  X,
  y.train,
  ind.train = rows_along(X),
  ind.col = cols_along(X),
  covar.train = NULL,
  base.train = NULL,
  pf.X = NULL,
  pf.covar = NULL,
  alphas = 1,
  K = 10,
  ind.sets = NULL,
  nlambda = 200,
  nlam.min = 50,
  n.abort = 10,
  dfmax = 50000,
  ncores = 1,
  ...
)
}
\arguments{
\item{X}{A \link[=FBM-class]{FBM}.}

\item{y.train}{Vector of responses, corresponding to \code{ind.train}.}

\item{ind.train}{An optional vector of the row indices that are used,
for the training part. If not specified, all rows are used.
\strong{Don't use negative indices.}}

\item{ind.col}{An optional vector of the column indices that are used.
If not specified, all columns are used. \strong{Don't use negative indices.}}

\item{covar.train}{Matrix of covariables to be added in each model to correct
for confounders (e.g. the scores of PCA), corresponding to \code{ind.train}.
Default is \code{NULL} and corresponds to only adding an intercept to each model.}

\item{base.train}{Vector of base predictions. Model will be learned starting
from these predictions. This can be useful if you want to previously fit
a model with large-effect variables that you don't want to penalize.}

\item{pf.X}{A multiplicative factor for the penalty applied to each coefficient.
If supplied, \code{pf.X} must be a numeric vector of the same length as \code{ind.col}.
Default is all \code{1}. The purpose of \code{pf.X} is to apply differential
penalization if some coefficients are thought to be more likely than others
to be in the model. Setting SOME to 0 allows to have unpenalized coefficients.}

\item{pf.covar}{Same as \code{pf.X}, but for \code{covar.train}.}

\item{alphas}{The elastic-net mixing parameter that controls the relative
contribution from the lasso (l1) and the ridge (l2) penalty. The penalty is
defined as \deqn{ \alpha||\beta||_1 + (1-\alpha)/2||\beta||_2^2.}
\code{alpha = 1} is the lasso penalty and \code{alpha} in between \code{0}
(\code{1e-4}) and \code{1} is the elastic-net penalty. Default is \code{1}. \strong{You can
pass multiple values, and only one will be used (optimized by grid-search).}}

\item{K}{Number of sets used in the Cross-Model Selection and Averaging
(CMSA) procedure. Default is \code{10}.}

\item{ind.sets}{Integer vectors of values between \code{1} and \code{K} specifying
which set each index of the training set is in. Default randomly assigns
these values but it can be useful to set this vector for reproducibility,
or if you want to refine the grid-search over \code{alphas} using the same sets.}

\item{nlambda}{The number of lambda values. Default is \code{200}.}

\item{nlam.min}{Minimum number of lambda values to investigate. Default is \code{50}.}

\item{n.abort}{Number of lambda values for which prediction on the validation
set must decrease before stopping. Default is \code{10}.}

\item{dfmax}{Upper bound for the number of nonzero coefficients. Default is
\code{50e3} because, for large data sets, computational burden may be
heavy for models with a large number of nonzero coefficients.}

\item{ncores}{Number of cores used. Default doesn't use parallelism.
You may use \link{nb_cores}.}

\item{...}{
  Arguments passed on to \code{\link[=COPY_biglasso_main]{COPY_biglasso_main}}
  \describe{
    \item{\code{lambda.min.ratio}}{The smallest value for lambda, \strong{as a fraction of
lambda.max}. Default is \code{.0001} if the number of observations is larger than
the number of variables and \code{.001} otherwise.}
    \item{\code{eps}}{Convergence threshold for inner coordinate descent.
The algorithm iterates until the maximum change in the objective after any
coefficient update is less than \code{eps} times the null deviance.
Default value is \code{1e-5}.}
    \item{\code{max.iter}}{Maximum number of iterations. Default is \code{1000}.}
    \item{\code{warn}}{Deprecated. Now return the reason of completion as \verb{$message}.}
    \item{\code{return.all}}{Deprecated. Now always return all models.}
  }}
}
\value{
Return an object of class \code{big_sp_list} (a list of \code{length(alphas)}
x \code{K}) that has 3 methods \code{predict}, \code{summary} and \code{plot}.
}
\description{
Fit lasso penalized linear regression path for a Filebacked Big Matrix.
Covariates can be added to correct for confounders.
}
\details{
\strong{This is a modified version of one function of
\href{https://github.com/YaohuiZeng/biglasso}{package biglasso}}.
It adds the possibility to train models with covariables and use many
types of \code{FBM} (not only \code{double} ones).
Yet, it only corresponds to \code{screen = "SSR"} (Sequential Strong Rules).

Also, to remove the choice of the lambda parameter, we introduce the
Cross-Model Selection and Averaging (CMSA) procedure:
\enumerate{
\item This function separates the training set in \code{K} folds (e.g. 10).
\item \strong{In turn},
\itemize{
\item each fold is considered as an inner validation set and the others
(K - 1) folds form an inner training set,
\item the model is trained on the inner training set and the corresponding
predictions (scores) for the inner validation set are computed,
\item the vector of scores which maximizes log-likelihood is determined,
\item the vector of coefficients corresponding to the previous vector of
scores is chosen.
}
\item The \code{K} resulting vectors of coefficients are then averaged into one final
vector of coefficients.
}
}
\examples{
set.seed(1)

# simulating some data
N <- 230
M <- 730
X <- FBM(N, M, init = rnorm(N * M, sd = 5))
y <- rowSums(X[, 1:10]) + rnorm(N)
covar <- matrix(rnorm(N * 3), N)

ind.train <- sort(sample(nrow(X), 150))
ind.test <- setdiff(rows_along(X), ind.train)

# fitting model for multiple lambdas and alphas
test <- big_spLinReg(X, y[ind.train], ind.train = ind.train,
                     covar.train = covar[ind.train, ],
                     alphas = c(1, 0.5, 0.1, 0.01))
# peek at the models
plot(test)
summary(test)

# prediction for other data -> only the best alpha is used
summary(test, best.only = TRUE)
pred <- predict(test, X, ind.row = ind.test, covar.row = covar[ind.test, ])
plot(pred, y[ind.test], pch = 20); abline(0, 1, col = "red")
}
\references{
Tibshirani, R., Bien, J., Friedman, J., Hastie, T.,
Simon, N., Taylor, J. and Tibshirani, R. J. (2012),
Strong rules for discarding predictors in lasso-type problems.
Journal of the Royal Statistical Society:
Series B (Statistical Methodology), 74: 245-266.
\url{https://doi.org/10.1111/j.1467-9868.2011.01004.x}.

Zeng, Y., and Breheny, P. (2017). The biglasso Package: A Memory- and
Computation-Efficient Solver for Lasso Model Fitting with Big Data in R.
arXiv preprint arXiv:1701.05936. \url{https://arxiv.org/abs/1701.05936}.

Privé, F., Aschard, H., and Blum, M. G.B. (2019). Efficient implementation of
penalized regression for genetic risk prediction. Genetics, 212: 65-74.
\url{https://doi.org/10.1534/genetics.119.302019}.
}
\seealso{
\link[glmnet:glmnet]{glmnet} \link[biglasso:biglasso]{biglasso}
}
