#' Estimate Improvement in R-Squared
#' and Generate the Corresponding Sampling Distribution
#' Using the Monte Carlo Method
#'
#' @author Ivan Jacob Agaloos Pesigan
#'
#' @details The vector of improvement in R-squared
#'   (\eqn{\Delta R^{2}})
#'   is derived from each randomly generated vector of parameter estimates.
#'   Confidence intervals are generated by obtaining
#'   percentiles corresponding to \eqn{100(1 - \alpha)\%}
#'   from the generated sampling
#'   distribution of \eqn{\Delta R^{2}},
#'   where \eqn{\alpha} is the significance level.
#'
#' @return Returns an object
#'   of class `betamc` which is a list with the following elements:
#'   \describe{
#'     \item{call}{Function call.}
#'     \item{args}{Function arguments.}
#'     \item{thetahatstar}{Sampling distribution of
#'       \eqn{\Delta R^{2}}.}
#'     \item{vcov}{Sampling variance-covariance matrix of
#'       \eqn{\Delta R^{2}}.}
#'     \item{est}{Vector of estimated
#'       \eqn{\Delta R^{2}}.}
#'     \item{fun}{Function used ("DeltaRSqMC").}
#'   }
#'
#' @inheritParams BetaMC
#'
#' @examples
#' # Data ---------------------------------------------------------------------
#' data("nas1982", package = "betaMC")
#'
#' # Fit Model in lm ----------------------------------------------------------
#' object <- lm(QUALITY ~ NARTIC + PCTGRT + PCTSUPP, data = nas1982)
#'
#' # MC -----------------------------------------------------------------------
#' mc <- MC(
#'   object,
#'   R = 100, # use a large value e.g., 20000L for actual research
#'   seed = 0508
#' )
#'
#' # DeltaRSqMC ---------------------------------------------------------------
#' out <- DeltaRSqMC(mc, alpha = 0.05)
#'
#' ## Methods -----------------------------------------------------------------
#' print(out)
#' summary(out)
#' coef(out)
#' vcov(out)
#' confint(out, level = 0.95)
#'
#' @family Beta Monte Carlo Functions
#' @keywords betaMC deltarsq
#' @export
DeltaRSqMC <- function(object,
                       alpha = c(0.05, 0.01, 0.001)) {
  sr <- SCorMC(object)
  out <- list(
    call = match.call(),
    args = list(
      object = object,
      alpha = alpha
    ),
    thetahatstar = lapply(
      X = sr$thetahatstar,
      FUN = function(x) {
        x^2
      }
    ),
    est = (sr$est)^2,
    fun = "DeltaRSqMC"
  )
  class(out) <- c(
    "betamc",
    class(out)
  )
  return(
    out
  )
}
