#' Uses "marker_list" from Cellmarker2 for cell annotation
#'
#' @param seurat_obj Enter the Seurat object with annotation columns such as
#'     "seurat_cluster" in meta.data to be annotated.
#' @param gene_list Enter the standard "Marker_list" generated by the Cellmarker2
#'     database for the SlimR package, generated by the "Markers_filter_Cellmarker2 ()"
#'     function.
#' @param species This parameter selects the species "Human" or "Mouse" for standard
#'     gene format correction of markers entered by "Marker_list".
#' @param cluster_col Enter annotation columns such as "seurat_cluster" in meta.data
#'     of the Seurat object to be annotated. Default parameters use "cluster_col =
#'     'seurat_clusters'".
#' @param assay Enter the assay used by the Seurat object, such as "RNA". Default
#'     parameters use "assay = "RNA"".
#' @param save_path The output path of the cell annotation picture. Example parameters
#'     use "save_path = './SlimR/Celltype_annotation_Cellmarker2/'".
#' @param min_counts The minimum number of counts of genes in "Marker_list" entered.
#'     This number represents the number of the same gene in the same species and
#'     the same location in the Cellmarker2 database used for annotation of this cell
#'     type. Default parameters use "min_counts = 1".
#' @param colour_low Color for lowest expression level. (default = "white")
#' @param colour_high Color for highest expression level. (default = "navy")
#' @param colour_low_mertic Color for lowest mertic level. (default = "white")
#' @param colour_high_mertic Color for highest mertic level. (default = "navy")
#'
#' @returns The cell annotation picture is saved in "save_path".
#' @export
#' @family Section_5_Other_Functions_Provided
#'
#' @importFrom ggplot2 ggplot aes theme labs element_text element_blank
#' @importFrom ggplot2 geom_tile scale_fill_gradient theme_minimal
#' @importFrom ggplot2 margin
#'
#' @examples
#' \dontrun{
#' Celltype_annotation_Cellmarker2(seurat_obj = sce,
#'     gene_list = Markers_list_Cellmarker2,
#'     species = "Human",
#'     cluster_col = "seurat_clusters",
#'     assay = "RNA",
#'     save_path = file.path(tempdir(),"SlimR_Celltype_annotation_Cellmarker2")
#'     colour_low = "white",
#'     colour_high = "navy",
#'     colour_low_mertic = "white",
#'     colour_high_mertic = "navy",
#'     )
#'     }
#'
Celltype_annotation_Cellmarker2 <- function(
    seurat_obj,
    gene_list,
    species,
    cluster_col = "seurat_clusters",
    assay = "RNA",
    save_path = NULL,
    min_counts = 1,
    colour_low = "white",
    colour_high = "navy",
    colour_low_mertic = "white",
    colour_high_mertic = "navy"
) {
  if (!requireNamespace("cowplot", quietly = TRUE)) {
    stop("Please install package 'cowplot': install.packages('cowplot')")
  }
  if (!inherits(seurat_obj, "Seurat")) {
    stop("Input object must be a Seurat object!")
  }
  if (!is.list(gene_list)) {
    stop("Gene list must be a list of data.frames with 'marker' and 'counts' columns!")
  }
  if (species != "Human" && species != "Mouse") {
    stop("species must be 'Human' or 'Mouse'")
  }
  if (missing(save_path)) {stop("Output path must be explicitly specified")}
  if (!interactive() && !grepl(tempdir(), save_path, fixed = TRUE)) {
    warning("Writing to non-temporary locations is restricted", immediate. = TRUE)
    path <- file.path(tempdir(), "fallback_output")
  }

  colour_low <- if (is.null(colour_low)) "white" else colour_low
  colour_high <- if (is.null(colour_high)) "navy" else colour_high
  colour_low_mertic <- if (is.null(colour_low_mertic)) colour_low else colour_low_mertic
  colour_high_mertic <- if (is.null(colour_high_mertic)) colour_high else colour_high_mertic

  dir.create(save_path, showWarnings = FALSE)

  seurat_genes <- rownames(seurat_obj@assays[[assay]])

  capitalize_first <- function(gene) {
    s <- as.character(gene)
    if (nchar(s) == 0) return(s)
    first_char <- toupper(substr(s, 1, 1))
    rest <- substr(s, 2, nchar(s))
    return(paste(first_char, rest, sep = ""))
  }

  cell_types <- names(gene_list)
  total <- length(cell_types)
  cycles <- 0

  message(paste0("SlimR: The input 'Markers_list' has ",total," cell types to be processed."))

  for (i in seq_along(cell_types)) {
    cell_type <- cell_types[i]
    message(paste0("\n","[", i, "/", total, "] Processing cell type: ", cell_type))

    current_df <- gene_list[[cell_type]]
    if (!all(c("marker", "counts") %in% names(current_df))) {
      warning(paste("Missing 'marker' or 'counts' column for cell type", cell_type))
      next
    }

    if (any(is.na(current_df$marker))) {
      warning(paste("NA values found in marker column for cell type:", cell_type))
      next
    }

    genes <- current_df$marker
    counts <- current_df$counts

    if (species == "Human") {
      genes_processed <- toupper(genes)
    } else if (species == "Mouse") {
      genes_processed <- sapply(genes, capitalize_first)
    }

    valid_genes <- genes_processed[genes_processed %in% seurat_genes]
    valid_genes <- unique(valid_genes)
    if (length(valid_genes) == 0) {
      warning(paste("No valid genes found for", cell_type))
      next
    }

    original_genes <- genes[genes_processed %in% valid_genes]
    original_counts <- counts[genes_processed %in% valid_genes]

    valid_genes_seurat <- intersect(valid_genes, seurat_genes)
    original_genes <- original_genes[genes_processed %in% valid_genes_seurat]
    original_counts <- original_counts[genes_processed %in% valid_genes_seurat]

    if (length(original_genes) == 0) {
      warning(paste("No genes exist in the Seurat object for", cell_type))
      next
    }

    filtered_genes <- original_genes[original_counts >= min_counts]
    filtered_counts <- original_counts[original_counts >= min_counts]

    if (length(filtered_genes) == 0) {
      warning(paste("No genes meet the minimum counts requirement for", cell_type))
      next
    }

    valid_features <- intersect(filtered_genes, rownames(seurat_obj@assays[[assay]]))
    if (length(valid_features) == 0) {
      warning(paste("No valid features found in Seurat object for cell type:", cell_type))
      next
    }

    num_idents <- length(unique(Idents(seurat_obj)))
    num_genes <- length(valid_features)

    plot_height <- max(6, num_idents * 0.8) + 1
    plot_width <- max(10, num_genes * 0.4)

    dp <- Seurat::DotPlot(
      seurat_obj,
      features = valid_features,
      assay = assay,
      group.by = cluster_col,
      cols = c(colour_low, colour_high)
    ) +
      theme(
        axis.text.x = element_text(
          angle = 45,
          hjust = 1,
          family = "sans",
          size = 10
        ),
        axis.title.x = element_blank(),
        axis.title.y = element_text(family = "sans")
      ) +
      labs(
        title = paste("Cell Type:", cell_type, "| Markers_list from Cellmarkers2 Database | SlimR"),
        subtitle = "Dot size: Expression percentage | Color: Normalized expression level"
      )

    counts_for_heatmap <- filtered_counts[match(valid_features, filtered_genes)]
    heatmap_data <- data.frame(
      Gene = factor(valid_features, levels = valid_features),
      Counts = counts_for_heatmap
    )

    hp <- ggplot(heatmap_data, aes(x = Gene, y = "Counts", fill = Counts)) +
      geom_tile(color = "gray") +
      scale_fill_gradient(
        low = colour_low_mertic,
        high = colour_high_mertic,
        limits = c(0, max(counts_for_heatmap))
      ) +
      theme_minimal() +
      theme(
        axis.text.x = element_text(
          angle = 45,
          hjust = 1,
          family = "sans",
          size = 10,
          face = "plain",
          margin = margin(t = 5, unit = "pt")
        ),
        axis.title.x = element_blank(),
        axis.title.y = element_blank(),
        axis.text.y = element_blank(),
        axis.ticks.y = element_blank(),
        panel.grid = element_blank(),
        plot.title = element_text(hjust = 0, size = 12),
        plot.margin = margin(t = 10, b = 15, unit = "pt")
      ) +
      labs(
        x = NULL,
        y = "Counts",
        title = "Gene counts in Cellmarkers2.0 database"
      )

    base_height <- 5
    heatmap_height_ratio <- 0.2

    combined_plot <- cowplot::plot_grid(
      dp,
      hp,
      ncol = 1,
      align = "v",
      rel_heights = c(1, heatmap_height_ratio)
    )

    ggsave(
      filename = file.path(save_path, paste0(cell_type, ".png")),
      plot = combined_plot,
      height = plot_height * (1 + heatmap_height_ratio),
      width = plot_width,
      limitsize = FALSE
    )
    cycles <- cycles + 1
    message(paste0("[", i, "/", total, "] Feature plots saved for: ", cell_type))
  }
  message(paste0("\n","SlimR: Out of the ",total," cell types in 'Markers_list', ",cycles," cell types have been processed. You can see the reason for not processing cell types by 'warnings()'."))
  message(paste0("\n","SlimR: Visualization saved to: ", normalizePath(save_path)))
}
