% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/source_compete.R
\name{bgp_bc}
\alias{bgp_bc}
\title{Bayesian Gaussian processes with a Box-Cox transformation}
\usage{
bgp_bc(
  y,
  locs,
  X = NULL,
  covfun_name = "matern_isotropic",
  locs_test = locs,
  X_test = NULL,
  nn = 30,
  emp_bayes = TRUE,
  lambda = NULL,
  sample_lambda = TRUE,
  nsave = 1000,
  nburn = 1000,
  nskip = 0
)
}
\arguments{
\item{y}{\code{n x 1} response vector}

\item{locs}{\code{n x d} matrix of locations}

\item{X}{\code{n x p} design matrix; if unspecified, use intercept only}

\item{covfun_name}{string name of a covariance function; see ?GpGp}

\item{locs_test}{\code{n_test x d} matrix of locations
at which predictions are needed; default is \code{locs}}

\item{X_test}{\code{n_test x p} design matrix for test data;
default is \code{X}}

\item{nn}{number of nearest neighbors to use; default is 30
(larger values improve the approximation but increase computing cost)}

\item{emp_bayes}{logical; if TRUE, use a (faster!) empirical Bayes
approach for estimating the mean function}

\item{lambda}{Box-Cox transformation; if NULL, estimate this parameter}

\item{sample_lambda}{logical; if TRUE, sample lambda, otherwise
use the fixed value of lambda above or the MLE (if lambda unspecified)}

\item{nsave}{number of MCMC iterations to save}

\item{nburn}{number of MCMC iterations to discard}

\item{nskip}{number of MCMC iterations to skip between saving iterations,
i.e., save every (nskip + 1)th draw}
}
\value{
a list with the following elements:
\itemize{
\item \code{coefficients} the posterior mean of the regression coefficients
\item \code{fitted.values} the posterior predictive mean at the test points \code{locs_test}
\item \code{fit_gp} the fitted \code{GpGp_fit} object, which includes
covariance parameter estimates and other model information
\item \code{post_ypred}: \code{nsave x n_test} samples
from the posterior predictive distribution at \code{locs_test}
\item \code{post_g}: \code{nsave} posterior samples of the transformation
evaluated at the unique \code{y} values
\item \code{post_lambda} \code{nsave} posterior samples of lambda
\item \code{model}: the model fit (here, \code{bgp_bc})
}
as well as the arguments passed in.
}
\description{
MCMC sampling for Bayesian Gaussian process regression with a
(known or unknown) Box-Cox transformation.
}
\details{
This function provides Bayesian inference for
transformed Gaussian processes. The transformation is
parametric from the Box-Cox family, which has one parameter \code{lambda}.
That parameter may be fixed in advanced or learned from the data.
For computational efficiency, the Gaussian process parameters are
fixed at point estimates, and the latent Gaussian process is only sampled
when \code{emp_bayes} = FALSE.
}
\note{
Box-Cox transformations may be useful in some cases, but
in general we recommend the nonparametric transformation (with
Monte Carlo, not MCMC sampling) in \code{\link{sbgp}}.
}
\examples{
\donttest{
# Simulate some data:
n = 200 # sample size
x = seq(0, 1, length = n) # observation points

# Transform a noisy, periodic function:
y = g_inv_bc(
  sin(2*pi*x) + sin(4*pi*x) + rnorm(n, sd = .5),
             lambda = .5) # Signed square-root transformation

# Fit a Bayesian Gaussian process with Box-Cox transformation:
fit = bgp_bc(y = y, locs = x)
names(fit) # what is returned
coef(fit) # estimated regression coefficients (here, just an intercept)
class(fit$fit_gp) # the GpGp object is also returned
round(quantile(fit$post_lambda), 3) # summary of unknown Box-Cox parameter

# Plot the model predictions (point and interval estimates):
pi_y = t(apply(fit$post_ypred, 2, quantile, c(0.05, .95))) # 90\% PI
plot(x, y, type='n', ylim = range(pi_y,y),
     xlab = 'x', ylab = 'y', main = paste('Fitted values and prediction intervals'))
polygon(c(x, rev(x)),c(pi_y[,2], rev(pi_y[,1])),col='gray', border=NA)
lines(x, y, type='p')
lines(x, fitted(fit), lwd = 3)
}

}
