\docType{package}
\name{PAICE-package}
\alias{PAICE}
\alias{PAICE-package}
\title{
    Phylogeographic Analysis of Island Colonization Events
}
\description{
    A package for inferring inter-island colonization events in island-like
    systems.
}
\details{
    Estimation of the number of infer inter-island colonization events in an
    island-like system by analyzing the geographic distribution of uniparentally
    inherited haplotypes and their genealogical relationships. Furthermore, by
    building rarefaction curves based on both genetic sampling (variable
    positions) and field sampling (populations/individuals), an estimation of
    the number of colonization events corrected by sampling effort could be
    done. The method used in the \pkg{PAICE} package is described in Coello et
    al. (2022).
}

\section{PAICE functions}{
    \code{\link{colonization}} to infer the minimun number of colonization
        events

    \code{\link{geneticResampling}} to simplify the genealogy by deleting a
        variable position

    \code{\link{maxCol}} to calculate asymptotic estimators considering genetic
        and field sampling

    \code{\link{plot.maxCol}} to plot curves generated by maxCol

    \code{\link{plot.rarecol}} to plot rarefaction curves

    \code{\link{rarecol}} to generate rarefaction curves of colonization events

    \code{\link{read.rarecol}} to read previously saved rarefaction curve files
}

\section{PAICE datasets}{
     \code{\link{CmonsData}} haplotype distribution of
        \emph{Cistus monspeliensis} in the Canary Islands

     \code{\link{CmonsNetwork}} genealogy of \emph{Cistus monspeliensis}

     \code{\link{CmonsRare}} example data of rarefaction curves for \emph{Cistus
        monspeliensis}
}

\author{
    Alberto J. Coello,
    Mario Fernandez-Mazuecos,
    Ruben H. Heleno
    and Pablo Vargas

    Maintainer: Alberto J. Coello <albjcoello@gmail.com>
}

\references{
    Coello, A.J., Fernandez-Mazuecos, M., Heleno, R.H., Vargas, P. (2022).
    PAICE: A new R package to estimate the number of inter-island colonizations
    considering haplotype data and sample size. \emph{Journal of Biogeography},
    49(4), 577-589.DOI:
    \href{https://onlinelibrary.wiley.com/doi/10.1111/jbi.14341}{10.1111/jbi.14341}
}


\examples{\donttest{
# Inference of minimum number of inter-island colonization events
data(CmonsData)
data(CmonsNetwork)
col <- colonization(data = CmonsData, network = CmonsNetwork)
col
summary(col)

# Asumptotic estimators of colonization events
# 25 replicates used in each sampling variable
set.seed(31)
CmonsRare <- rarecol(data = CmonsData, network = CmonsNetwork,
    replicates_field = 25, replicates_genetic = 25, monitor = TRUE,
    mode = c(1, 2))
maxcol <- maxCol(data = CmonsRare)
maxcol
summary(maxcol)

# Plotting results
old.par <- par(no.readonly = TRUE) # To restore previous options
par(mfrow = c(2, 2))
plot(CmonsRare)
par(fig = c(0, 1, 0, 0.5), new = TRUE)
plot(maxcol)
par(old.par)
}}

\keyword{ package }
