% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/plot_AbanicoPlot.R
\name{plot_AbanicoPlot}
\alias{plot_AbanicoPlot}
\title{Function to create an Abanico Plot.}
\usage{
plot_AbanicoPlot(
  data,
  na.rm = TRUE,
  log.z = TRUE,
  z.0 = "mean.weighted",
  dispersion = "qr",
  plot.ratio = 0.75,
  rotate = FALSE,
  mtext,
  summary,
  summary.pos,
  summary.method = "MCM",
  legend,
  legend.pos,
  stats,
  rug = FALSE,
  kde = TRUE,
  hist = FALSE,
  dots = FALSE,
  boxplot = FALSE,
  y.axis = TRUE,
  error.bars = FALSE,
  bar,
  bar.col,
  polygon.col,
  line,
  line.col,
  line.lty,
  line.label,
  grid.col,
  frame = 1,
  bw = "SJ",
  output = TRUE,
  interactive = FALSE,
  ...
)
}
\arguments{
\item{data}{\link{data.frame} or \linkS4class{RLum.Results} object (\strong{required}):
for \code{data.frame} two columns: De (\code{data[,1]}) and De error (\code{data[,2]}).
To plot several data sets in one plot the data sets must be provided as
\code{list}, e.g. \code{list(data.1, data.2)}.}

\item{na.rm}{\link{logical} (\emph{with default}):
exclude NA values from the data set prior to any further operations.}

\item{log.z}{\link{logical} (\emph{with default}):
Option to display the z-axis in logarithmic scale. Default is \code{TRUE}.}

\item{z.0}{\link{character} or \link{numeric}:
User-defined central value, used for centering of data. One out of \code{"mean"},
\code{"mean.weighted"} and \code{"median"} or a numeric value (not its logarithm).
Default is \code{"mean.weighted"}.}

\item{dispersion}{\link{character} (\emph{with default}):
measure of dispersion, used for drawing the scatter polygon. One out of
\itemize{
\item \code{"qr"} (quartile range),
\item \code{"pnn"} (symmetric percentile range with nn the lower percentile, e.g.
\item \code{"p05"} depicting the range between 5 and 95 \%),
\item \code{"sd"} (standard deviation) and
\item \code{"2sd"} (2 standard deviations),
}

The default is \code{"qr"}. Note that \code{"sd"} and \code{"2sd"} are only meaningful in
combination with \code{"z.0 = 'mean'"} because the unweighted mean is used to
center the polygon.}

\item{plot.ratio}{\link{numeric}:
Relative space, given to the radial versus the cartesian plot part,
default is \code{0.75}.}

\item{rotate}{\link{logical}:
Option to turn the plot by 90 degrees.}

\item{mtext}{\link{character}:
additional text below the plot title.}

\item{summary}{\link{character} (\emph{optional}):
add statistic measures of centrality and dispersion to the plot.
Can be one or more of several keywords. See details for available keywords.
Results differ depending on the log-option for the z-scale (see details).}

\item{summary.pos}{\link{numeric} or \link{character} (\emph{with default}):
optional position coordinates or keyword (e.g. \code{"topright"}) for the
statistical summary. Alternatively, the keyword \code{"sub"} may be
specified to place the summary below the plot header. However, this latter
option in only possible if \code{mtext} is not used.}

\item{summary.method}{\link{character} (\emph{with default}):
keyword indicating the method used to calculate the statistic summary.
One out of
\itemize{
\item \code{"unweighted"},
\item \code{"weighted"} and
\item \code{"MCM"}.
}

See \link{calc_Statistics} for details.}

\item{legend}{\link{character} vector (\emph{optional}):
legend content to be added to the plot.}

\item{legend.pos}{\link{numeric} or \link{character} (\emph{with default}):
optional position coordinates or keyword (e.g. \code{"topright"})
for the legend to be plotted.}

\item{stats}{\link{character}:
additional labels of statistically important values in the plot.
One or more out of the following:
\itemize{
\item \code{"min"},
\item \code{"max"},
\item \code{"median"}.
}}

\item{rug}{\link{logical}:
Option to add a rug to the KDE part, to indicate the location of individual values.}

\item{kde}{\link{logical}:
Option to add a KDE plot to the dispersion part, default is \code{TRUE}.}

\item{hist}{\link{logical}:
Option to add a histogram to the dispersion part. Only meaningful when not
more than one data set is plotted.}

\item{dots}{\link{logical}:
Option to add a dot plot to the dispersion part. If number of dots exceeds
space in the dispersion part, a square indicates this.}

\item{boxplot}{\link{logical}:
Option to add a boxplot to the dispersion part, default is \code{FALSE}.}

\item{y.axis}{\link{logical}:
Option to hide y-axis labels. Useful for data with small scatter.}

\item{error.bars}{\link{logical}:
Option to show De-errors as error bars on De-points. Useful in combination
with \verb{y.axis = FALSE, bar.col = "none"}.}

\item{bar}{\link{numeric} (\emph{with default}):
option to add one or more dispersion bars (i.e., bar showing the 2-sigma range)
centered at the defined values. By default a bar is drawn according to \code{"z.0"}.
To omit the bar set \code{"bar = FALSE"}.}

\item{bar.col}{\link{character} or \link{numeric} (\emph{with default}):
colour of the dispersion bar. Default is \code{"grey60"}.}

\item{polygon.col}{\link{character} or \link{numeric} (\emph{with default}):
colour of the polygon showing the data scatter. Sometimes this
polygon may be omitted for clarity. To disable it use \code{FALSE} or
\code{polygon = FALSE}. Default is \code{"grey80"}.}

\item{line}{\link{numeric}:
numeric values of the additional lines to be added.}

\item{line.col}{\link{character} or \link{numeric}:
colour of the additional lines.}

\item{line.lty}{\link{integer}:
line type of additional lines}

\item{line.label}{\link{character}:
labels for the additional lines.}

\item{grid.col}{\link{character} or \link{numeric} (\emph{with default}):
colour of the grid lines (originating at \verb{[0,0]} and strechting to
the z-scale). To disable grid lines use \code{FALSE}. Default is \code{"grey"}.}

\item{frame}{\link{numeric} (\emph{with default}):
option to modify the plot frame type. Can be one out of
\itemize{
\item \code{0} (no frame),
\item \code{1} (frame originates at 0,0 and runs along min/max isochrons),
\item \code{2} (frame embraces the 2-sigma bar),
\item \code{3} (frame embraces the entire plot as a rectangle).
}

Default is \code{1}.}

\item{bw}{\link{character} (\emph{with default}):
bin-width for KDE, choose a numeric value for manual setting.}

\item{output}{\link{logical}:
Optional output of numerical plot parameters. These can be useful to
reproduce similar plots. Default is \code{TRUE}.}

\item{interactive}{\link{logical} (\emph{with default}):
create an interactive abanico plot (requires the 'plotly' package)}

\item{...}{Further plot arguments to pass. \code{xlab} must be a vector of
length 2, specifying the upper and lower x-axes labels.}
}
\value{
returns a plot object and, optionally, a list with plot calculus data.
}
\description{
A plot is produced which allows comprehensive presentation of data precision
and its dispersion around a central value as well as illustration of a
kernel density estimate, histogram and/or dot plot of the dose values.
}
\details{
The Abanico Plot is a combination of the classic Radial Plot
(\code{plot_RadialPlot}) and a kernel density estimate plot (e.g
\code{plot_KDE}). It allows straightforward visualisation of data precision,
error scatter around a user-defined central value and the combined
distribution of the values, on the actual scale of the measured data (e.g.
seconds, equivalent dose, years). The principle of the plot is shown in
Galbraith & Green (1990). The function authors are thankful for the
thoughtprovocing figure in this article.

The semi circle (z-axis) of the classic Radial Plot is bent to a straight
line here, which actually is the basis for combining this polar (radial)
part of the plot with any other cartesian visualisation method
(KDE, histogram, PDF and so on). Note that the plot allows dispaying
two measures of distribution. One is the 2-sigma
bar, which illustrates the spread in value errors, and the other is the
polygon, which stretches over both parts of the Abanico Plot (polar and
cartesian) and illustrates the actual spread in the values themselves.

Since the 2-sigma-bar is a polygon, it can be (and is) filled with shaded
lines. To change density (lines per inch, default is 15) and angle (default
is 45 degrees) of the shading lines, specify these parameters. See
\code{?polygon()} for further help.

The Abanico Plot supports other than the weighted mean as measure of
centrality. When it is obvious that the data
is not (log-)normally distributed, the mean (weighted or not) cannot be a
valid measure of centrality and hence central dose. Accordingly, the median
and the weighted median can be chosen as well to represent a proper measure
of centrality (e.g. \code{centrality = "median.weighted"}). Also
user-defined numeric values (e.g. from the central age model) can be used if
this appears appropriate.

The proportion of the polar part and the cartesian part of the Abanico Plot
can be modfied for display reasons (\code{plot.ratio = 0.75}). By default,
the polar part spreads over 75 \\% and leaves 25 \\% for the part that
shows the KDE graph.

A statistic summary, i.e. a collection of statistic measures of
centrality and dispersion (and further measures) can be added by specifying
one or more of the following keywords:
\itemize{
\item \code{"n"} (number of samples)
\item \code{"mean"} (mean De value)
\item \code{"median"} (median of the De values)
\item \code{"sd.rel"} (relative standard deviation in percent)
\item \code{"sd.abs"} (absolute standard deviation)
\item \code{"se.rel"} (relative standard error)
\item \code{"se.abs"} (absolute standard error)
\item \code{"in.2s"} (percent of samples in 2-sigma range)
\item \code{"kurtosis"} (kurtosis)
\item \code{"skewness"} (skewness)
}

\strong{Note} that the input data for the statistic summary is sent to the function
\code{calc_Statistics()} depending on the log-option for the z-scale. If
\code{"log.z = TRUE"}, the summary is based on the logarithms of the input
data. If \code{"log.z = FALSE"} the linearly scaled data is used.

\strong{Note} as well, that \code{"calc_Statistics()"} calculates these statistic
measures in three different ways: \code{unweighted}, \code{weighted} and
\code{MCM-based} (i.e., based on Monte Carlo Methods). By default, the
MCM-based version is used. If you wish to use another method, indicate this
with the appropriate keyword using the argument \code{summary.method}.

The optional parameter \code{layout} allows to modify the entire plot more
sophisticated. Each element of the plot can be addressed and its properties
can be defined. This includes font type, size and decoration, colours and
sizes of all plot items. To infer the definition of a specific layout style
cf. \code{get_Layout()} or type eg. for the layout type \code{"journal"}
\code{get_Layout("journal")}. A layout type can be modified by the user by
assigning new values to the list object.

It is possible for the z-scale to specify where ticks are to be drawn
by using the parameter \code{at}, e.g. \code{at = seq(80, 200, 20)}, cf. function
documentation of \code{axis}. Specifying tick positions manually overrides a
\code{zlim}-definition.
}
\section{Function version}{
 0.1.11
}

\examples{

## load example data and recalculate to Gray
data(ExampleData.DeValues, envir = environment())
ExampleData.DeValues <- ExampleData.DeValues$CA1

## plot the example data straightforward
plot_AbanicoPlot(data = ExampleData.DeValues)

## now with linear z-scale
plot_AbanicoPlot(data = ExampleData.DeValues,
                 log.z = FALSE)

## now with output of the plot parameters
plot1 <- plot_AbanicoPlot(data = ExampleData.DeValues,
                          output = TRUE)
str(plot1)
plot1$zlim

## now with adjusted z-scale limits
plot_AbanicoPlot(data = ExampleData.DeValues,
                 zlim = c(10, 200))

## now with adjusted x-scale limits
plot_AbanicoPlot(data = ExampleData.DeValues,
                 xlim = c(0, 20))

## now with rug to indicate individual values in KDE part
plot_AbanicoPlot(data = ExampleData.DeValues,
                 rug = TRUE)

## now with a smaller bandwidth for the KDE plot
plot_AbanicoPlot(data = ExampleData.DeValues,
                 bw = 0.04)

## now with a histogram instead of the KDE plot
plot_AbanicoPlot(data = ExampleData.DeValues,
                 hist = TRUE,
                 kde = FALSE)

## now with a KDE plot and histogram with manual number of bins
plot_AbanicoPlot(data = ExampleData.DeValues,
                 hist = TRUE,
                 breaks = 20)

## now with a KDE plot and a dot plot
plot_AbanicoPlot(data = ExampleData.DeValues,
                 dots = TRUE)

## now with user-defined plot ratio
plot_AbanicoPlot(data = ExampleData.DeValues,
                 plot.ratio = 0.5)
## now with user-defined central value
plot_AbanicoPlot(data = ExampleData.DeValues,
                 z.0 = 70)

## now with median as central value
plot_AbanicoPlot(data = ExampleData.DeValues,
                 z.0 = "median")

## now with the 17-83 percentile range as definition of scatter
plot_AbanicoPlot(data = ExampleData.DeValues,
                 z.0 = "median",
                 dispersion = "p17")

## now with user-defined green line for minimum age model
CAM <- calc_CentralDose(ExampleData.DeValues,
                        plot = FALSE)

plot_AbanicoPlot(data = ExampleData.DeValues,
                 line = CAM,
                 line.col = "darkgreen",
                 line.label = "CAM")

## now create plot with legend, colour, different points and smaller scale
plot_AbanicoPlot(data = ExampleData.DeValues,
                 legend = "Sample 1",
                 col = "tomato4",
                 bar.col = "peachpuff",
                 pch = "R",
                 cex = 0.8)

## now without 2-sigma bar, polygon, grid lines and central value line
plot_AbanicoPlot(data = ExampleData.DeValues,
                 bar.col = FALSE,
                 polygon.col = FALSE,
                 grid.col = FALSE,
                 y.axis = FALSE,
                 lwd = 0)

## now with direct display of De errors, without 2-sigma bar
plot_AbanicoPlot(data = ExampleData.DeValues,
                 bar.col = FALSE,
                 ylab = "",
                 y.axis = FALSE,
                 error.bars = TRUE)

## now with user-defined axes labels
plot_AbanicoPlot(data = ExampleData.DeValues,
                 xlab = c("Data error (\%)",
                          "Data precision"),
                 ylab = "Scatter",
                 zlab = "Equivalent dose [Gy]")

## now with minimum, maximum and median value indicated
plot_AbanicoPlot(data = ExampleData.DeValues,
                 stats = c("min", "max", "median"))

## now with a brief statistical summary as subheader
plot_AbanicoPlot(data = ExampleData.DeValues,
                 summary = c("n", "in.2s"))

## now with another statistical summary
plot_AbanicoPlot(data = ExampleData.DeValues,
                 summary = c("mean.weighted", "median"),
                 summary.pos = "topleft")

## now a plot with two 2-sigma bars for one data set
plot_AbanicoPlot(data = ExampleData.DeValues,
                 bar = c(30, 100))

## now the data set is split into sub-groups, one is manipulated
data.1 <- ExampleData.DeValues[1:30,]
data.2 <- ExampleData.DeValues[31:62,] * 1.3

## now a common dataset is created from the two subgroups
data.3 <- list(data.1, data.2)

## now the two data sets are plotted in one plot
plot_AbanicoPlot(data = data.3)

## now with some graphical modification
plot_AbanicoPlot(data = data.3,
                 z.0 = "median",
                 col = c("steelblue4", "orange4"),
                 bar.col = c("steelblue3", "orange3"),
                 polygon.col = c("steelblue1", "orange1"),
                 pch = c(2, 6),
                 angle = c(30, 50),
                 summary = c("n", "in.2s", "median"))

## create Abanico plot with predefined layout definition
plot_AbanicoPlot(data = ExampleData.DeValues,
                 layout = "journal")

## now with predefined layout definition and further modifications
plot_AbanicoPlot(data = data.3,
                 z.0 = "median",
                 layout = "journal",
                 col = c("steelblue4", "orange4"),
                 bar.col = adjustcolor(c("steelblue3", "orange3"),
                                       alpha.f = 0.5),
                 polygon.col = c("steelblue3", "orange3"))

## for further information on layout definitions see documentation
## of function get_Layout()

## now with manually added plot content
## create empty plot with numeric output
AP <- plot_AbanicoPlot(data = ExampleData.DeValues,
                       pch = NA,
                       output = TRUE)

## identify data in 2 sigma range
in_2sigma <- AP$data[[1]]$data.in.2s

## restore function-internal plot parameters
par(AP$par)

## add points inside 2-sigma range
points(x = AP$data[[1]]$precision[in_2sigma],
       y = AP$data[[1]]$std.estimate.plot[in_2sigma],
       pch = 16)

## add points outside 2-sigma range
points(x = AP$data[[1]]$precision[!in_2sigma],
       y = AP$data[[1]]$std.estimate.plot[!in_2sigma],
       pch = 1)

} 

\section{How to cite}{
Dietze, M., Kreutzer, S., 2020. plot_AbanicoPlot(): Function to create an Abanico Plot.. Function version 0.1.11. In: Kreutzer, S., Burow, C., Dietze, M., Fuchs, M.C., Schmidt, C., Fischer, M., Friedrich, J., Riedesel, S., Autzen, M., 2020. Luminescence: Comprehensive Luminescence Dating Data Analysis. R package version 0.9.8. https://r-luminescence.org
}

\references{
Galbraith, R. & Green, P., 1990. Estimating the component ages
in a finite mixture. International Journal of Radiation Applications and
Instrumentation. Part D. Nuclear Tracks and Radiation Measurements, 17 (3),
197-206.

Dietze, M., Kreutzer, S., Burow, C., Fuchs, M.C., Fischer, M., Schmidt, C., 2015.
The abanico plot: visualising chronometric data with individual standard errors.
Quaternary Geochronology. doi:10.1016/j.quageo.2015.09.003
}
\seealso{
\link{plot_RadialPlot}, \link{plot_KDE}, \link{plot_Histogram}
}
\author{
Michael Dietze, GFZ Potsdam (Germany)\cr
Sebastian Kreutzer, RAMAT-CRP2A, Universite Bordeaux Montaigne (France)\cr
Inspired by a plot introduced by Galbraith & Green (1990)
, RLum Developer Team}
