\name{skeleton}
\alias{skeleton}
\alias{print.skeleton}
\title{Create a Translation Template for Sumerian Text}
\description{
Creates a structured template (skeleton) for translating Sumerian text. The template displays each token and subexpression with its syllabic reading, sign name, and cuneiform representation, providing a framework for adding translations.

The input may contain three types of brackets to control how the template is generated (see Details). Optionally, the template can be pre-filled with translations from one or more dictionaries using \code{\link{guess_substr_info}}.

The function \code{skeleton} computes the template and returns an object of class \code{"skeleton"}. The \code{print} method displays the template in the console.
}
\usage{
skeleton(x, mapping = NULL, fill = NULL, space = FALSE)

\method{print}{skeleton}(x, ...)
}
\arguments{
  \item{x}{For \code{skeleton}: A character string of length 1 containing transliterated Sumerian text (transliteration, sign names, or cuneiform characters). Tokens may be grouped with brackets to control template generation (see Details).

  For \code{print.skeleton}: An object of class \code{"skeleton"} as returned by \code{skeleton}.}
  \item{mapping}{A data frame containing the sign mapping table with columns \code{syllables}, \code{name}, and \code{cuneiform}. If \code{NULL} (the default), the package's internal mapping file \file{etcsl_mapping.txt} is loaded.}
  \item{fill}{A data frame as returned by \code{\link{guess_substr_info}}, containing translations and grammatical types for all substrings of \code{x}. If provided, the template lines are pre-filled with the corresponding type and translation. If \code{NULL} (the default), the template lines are left empty.}
  \item{space}{Logical. If \code{TRUE}, an empty line is inserted before each entry at nesting depth 1, visually separating top-level groups. Defaults to \code{FALSE}.}
  \item{...}{Additional arguments passed to the print method (currently unused).}
}
\details{
The function generates a hierarchical template from a Sumerian text string. The input is first converted to cuneiform with \code{\link{as.cuneiform}}. The input string may contain three types of brackets that control how entries in the template are generated:

\describe{
  \item{Angle brackets \code{< >}}{The enclosed token sequence is treated as a fixed term. No individual skeleton entries are generated for the tokens inside. For example, \code{<d-nu-dim2-mud>} is treated as a single unit.}
  \item{Round brackets \code{( )}}{The enclosed token sequence is a coherent term for which a single skeleton entry is generated, in addition to entries for its individual tokens. Nesting is allowed.}
  \item{Curly braces \code{\{ \}}}{Ignored during skeleton generation. They can be used in the input to indicate which tokens serve as arguments to an operator, but this information is not needed for the skeleton.}
}

In addition, a skeleton entry is generated for every individual token that does not appear inside angle brackets.

Each line in the resulting template follows the format:

\code{|[tabs]reading=SIGN.NAME=cuneiform:type:translation}

When \code{fill} is not provided, the type and translation fields are left empty:

\code{|[tabs]reading=SIGN.NAME=cuneiform::}

The template should then be filled in as follows:
\itemize{
  \item Between the two colons: the grammatical type of the expression (e.g., \code{S} for noun phrases, \code{V} for verbs). See \code{\link{make_dictionary}} for details.
  \item After the second colon: the translation.
}

The indentation level (number of tabs) reflects the nesting depth: top-level entries have no indentation, their sub-entries have one tab, and so on.

The template format is designed to be saved as a text file (.txt) or Word document (.docx), edited manually, and then used as input for \code{\link{make_dictionary}} to create a custom dictionary.

If \code{fill} is provided, the function validates that \code{fill} matches \code{x}: the cuneiform tokens of the first row in \code{fill} must be identical to the tokens of \code{x}, and the number of rows must equal \eqn{N(N+1)/2} where \eqn{N} is the number of tokens.
}
\value{
\code{skeleton} returns a character vector of class \code{c("skeleton", "character")} containing the template lines. The first line is the header with the full reading of the input, followed by one line per skeleton entry. If \code{space = TRUE}, empty strings are inserted as separator lines.

\code{print.skeleton} prints the template to the console (one line per element) and returns \code{x} invisibly.
}
\seealso{
\code{\link{guess_substr_info}} for generating the \code{fill} data frame,
\code{\link{mark_skeleton_entries}} for the bracket normalization step,
\code{\link{extract_skeleton_entries}} for the hierarchical extraction step,
\code{\link{substr_position}} for computing row indices in the fill data frame,
\code{\link{look_up}} for looking up translations of Sumerian signs and words,
\code{\link{make_dictionary}} for creating a dictionary from filled-in templates,
\code{\link{info}} for retrieving detailed sign information
}
\examples{
# Create an empty template
x <- "<d-nu-dim2-mud> ki a. jal2 (e2{kur}) ra. gaba jal2. an ki a"
skeleton(x)

# Pre-fill the template with dictionary translations
dic <- read_dictionary()
fill <- guess_substr_info(x, dic)
skeleton(x, fill = fill)

# Use spacing to visually separate top-level groups
skeleton(x, fill = fill, space = TRUE)
}
\keyword{utilities}
\keyword{character}
