extract_selected_datetime_range <- function(range, parent_range, reset) {
  if (identical(range, c(Inf, -Inf)) || length(range) == 0) {
    return(range)
  }
  
  if (inherits(range, "character") || inherits(range, "POSIXct")) {
    if (length(range) == 1) range <- c(range, Inf)
    
    range <- as.POSIXct(range, origin = "1970-01-01 UTC")
    parent_range <- as.POSIXct(parent_range, origin = "1970-01-01 UTC")
  }
  
  if (reset || identical(range, NA) || !any(dplyr::between(range, parent_range[1], parent_range[2]))) {
    return(parent_range)
  }
  if (anyNA(range[1]) || range[1] < parent_range[1]) {
    range[1] <- parent_range[1]
  }
  if (anyNA(range[2]) || range[2] > parent_range[2]) {
    range[2] <- parent_range[2]
  }
  
  return(range)
}

#' @rdname gui-filter-layer
#' @export
.gui_filter.datetime_range <- function(filter, ...) {
  list(
    input = function(input_id, cohort) {
      input_params <- range_input_params(filter, input_id, cohort, ...)
      shiny::tagList(
        if (is_gui_type(filter, "datetimepicker")) {
          input_params$minDate <- input_params$min
          input_params$min <- NULL
          input_params$maxDate <- input_params$max
          input_params$max <- NULL
          input_params$step <- NULL
          .cb_input(
            do.call(
              shinyWidgets::airDatepickerInput,
              modify_list(
                list(
                  range = TRUE, timepicker = TRUE, update_on = "close",
                  autoClose = TRUE, addon = "none"
                ),
                suff_id(input_params, "datetimepicker")
              )
            ),
            filter$input_param
          )
        } else if (is_gui_type(filter, "slider")) {
          .cb_input(
            do.call(
              shiny::sliderInput,
              modify_list(
                list(ticks = FALSE, round = 1),
                suff_id(input_params, "slider")
              )
            ),
            filter$input_param
          )
        },
        .cb_input(
          .keep_na_input(input_id, filter, cohort),
          "keep_na"
        )
      )
    },
    
    feedback = function(input_id, cohort, empty = FALSE) {
      list(
        plot_id = shiny::NS(input_id, "feedback_plot") ,
        output_fun = shiny::plotOutput,
        render_fun = if (!is.null(empty)) {
          shiny::renderPlot(bg = "transparent", height = 60, {
            if(empty || is.null(filter$get_params("range"))) { # when no data in parent step
              return(
                ggplot2::ggplot()
              )
            }
            step_id <- filter$step_id
            filter_id <- filter$id
            
            filter_cache <- cohort$get_cache(step_id, filter_id, state = "pre")

            filter_range <- extract_selected_datetime_range(
              filter$get_params("range"),
              freq_range(filter_cache$frequencies),
              FALSE
            )
            
            plot_data <- filter_cache$frequencies %>%
              dplyr::mutate(# we take l_bound to limit upper cause last break have l_bound == u_bound
                count = ifelse(l_bound >= filter_range[1] & l_bound <= filter_range[2], count, 0)
              ) 
            n_missing <- filter_cache$n_missing
            n_total <- filter_cache$n_data
            if (identical(filter$get_params("keep_na"), FALSE)) {
              n_missing <- 0
            }
            
            plot_feedback_hist(plot_data, n_missing, n_total)
          })
        }
      )
    },
    server = function(input_id, input, output, session, cohort) {},
    update = function(session, input_id, cohort, reset = FALSE, ...) {
      input_params <- append(
        list(session = session),
        range_input_params(filter, input_id, cohort, reset, TRUE, ...)
      )
      if (is_gui_type(filter, "datetimepicker")) {
        input_params$min <- NULL
        input_params$max <- NULL
        input_params$step <- NULL
        do.call(
          shinyWidgets::updateAirDateInput,
          suff_id(input_params, "slider")
        )
      } else if (is_gui_type(filter, "slider")) {
        do.call(
          shiny::updateSliderInput,
          suff_id(input_params, "slider")
        )
      }  
      .update_keep_na_input(session, input_id, filter, cohort)
    },
    
    post_stats = FALSE,
    multi_input = FALSE
  )
}
