#ifndef COXREGRESSION_HPP_
#define COXREGRESSION_HPP_

#include "DataSet.hpp"
#include "CoxRegressionResult.hpp"
// #include <list>

class CoxRegression {
private:
    /**
     * The data set.
     */
    DataSet data;
    
    /**
     * The data matrix.
     */
    arma::mat dataMat;

    /**
     * The variables.
     */
    std::vector<Node> variables;

    /**
     * The significance level for determining which regressors are significant
     * based on their p values.
     */
    double alpha = 0.05;

    /**
     * The graph of significant regressors into the target.
     */

    arma::uvec rows;

    int maxIter = 100;


public:
    //============================CONSTRUCTORS==========================//
    CoxRegression() {}
    CoxRegression(DataSet& data);
    CoxRegression(const CoxRegression& other) = default;
    CoxRegression(CoxRegression&& other) = default;
    // Coxregression(arma::mat data, std::vector<Node>&  variables);

    CoxRegression& operator=(const CoxRegression& other) = default;
    CoxRegression& operator=(CoxRegression&& other) = default;

    //===========================PUBLIC METHODS========================//

    /**
     * Sets the alpha level for deciding which regressors are significant
     * based on their p values.
     */
    void setAlpha(double alpha) {this->alpha = alpha;}

    /**
     * Cox regression of the target given the regressors.
     *
     * @param target     The target variable.
     * @param regressors The regressor variables.
     * @return The regression plane, specifying for each regressors its
     * coefficeint, se, t, and p values, and specifying the same for the
     * constant.
     */
    CoxRegressionResult regress(const Node& target, std::vector<Node>& regressors);

    /**
     * Cox regression of the target given the regressors.
     *
     * @param target     The target variable.
     * @param regressors The regressor variables.
     * @param _rows The samples to be used in regression.
     * @return The regression plane, specifying for each regressors its
     * coefficeint, se, t, and p values, and specifying the same for the
     * constant.
     */
    CoxRegressionResult regress(const Node& target, std::vector<Node>& regressors,
				arma::uvec& _rows);

    double loss(arma::vec& beta, arma::mat& X, const Node& target);

    double gradHess(arma::vec& beta, arma::vec& grad, arma::mat& hess,
		    arma::mat& X, const Node& target);

    double etaGradHess(arma::vec& eta, arma::vec& grad, arma::vec& diagHess,
		       const Node& target);

    arma::uvec getRows() {return rows;}

    void setRows(arma::uvec rows) {this->rows = rows;}

    // arma::vec getResidualsWithoutFirstRegressor() {return res2;}

    friend void CoxRegressionTest(const Rcpp::DataFrame& df, std::string targetName,
				  std::vector<std::string>& regressorNames, int repetitions);

};
#endif /* COXREGRESSION_HPP_ */
