# Meta-analytic latent variable model (single-stage random effects)
meta_lvm <- function(
  covs, # List of covariance matrices as input. Must contain NAs for missing variables
  nobs, # vector of sample sizes as input

  # New standardized input arguments:
  data, # Raw data (use with studyvar to split by study)
  cors, # List of correlation matrices (treated as covariances with warning)
  studyvar, # Column name in data indicating study membership
  groups, # deprecated, use groupvar instead
  groupvar, # grouping variable (errors: multi-group not yet supported)
  corinput, # defaults to FALSE for meta_lvm

  Vmats, # Optional list of V matrices for each group. Will be averaged.
  Vmethod = c("individual","pooled","metaSEM_individual","metaSEM_weighted"), # How to obtain V matrices if Vmats is not supplied?
  Vestimation = c("averaged","per_study"),

  # LVM structure:
  lambda, # REQUIRED: factor loading matrix (nvar x nlat), or a pattern matrix
  beta = "zero", # latent regression matrix

  # Latent covariance structure:
  latent = c("cov","chol","prec","ggm"),
  sigma_zeta = "full",
  kappa_zeta = "full",
  omega_zeta = "full",
  lowertri_zeta = "full",
  delta_zeta = "full",

  # Residual covariance structure:
  residual = c("cov","chol","prec","ggm"),
  sigma_epsilon = "diag",
  kappa_epsilon = "diag",
  omega_epsilon = "zero",
  lowertri_epsilon = "diag",
  delta_epsilon = "diag",

  # Identification:
  identify = TRUE,
  identification = c("loadings","variance"),
  
  # Random effects setup:
  randomEffects = c("chol","cov","prec","ggm","cor"),
  sigma_randomEffects = "full",
  kappa_randomEffects = "full",
  omega_randomEffects = "full",
  lowertri_randomEffects = "full",
  delta_randomEffects = "full",
  rho_randomEffects = "full",
  SD_randomEffects = "full",

  # Naming:
  vars, # character vector of variable names
  latents, # Name of latent variables

  # Some extra stuff:
  baseline_saturated = TRUE,
  optimizer,
  estimator = c("FIML","ML"),

  sampleStats,
  verbose = FALSE,
  bootstrap = FALSE,
  boot_sub,
  boot_resample
){
  
  message(paste0("Note: 'meta_lvm()' is experimental in psychonetrics ",
                 utils::packageVersion("psychonetrics"),
                 ". Please report any unexpected behavior to https://github.com/SachaEpskamp/psychonetrics/issues"))

  # Standardize input arguments:
  si <- standardize_input(
    data = if(missing(data)) NULL else data,
    covs = if(missing(covs)) NULL else covs,
    cors = if(missing(cors)) NULL else cors,
    nobs = if(missing(nobs)) NULL else nobs,
    corinput = if(missing(corinput)) NULL else corinput,
    groups = if(missing(groups)) NULL else groups,
    groupvar = if(missing(groupvar)) NULL else groupvar,
    studyvar = if(missing(studyvar)) NULL else studyvar,
    vars = if(missing(vars)) NULL else vars,
    family = "meta_lvm", is_meta = TRUE,
    caller = "meta_lvm()", estimator = match.arg(estimator)
  )
  covs <- si$covs
  nobs <- si$nobs
  corinput <- if(!is.null(si$corinput)) si$corinput else FALSE

  sampleSizes <- nobs
  estimator <- match.arg(estimator)

  randomEffects <- match.arg(randomEffects)
  latent <- match.arg(latent)
  residual <- match.arg(residual)
  Vmethod <- match.arg(Vmethod)
  Vestimation <- match.arg(Vestimation)
  identification <- match.arg(identification)

  # lambda is required:
  if (missing(lambda)){
    stop("'lambda' may not be missing")
  }
  if (is.character(lambda)){
    stop("'lambda' may not be a string")
  }

  # Set the labels:
  if (missing(vars)){
    vars <- unique(unlist(lapply(covs,colnames)))
    if (is.null(vars)){
      vars <- paste0("V",seq_len(max(sapply(covs,ncol))))
      if (length(unique(sapply(covs,colnames))) > 1){
        stop("Correlation matrices of different dimensions not supported without column labels.")
      }
      for (i in seq_along(covs)){
        rownames(covs[[i]]) <- colnames(covs[[i]]) <- vars
      }
    }
  } else {
    if (is.null(colnames(covs[[1]]))){
      for (i in seq_along(covs)){
        rownames(covs[[i]]) <- colnames(covs[[i]]) <- vars
      }
    }
  }

  # Number of nodes:
  nNode <- length(vars)

  # Reorder correlation matrices and add NAs:
  covsOld <- covs
  covs <- list()
  for (i in seq_along(covsOld)){
    covs[[i]] <- matrix(NA,nrow=nNode, ncol = nNode)
    rownames(covs[[i]]) <- colnames(covs[[i]]) <- vars
    varsOfStudy <- colnames(covsOld[[i]])
    matched <- match(varsOfStudy,vars)
    covs[[i]][matched,matched] <- as.matrix(covsOld[[i]])
  }

  # Form the dataset from the lower triangles of cor matrices:
  data <- dplyr::bind_rows(lapply(covs,function(x){
    df <- as.data.frame(t(x[lower.tri(x,diag=!corinput)]))
    names(df) <- paste0(rownames(x)[row(x)[lower.tri(x,diag=!corinput)]], " -- ",colnames(x)[col(x)[lower.tri(x,diag=!corinput)]])
    df
  }))

  # Labels for the correlations:
  corvars <- colnames(data)

  # Obtain sample stats:
  if (missing(sampleStats)){
    sampleStats <- samplestats(data = data,
                               vars = corvars,
                               missing = "pairwise",
                               fimldata = estimator == "FIML",
                               storedata = FALSE,
                               meanstructure = TRUE,
                               verbose=verbose,
                               fullFIML = (Vestimation == "per_study"),
                               bootstrap=bootstrap,
                               boot_sub = boot_sub,
                               boot_resample = boot_resample)
    
    # FIXME: If covs and not cors, columns with 1 appear and this error appears:
    # 
    # Warning message:
    #   In cov2cor(cov) :
    #   diag(V) had non-positive or NA entries; the non-finite result may be dubious
    
    # Because cov matrix is not invertable.
  }

  # Treat correlations as covariances (no corinput constraint):
  sampleStats@corinput <- FALSE

  # Check some things:
  nCov <- nrow(sampleStats@variables)

  # Number of latents:
  nLatent <- ncol(lambda)

  # If latents is not provided, make it:
  if (missing(latents)){
    latents <- paste0("Eta_",seq_len(nLatent))
  }
  if (length(latents) != nLatent){
    stop("Length of 'latents' is not equal to number of latent variables in model.")
  }


  # Generate model object:
  model <- generate_psychonetrics(model = "meta_lvm", sample = sampleStats, computed = FALSE,
                                  optimizer =  defaultoptimizer(), estimator = estimator, distribution = "Gaussian",
                                  types = list(latent = latent, residual = residual, randomEffects = randomEffects),
                                  meanstructure = TRUE, verbose = verbose,identification = identification)

  # Number of groups:
  nGroup <- 1 # FIXME: Multiple groups not possible?

  # Number of means:
  nMeans <- sum(sapply(model@sample@means,function(x)sum(!is.na(x))))

  # Add number of observations:
  model@sample@nobs <-
    nCov * (nCov-1) / 2 * nGroup + # Covariances per group
    nCov * nGroup + # Variances
    nMeans

  ### Estimate V matrices ###
  if (!missing(Vmats)){
    avgVmat <- Reduce("+", Vmats) / length(Vmats)
  } else {
    if (verbose){
      message("Computing sampling error approximation...")
    }

    # For the elimination matrix, I need this dummy matrix with indices:
    dumSig <- matrix(0,nNode,nNode)
    dumSig[lower.tri(dumSig,diag=!corinput)] <- seq_len(sum(lower.tri(dumSig,diag=!corinput)))

    if (Vmethod == "individual"){

      # For each group, make a model and obtain VCOV:
      Vmats <- lapply(seq_along(covs),function(i){
        # Find the missing nodes:
        obs <- !apply(covs[[i]],2,function(x)all(is.na(x)))

        # Indices:
        inds <- c(dumSig[obs,obs,drop=FALSE])
        inds <- inds[inds!=0]

        # Elimination matrix:
        L <- sparseMatrix(i=seq_along(inds),j=inds,dims=c(length(inds),nNode*(nNode + ifelse(corinput,-1,1))/2))
        L <- as(L, "dMatrix")

        # Now obtain only the full subset correlation matrix:
        cmat <- as(covs[[i]][obs,obs], "matrix")

        k <- solve_symmetric_cpp_matrixonly(cmat)
        D2 <- duplicationMatrix(ncol(cmat), !corinput)
        v <- 0.5 * nobs[i] * t(D2) %*% (k %x% k) %*% D2
        vcov <- solve_symmetric_cpp_matrixonly(as.matrix(v))

        # Now expand using the elimination matrix:
        res <- as.matrix(t(L) %*% vcov %*% L)
        return(0.5 * (res + t(res)))
      })

      avgVmat <- Reduce("+", Vmats) / Reduce("+",lapply(Vmats,function(x)x!=0))

    } else if (Vmethod == "pooled") {
      # If there are any NAs, use covariance input to compute model using FIML:
      if(any(is.na(unlist(lapply(covs,as.vector))))){
        mod <- varcov(covs=covs,nobs=sampleSizes, corinput = FALSE, type =  ifelse(corinput,"cor","cov"), equal = c("sigma","SD","rho","mu"), baseline_saturated = FALSE, verbose = FALSE,
                      estimator = "FIML", covtype = "ML", meanstructure = TRUE)
        mod <- runmodel(mod, addfit = FALSE, addMIs = FALSE, addSEs = FALSE, verbose = FALSE)
        # acov <- getVCOV(mod)
        # avgVmat <- acov
        # ind <- which(mod@parameters$matrix[match(seq_len(max(mod@parameters$par)),mod@parameters$par)] == ifelse(corinput,"rho","sigma"))
        # avgVmat <- acov[ind,ind] * length(covs)
      } else {
        mod <- varcov(covs=covs,nobs=sampleSizes, corinput = FALSE, type =  ifelse(corinput,"cor","cov"), equal = c("sigma","rho"), baseline_saturated = FALSE, verbose = FALSE)
        mod <- runmodel(mod, addfit = FALSE, addMIs = FALSE, addSEs = FALSE, verbose = FALSE)
        # acov <- getVCOV(mod)
        # avgVmat <- acov * length(covs)
      }
      acov <- getVCOV(mod)
      avgVmat <- acov
      ind <- which(mod@parameters$matrix[match(seq_len(max(mod@parameters$par)),mod@parameters$par)] == ifelse(corinput,"rho","sigma"))
      avgVmat <- acov[ind,ind] * length(covs)

      # Compute Vmat per dataset:
      Vmats <- lapply(nobs,function(n) mean(nobs)/n * avgVmat)

    } else if (Vmethod == "metaSEM_individual"){
      acovs <- metaSEM::asyCov(covs, cor.analysis = corinput, sampleSizes, acov = "individual")
      acovs[is.na(acovs)] <- 0
      Vmats <- list()
      for (i in seq_len(nrow(acovs))){
        Vmats[[i]] <- matrix(0, nCov, nCov)
        Vmats[[i]][lower.tri(Vmats[[i]],diag=TRUE)] <- acovs[i,]
        Vmats[[i]][upper.tri(Vmats[[i]],diag=TRUE)] <- t(Vmats[[i]])[upper.tri(Vmats[[i]],diag=TRUE)]
      }
      avgVmat <- Reduce("+", Vmats) / Reduce("+",lapply(Vmats,function(x)x!=0))

    } else if (Vmethod == "metaSEM_weighted"){
      acovs <- metaSEM::asyCov(covs, cor.analysis = corinput, sampleSizes, acov = "weighted")
      acovs[is.na(acovs)] <- 0
      Vmats <- list()
      for (i in seq_len(nrow(acovs))){
        Vmats[[i]] <- matrix(0, nCov, nCov)
        Vmats[[i]][lower.tri(Vmats[[i]],diag=TRUE)] <- acovs[i,]
        Vmats[[i]][upper.tri(Vmats[[i]],diag=TRUE)] <- t(Vmats[[i]])[upper.tri(Vmats[[i]],diag=TRUE)]
      }
      avgVmat <- Reduce("+", Vmats) / Reduce("+",lapply(Vmats,function(x)x!=0))
    }
  }

  #### LVM Model matrices ####
  modMatrices <- list()

  # Setup lambda (using dummy covariance matrices for starting values):
  # Construct dummy covariance matrices from the average correlation:
  expcovsVec <- model@sample@means[[1]]
  expcovs <- matrix(1,nNode,nNode)
  expcovs[lower.tri(expcovs, diag = !corinput)] <- expcovsVec
  expcovs[upper.tri(expcovs, diag = !corinput)] <- t(expcovs)[upper.tri(expcovs, diag = !corinput)]

  modMatrices$lambda <- matrixsetup_lambda(lambda, expcov=list(expcovs), nGroup = nGroup, equal = FALSE,
                                           observednames = vars, latentnames = latents,
                                           sampletable = sampleStats, identification = identification, simple = FALSE)

  # Compute the expected latent and residual cov matrices from lambda setup:
  expLatSigma <- lapply(1:nGroup,function(x)matrix(0,nLatent,nLatent))
  expResidSigma <- lapply(1:nGroup,function(x)matrix(0,nNode,nNode))

  for (g in 1:nGroup){
    expResidSigma[[g]] <- modMatrices$lambda$sigma_epsilon_start[,,g]
    expLatSigma[[g]] <-  modMatrices$lambda$sigma_zeta_start[,,g]
  }

  # Setup beta:
  modMatrices$beta <- matrixsetup_beta(beta, nNode = nLatent, nGroup = nGroup,
                                        labels = latents, sampletable = sampleStats,
                                        equal = FALSE)

  # Latent varcov:
  if (latent == "cov"){
    modMatrices$sigma_zeta <- matrixsetup_sigma(sigma_zeta,
                                                name = "sigma_zeta",
                                                expcov=expLatSigma,
                                                nNode = nLatent,
                                                nGroup = nGroup,
                                                labels = latents,
                                                equal = FALSE, sampletable = sampleStats,
                                                beta = modMatrices$beta[[1]])
  } else if (latent == "chol"){
    modMatrices$lowertri_zeta <- matrixsetup_lowertri(lowertri_zeta,
                                                      name = "lowertri_zeta",
                                                      expcov=expLatSigma,
                                                      nNode = nLatent,
                                                      nGroup = nGroup,
                                                      labels = latents,
                                                      equal = FALSE, sampletable = sampleStats,
                                                      beta = modMatrices$beta[[1]])
  } else if (latent == "ggm"){
    modMatrices$omega_zeta <- matrixsetup_omega(omega_zeta,
                                                name = "omega_zeta",
                                                expcov=expLatSigma,
                                                nNode = nLatent,
                                                nGroup = nGroup,
                                                labels = latents,
                                                equal = FALSE, sampletable = sampleStats,
                                                beta = modMatrices$beta[[1]])
    modMatrices$delta_zeta <- matrixsetup_delta(delta_zeta,
                                                name = "delta_zeta",
                                                expcov=expLatSigma,
                                                nNode = nLatent,
                                                nGroup = nGroup,
                                                labels = latents,
                                                equal = FALSE, sampletable = sampleStats,
                                                omegaStart =  modMatrices$omega_zeta$start)
  } else if (latent == "prec"){
    modMatrices$kappa_zeta <- matrixsetup_kappa(kappa_zeta,
                                                name = "kappa_zeta",
                                                expcov=expLatSigma,
                                                nNode = nLatent,
                                                nGroup = nGroup,
                                                labels = latents,
                                                equal = FALSE, sampletable = sampleStats,
                                                beta = modMatrices$beta[[1]])
  }

  ### Residual varcov ###
  if (residual == "cov"){
    modMatrices$sigma_epsilon <- matrixsetup_sigma(sigma_epsilon,
                                                   name = "sigma_epsilon",
                                                   expcov=expResidSigma,
                                                   nNode = nNode,
                                                   nGroup = nGroup,
                                                   labels = vars,
                                                   equal = FALSE, sampletable = sampleStats)
  } else if (residual == "chol"){
    modMatrices$lowertri_epsilon <- matrixsetup_lowertri(lowertri_epsilon,
                                                         name = "lowertri_epsilon",
                                                         expcov=expResidSigma,
                                                         nNode = nNode,
                                                         nGroup = nGroup,
                                                         labels = vars,
                                                         equal = FALSE, sampletable = sampleStats)
  } else if (residual == "ggm"){
    modMatrices$omega_epsilon <- matrixsetup_omega(omega_epsilon,
                                                   name = "omega_epsilon",
                                                   expcov=expResidSigma,
                                                   nNode = nNode,
                                                   nGroup = nGroup,
                                                   labels = vars,
                                                   equal = FALSE, sampletable = sampleStats)
    modMatrices$delta_epsilon <- matrixsetup_delta(delta_epsilon,
                                                   name = "delta_epsilon",
                                                   expcov=expResidSigma,
                                                   nNode = nNode,
                                                   nGroup = nGroup,
                                                   labels = vars,
                                                   equal = FALSE, sampletable = sampleStats,
                                                   omegaStart =  modMatrices$omega_epsilon$start)
  } else if (residual == "prec"){
    modMatrices$kappa_epsilon <- matrixsetup_kappa(kappa_epsilon,
                                                   name = "kappa_epsilon",
                                                   expcov=expResidSigma,
                                                   nNode = nNode,
                                                   nGroup = nGroup,
                                                   labels = vars,
                                                   equal = FALSE, sampletable = sampleStats)
  }

  #### Random effects matrices ####
  # Compute expected random effects matrix:
  expRanEffects <- as.matrix(spectralshift(sampleStats@covs[[1]] - avgVmat))

  if (randomEffects == "cov"){
    modMatrices$sigma_randomEffects <- matrixsetup_sigma(sigma_randomEffects,
                                                         expcov=list(expRanEffects),
                                                         nNode = nCov,
                                                         nGroup = nGroup,
                                                         labels = corvars,
                                                         equal = FALSE,
                                                         sampletable = sampleStats,
                                                         name = "sigma_randomEffects")
  } else if (randomEffects == "chol"){
    modMatrices$lowertri_randomEffects <- matrixsetup_lowertri(lowertri_randomEffects,
                                                               expcov=list(expRanEffects),
                                                               nNode = nCov,
                                                               nGroup = nGroup,
                                                               labels = corvars,
                                                               equal = FALSE,
                                                               sampletable = sampleStats,
                                                               name = "lowertri_randomEffects")
  } else if (randomEffects == "ggm"){
    modMatrices$omega_randomEffects <- matrixsetup_omega(omega_randomEffects,
                                                         expcov=list(expRanEffects),
                                                         nNode = nCov,
                                                         nGroup = nGroup,
                                                         labels = corvars,
                                                         equal = FALSE,
                                                         sampletable = sampleStats,
                                                         name = "omega_randomEffects")
    modMatrices$delta_randomEffects <- matrixsetup_delta(delta_randomEffects,
                                                         expcov=list(expRanEffects),
                                                         nNode = nCov,
                                                         nGroup = nGroup,
                                                         labels = corvars,
                                                         equal = FALSE,
                                                         sampletable = sampleStats,
                                                         name = "delta_randomEffects",
                                                         omegaStart =  modMatrices$omega_randomEffects$start)
  } else if (randomEffects == "prec"){
    modMatrices$kappa_randomEffects <- matrixsetup_kappa(kappa_randomEffects,
                                                         expcov=list(expRanEffects),
                                                         nNode = nCov,
                                                         nGroup = nGroup,
                                                         labels = corvars,
                                                         equal = FALSE,
                                                         sampletable = sampleStats,
                                                         name = "kappa_randomEffects")
  } else if (randomEffects == "cor"){
    modMatrices$rho_randomEffects <- matrixsetup_rho(rho_randomEffects,
                                                     expcov=list(expRanEffects),
                                                     nNode = nCov,
                                                     nGroup = nGroup,
                                                     labels = corvars,
                                                     equal = FALSE,
                                                     sampletable = sampleStats,
                                                     name = "rho_randomEffects")
    modMatrices$SD_randomEffects <- matrixsetup_SD(SD_randomEffects,
                                                   expcov=list(expRanEffects),
                                                   nNode = nCov,
                                                   nGroup = nGroup,
                                                   labels = corvars,
                                                   equal = FALSE,
                                                   sampletable = sampleStats,
                                                   name = "SD_randomEffects")
  }

  # Generate the full parameter table:
  pars <- do.call(generateAllParameterTables, modMatrices)

  # Store in model:
  model@parameters <- pars$partable
  model@matrices <- pars$mattable

  # Extra matrices (both LVM and meta-analytic):
  model@extramatrices <- list(
    # LVM matrices (nNode dimension):
    D = psychonetrics::duplicationMatrix(nNode),
    L = psychonetrics::eliminationMatrix(nNode),
    Lstar = psychonetrics::eliminationMatrix(nNode, diag=FALSE),
    Dstar = psychonetrics::duplicationMatrix(nNode,diag = FALSE),
    In = as(diag(nNode),"dMatrix"),
    A = psychonetrics::diagonalizationMatrix(nNode),
    C = as(lavaan::lav_matrix_commutation(nNode, nLatent),"dMatrix"),
    C_chol = as(lavaan::lav_matrix_commutation(nNode, nNode),"dMatrix"),

    # LVM matrices (nLatent dimension):
    Deta = psychonetrics::duplicationMatrix(nLatent),
    L_eta = psychonetrics::eliminationMatrix(nLatent),
    Dstar_eta = psychonetrics::duplicationMatrix(nLatent,diag = FALSE),
    Inlatent = as(diag(nLatent),"dMatrix"),
    Cbeta = as(lavaan::lav_matrix_commutation(nLatent, nLatent),"dMatrix"),
    Aeta = psychonetrics::diagonalizationMatrix(nLatent),

    # Random effects matrices (nCov dimension):
    D_c = psychonetrics::duplicationMatrix(nCov),
    L_c = psychonetrics::eliminationMatrix(nCov),
    Lstar_c = psychonetrics::eliminationMatrix(nCov, diag=FALSE),
    Dstar_c = psychonetrics::duplicationMatrix(nCov,diag = FALSE),
    In_c = as(diag(nCov),"dMatrix"),
    A_c = psychonetrics::diagonalizationMatrix(nCov),
    C_c = as(lavaan::lav_matrix_commutation(nCov,nCov),"dMatrix"),

    # V matrices:
    V = avgVmat,
    Vall = Vmats,
    Vmethod = Vmethod,
    Vestimation = Vestimation
  )

  # Form the model matrices
  model@modelmatrices <- formModelMatrices(model)

  ### Baseline model ###
  if (baseline_saturated){
    # Dummy sample stats (not corinput for baseline/saturated):
    sampleStats2 <- sampleStats
    sampleStats2@corinput <- FALSE

    # Form baseline model:
    model@baseline_saturated$baseline <- varcov(data,
                                                mu = rep(0,nCov),
                                                type = "chol",
                                                lowertri = "diag",
                                                vars = corvars,
                                                missing = "pairwise",
                                                estimator = estimator,
                                                baseline_saturated = FALSE,
                                                sampleStats=sampleStats2)

    model@baseline_saturated$baseline@sample@fullFIML <- FALSE

    ### Saturated model ###
    model@baseline_saturated$saturated <- varcov(data,
                                                 type = "chol",
                                                 lowertri = "full",
                                                 vars = corvars,
                                                 missing = "pairwise",
                                                 estimator = estimator,
                                                 baseline_saturated = FALSE,
                                                 sampleStats=sampleStats2)

    model@baseline_saturated$saturated@sample@fullFIML <- FALSE

    # if not FIML, Treat as computed:
    if (estimator != "FIML"){
      model@baseline_saturated$saturated@computed <- TRUE
      model@baseline_saturated$saturated@objective <- psychonetrics_fitfunction(parVector(model@baseline_saturated$saturated),model@baseline_saturated$saturated)
    }
  }

  # Identify model (reuse LVM identification):
  # model@identification <- identification
  if (identify){
    model <- identify(model)
    
  }

  if (missing(optimizer)){
    model <- setoptimizer(model, "default")
  } else {
    model <- setoptimizer(model, optimizer)
  }

  # Return model:
  return(model)
}
