% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ps_trunc.R
\name{ps_trunc}
\alias{ps_trunc}
\title{Truncate (Winsorize) Propensity Scores}
\usage{
ps_trunc(
  ps,
  method = c("ps", "pctl", "cr"),
  lower = NULL,
  upper = NULL,
  .exposure = NULL,
  .focal_level = NULL,
  .reference_level = NULL,
  ...,
  .treated = NULL,
  .untreated = NULL
)
}
\arguments{
\item{ps}{A numeric vector of propensity scores between 0 and 1 (binary
exposures), or a matrix/data.frame where each column contains propensity
scores for one level of a categorical exposure.}

\item{method}{One of \code{"ps"}, \code{"pctl"}, or \code{"cr"}:
\itemize{
\item \code{"ps"} (default): Truncate directly on propensity score values. Values
outside \verb{[lower, upper]} are set to the nearest bound. For categorical
exposures, applies symmetric truncation using \code{lower} as the threshold
(delta) and renormalizes rows to sum to 1.
\item \code{"pctl"}: Truncate at quantiles of the propensity score distribution.
The \code{lower} and \code{upper} arguments specify quantile probabilities. For
categorical exposures, quantiles are computed across all columns.
\item \code{"cr"}: Truncate to the common range of propensity scores across
exposure groups (binary exposures only). Bounds are
\verb{[min(ps[focal]), max(ps[reference])]}. Requires \code{.exposure}.
}}

\item{lower, upper}{Bounds for truncation. Interpretation depends on \code{method}:
\itemize{
\item \code{method = "ps"}: Propensity score values (defaults: 0.1 and 0.9). For
categorical exposures, \code{lower} is the truncation threshold delta
(default: 0.01); \code{upper} is ignored.
\item \code{method = "pctl"}: Quantile probabilities (defaults: 0.05 and 0.95;
categorical defaults: 0.01 and 0.99).
\item \code{method = "cr"}: Ignored; bounds are determined by the data.
}}

\item{.exposure}{An exposure vector. Required for method \code{"cr"} (binary
exposure vector) and for categorical exposures (factor or character vector)
with any method.}

\item{.focal_level}{The value of \code{.exposure} representing the focal
(treated) group. For binary exposures, defaults to the higher value.
Required for \code{wt_att()} and \code{wt_atu()} with categorical exposures.}

\item{.reference_level}{The value of \code{.exposure} representing the reference
(control) group. Automatically detected if not supplied.}

\item{...}{Additional arguments passed to methods.}

\item{.treated}{\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#deprecated}{\figure{lifecycle-deprecated.svg}{options: alt='[Deprecated]'}}}{\strong{[Deprecated]}} Use \code{.focal_level} instead.}

\item{.untreated}{\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#deprecated}{\figure{lifecycle-deprecated.svg}{options: alt='[Deprecated]'}}}{\strong{[Deprecated]}} Use \code{.reference_level} instead.}
}
\value{
A \code{ps_trunc} object (a numeric vector for binary exposures, or a
matrix for categorical exposures). Use \code{\link[=ps_trunc_meta]{ps_trunc_meta()}} to inspect
metadata including \code{method}, \code{lower_bound}, \code{upper_bound}, and
\code{truncated_idx} (positions of modified values).
}
\description{
\code{ps_trunc()} bounds extreme propensity scores to fixed limits, replacing
out-of-range values with the boundary value (a form of \emph{winsorizing}). The
result is a vector or matrix of the same length and dimensions as \code{ps}, with
no observations removed. This contrasts with \code{\link[=ps_trim]{ps_trim()}}, which sets extreme
values to \code{NA} (effectively removing those observations from analysis).
}
\details{
Unlike \code{\link[=ps_trim]{ps_trim()}}, truncation preserves all observations. No \code{NA} values
are introduced; out-of-range scores are replaced with the boundary value.

For \strong{binary exposures}, each propensity score \eqn{e_i} is bounded:
\itemize{
\item If \eqn{e_i < l}, set \eqn{e_i = l} (the lower bound).
\item If \eqn{e_i > u}, set \eqn{e_i = u} (the upper bound).
}

For \strong{categorical exposures}, values below the threshold are set to the
threshold and each row is renormalized to sum to 1.

\strong{Arithmetic behavior}: Arithmetic operations on \code{ps_trunc} objects return
plain numeric vectors. Once propensity scores are transformed (e.g., into
weights), the result is no longer a propensity score.

\strong{Combining behavior}: Combining \code{ps_trunc} objects with \code{c()} requires
matching truncation parameters. Mismatched parameters produce a warning and
return a plain numeric vector.
}
\examples{
set.seed(2)
n <- 200
x <- rnorm(n)
z <- rbinom(n, 1, plogis(1.2 * x))
fit <- glm(z ~ x, family = binomial)
ps <- predict(fit, type = "response")

# Truncate to [0.1, 0.9]
ps_t <- ps_trunc(ps, method = "ps", lower = 0.1, upper = 0.9)
ps_t

# Truncate at the 1st and 99th percentiles
ps_trunc(ps, method = "pctl", lower = 0.01, upper = 0.99)

# Use truncated scores to calculate weights
wt_ate(ps_t, .exposure = z)

# Inspect which observations were truncated
is_unit_truncated(ps_t)

}
\references{
Crump, R. K., Hotz, V. J., Imbens, G. W., & Mitnik, O. A. (2009). Dealing
with limited overlap in estimation of average treatment effects.
\emph{Biometrika}, 96(1), 187--199.

Walker, A. M., Patrick, A. R., Lauer, M. S., et al. (2013). A tool for
assessing the feasibility of comparative effectiveness research.
\emph{Comparative Effectiveness Research}, 3, 11--20.
}
\seealso{
\code{\link[=ps_trim]{ps_trim()}} for removing (rather than bounding) extreme values,
\code{\link[=ps_refit]{ps_refit()}} for refitting the propensity model after trimming,
\code{\link[=is_ps_truncated]{is_ps_truncated()}}, \code{\link[=is_unit_truncated]{is_unit_truncated()}}, \code{\link[=ps_trunc_meta]{ps_trunc_meta()}}
}
