% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fit_aci_response.R
\name{fit_aci_response}
\alias{fit_aci_response}
\title{Fitting ACi curves}
\usage{
fit_aci_response(
  data,
  varnames = list(A_net = "A_net", T_leaf = "T_leaf", C_i = "C_i", PPFD = "PPFD", g_mc =
    "g_mc"),
  P = 100,
  fitTPU = TRUE,
  alpha_g = 0,
  R_d_meas = NULL,
  useR_d = FALSE,
  useg_mc = FALSE,
  useg_mct = FALSE,
  usegamma_star = FALSE,
  useK_M = FALSE,
  useK_C_K_O = FALSE,
  alpha = 0.24,
  theta_J = 0.85,
  gamma_star25 = 42.75,
  Ea_gamma_star = 37830,
  K_M25 = 718.4,
  Ea_K_M = 65508.28,
  g_mc25 = 0.08701,
  Ea_g_mc = 0,
  K_C25 = NULL,
  Ea_K_C = NULL,
  K_O25 = NULL,
  Ea_K_O = NULL,
  Oconc = 21,
  gamma_star_set = NULL,
  K_M_set = NULL,
  ...
)
}
\arguments{
\item{data}{Dataframe for A-Ci curve fitting}

\item{varnames}{List of variable names. varnames = list(A_net = "A_net",
T_leaf = "T_leaf", C_i = "C_i", PPFD = "PPFD", g_mc = "g_mc"), where A_net
is net CO2 assimilation, T_leaf is leaf temperature in Kelvin, C_i is
intercellular CO2 concentration in umol/mol, PPFD is incident irradiance
in umol m-2 s-1 (note that it is ASSUMED to be absorbed irradiance, so be
sure to adjust according to light absorbance and PSI/PSII partitioning
accordingly OR interpret the resultant values of J and J_max with caution),
g_mc is mesophyll conductance to CO2 in mol m-2 s-1 Pa-1.}

\item{P}{Atmospheric pressure in kPa}

\item{fitTPU}{Should triose phosphate utilization (V_TPU) be fit?}

\item{alpha_g}{Fraction of respiratory glycolate carbon that is not returned
to the chloroplast (von Caemmerer, 2000). If ACi curves show high-CO2
decline, then this value should be > 0.}

\item{R_d_meas}{Measured value of respiratory CO2 efflux in umol m-2 s-1.
Input value should be positive to work as expected with the equations.}

\item{useR_d}{Use a measured value of R_d? Set to TRUE if using R_d_meas.}

\item{useg_mc}{Use mesophyll conductance? Set to TRUE if specifying g_mc
in varnames above.}

\item{useg_mct}{Use mesophyll conductance temperature response? Set to TRUE
if using a temperature response of mesophyll conductance.}

\item{usegamma_star}{Specify gamma_star value? If FALSE, uses a temperature
response function with Nicotiana tabacum defaults from Bernacchi et al.
2001.}

\item{useK_M}{Specify K_M? If FALSE, uses an Arrhenius temperature response
function with Nicotiana tabacum defaults from Bernacchi et al. 2001.}

\item{useK_C_K_O}{Use individual carboxylation/oxygenation constants for
rubisco? If TRUE, need to specify values at 25C and activation energy for
the Arrhenius temperature response function.}

\item{alpha}{Quantum yield of CO2 assimilation}

\item{theta_J}{Curvature of the photosynthetic light response curve}

\item{gamma_star25}{gamma_star at 25C in umol mol-1}

\item{Ea_gamma_star}{Activation energy of gamma_star in J mol-1}

\item{K_M25}{Michaelis-Menten constant for rubisco at 25C}

\item{Ea_K_M}{Activation energy for K_M in J mol-1}

\item{g_mc25}{Mesophyll conductance at 25C in mol m-2 s-1}

\item{Ea_g_mc}{Activation energy of g_mc in J mol-1}

\item{K_C25}{Michaelis-Menten constant for rubisco carboxylation at 25C}

\item{Ea_K_C}{Activation energy for K_C in J mol-1}

\item{K_O25}{Michaelis-Menten constant for rubisco oxygenation at 25C}

\item{Ea_K_O}{Activation energy for K_O in J mol-2}

\item{Oconc}{O2 concentration in \%. Used with P to calculate
intracellular O2 when using K_C_K_O}

\item{gamma_star_set}{Value of gamma_star to use (in ppm) if
usegamma_star = TRUE}

\item{K_M_set}{Value of K_M to use if useK_M = TRUE}

\item{...}{Other arguments to pass on}
}
\value{
fit_aci_response fits ACi curves using an approach similar to
Gu et al. 2010. Iterates all possible C_i transition points and checks for
inadmissible curve fits. If no curves are admissible (either due to poor data
or poor assumed parameters), the output will include a dataframe of NA values.
Default parameters are all from Bernacchi et al. 2001, 2002.
}
\description{
Fitting ACi curves
}
\examples{
\donttest{
# Read in your data
# Note that this data is coming from data supplied by the package
# hence the complicated argument in read.csv()
# This dataset is a CO2 by light response curve for a single sunflower
data <- read.csv(system.file("extdata", "A_Ci_Q_data_1.csv",
  package = "photosynthesis"
))

# Define a grouping factor based on light intensity to split the ACi
# curves
data$Q_2 <- as.factor((round(data$Qin, digits = 0)))

# Convert leaf temperature to K
data$T_leaf <- data$Tleaf + 273.15

# Fit ACi curve. Note that we are subsetting the dataframe
# here to fit for a single value of Q_2
fit <- fit_aci_response(data[data$Q_2 == 1500, ],
  varnames = list(
    A_net = "A",
    T_leaf = "T_leaf",
    C_i = "Ci",
    PPFD = "Qin"
  )
)

# View fitted parameters
fit[[1]]

# View graph
fit[[2]]

# View data with modelled parameters attached
fit[[3]]

# Fit many curves
fits <- fit_many(
  data = data,
  varnames = list(
    A_net = "A",
    T_leaf = "T_leaf",
    C_i = "Ci",
    PPFD = "Qin"
  ),
  funct = fit_aci_response,
  group = "Q_2"
)

# Print the parameters
# First set of double parentheses selects an individual group value
# Second set selects an element of the sublist
fits[[3]][[1]]

# Print the graph
fits[[3]][[2]]

# Compile graphs into a list for plotting
fits_graphs <- compile_data(fits,
  list_element = 2
)

# Compile parameters into dataframe for analysis
fits_pars <- compile_data(fits,
  output_type = "dataframe",
  list_element = 1
)
}
}
\references{
Bernacchi CJ, Singsaas EL, Pimentel C, Portis AR, Long SP. 2001. Improved
temperature response functions for models of rubisco-limited photosynthesis.
Plant Cell Environment 24:253-259.

Bernacchi CJ, Portis AR, Nakano H, von Caemmerer S, Long SP. 2002. Temperature
response of mesophyll conductance. Implications for the determination of rubisco
enzyme kinetics and for limitations to photosynthesis in vivo. Plant Physiology
130:1992-1998.

Gu L, Pallardy SG, Tu K, Law BE, Wullschleger SD. 2010. Reliable estimation
of biochemical parameters from C3 leaf photosynthesis-intercellular carbon
dioxide response curves. Plant Cell Environment 33:1852-1874.

von Caemmerer S. 2000. Biochemical models of leaf photosynthesis. CSIRO
Publishing, Collingwood.
}
