% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/simu_multiscale.R
\name{simu_multiscale}
\alias{simu_multiscale}
\title{Simulate Data Generating Processes (DGP) for Multiscale GWR}
\usage{
simu_multiscale(
  n = 1000,
  myseed = 1,
  type = "GG2024",
  constant = NULL,
  nuls = NULL,
  lambda = NULL,
  NN = 4,
  config_beta = "default",
  config_snr = 0.7,
  config_eps = "normal"
)
}
\arguments{
\item{n}{Integer. The number of observations to simulate. Default is 1000.}

\item{myseed}{Integer. Random seed for reproducibility. Default is 1.}

\item{type}{Character. The type of DGP to use. Options are \code{'FT2017'}, \code{'Gao2021'}, or \code{'GG2024'} (default).}

\item{constant}{Vector. Indices or names of explanatory variables that should have constant (spatially stationary) coefficients. Default is \code{NULL} (all coefficients vary).}

\item{nuls}{Vector. Indices of explanatory variables that should have a null effect (coefficient = 0). Default is \code{NULL}.}

\item{lambda}{Numeric. The spatial autoregressive parameter (rho) if a SAR process is desired. Default is \code{NULL} (no spatial autocorrelation).}

\item{NN}{Integer. The number of nearest neighbors used to construct the spatial weight matrix \code{W} if \code{lambda} is provided. Default is 4.}

\item{config_beta}{Character. Configuration of the spatial pattern for Beta coefficients (e.g., \code{'default'}).}

\item{config_snr}{Numeric. The desired Signal-to-Noise Ratio (SNR). Default is 0.7.}

\item{config_eps}{Character. The distribution of the error term. Options are \code{'normal'} (default), \code{'unif'}, or \code{'Chi2'}.}
}
\value{
A named list containing:
\describe{
  \item{mydata}{A data frame with the simulated response variable \code{y} and explanatory variables.}
  \item{coords}{A matrix of spatial coordinates.}
  \item{Betav}{A matrix of the true spatially varying coefficients.}
  \item{W}{The spatial weight matrix (if \code{lambda} is not NULL).}
}
}
\description{
The \code{simu_multiscale} function generates synthetic datasets with spatially varying coefficients
based on Data Generating Processes (DGP) proposed in the literature. It supports
formulations from Fotheringham et al. (2017), Gao et al. (2021), and Geniaux (2024).
It also allows for the introduction of spatial autocorrelation (SAR) in the response variable.
}
\examples{
\donttest{
 library(mgwrsar)
 library(ggplot2)
 library(gridExtra)
 library(grid)

 # Simulate data using Geniaux (2024) DGP
 simu <- simu_multiscale(n = 1000, type = 'GG2024', config_snr = 0.7)
 mydata <- simu$mydata
 coords <- simu$coords

 # Visualizing the spatial patterns of the coefficients
 p1 <- ggplot(mydata, aes(x, y, col = Beta1)) + geom_point() + scale_color_viridis_c()
 p2 <- ggplot(mydata, aes(x, y, col = Beta2)) + geom_point() + scale_color_viridis_c()
 p3 <- ggplot(mydata, aes(x, y, col = Beta3)) + geom_point() + scale_color_viridis_c()
 p4 <- ggplot(mydata, aes(x, y, col = Beta4)) + geom_point() + scale_color_viridis_c()

 grid.arrange(p1, p2, p3, p4, nrow = 2, ncol = 2,
              top = textGrob("DGP Geniaux (2024)", gp = gpar(fontsize = 20, font = 3)))
}
}
\references{
Fotheringham, A. S., Yang, W., & Kang, W. (2017). Multiscale geographically weighted regression (MGWR). \emph{Annals of the American Association of Geographers}, 107(6), 1247-1265.

Gao, S., Mei, C. L., Xu, Q. X., & Wang, N. (2021). Non-iterative multiscale estimation for spatial autoregressive geographically weighted regression models. \emph{Entropy}, 25(2), 320.

Geniaux, G. (2026). Top-down scale approaches for multiscale gwr with locally adaptive bandwidths. \emph{Journal of Geographical Systems}: 1–50.
}
