#' Summary Method for jmSurface Objects
#'
#' @description
#' Provides a comprehensive summary of the fitted joint model including
#' longitudinal submodel parameters, transition-specific surface EDF and
#' deviance explained, and model configuration.
#'
#' @param object A \code{"jmSurface"} object.
#' @param ... Ignored.
#'
#' @return Invisibly returns a list of summary components.
#' @export
summary.jmSurface <- function(object, ...) {

  cat("\n")
  cat("==========================================================\n")
  cat("  Joint Longitudinal-Survival Model with Association Surfaces\n")
  cat("  jmSurface v0.1.0\n")
  cat("==========================================================\n\n")

  cat("Call:\n")
  print(object$call)
  cat("\n")

  ## Data summary
  cat("--- Data Summary ---\n")
  cat("  Patients:      ", object$n_patients, "\n")
  cat("  Biomarkers:    ", paste(object$biomarkers, collapse = ", "), "\n")
  cat("  Transitions:   ", length(object$transitions), "\n")
  cat("  Covariates:    ", paste(object$covariates, collapse = ", "), "\n")
  cat("  Basis (k):     ", paste(object$k_marginal, collapse = "x"),
      " (tensor), ", object$k_additive, " (additive)\n")
  cat("  Smoothing:     ", object$method, "\n\n")

  ## Longitudinal submodels
  cat("--- Stage 1: Longitudinal Submodels ---\n")
  for (mk in names(object$lme_fits)) {
    if (is.null(object$lme_fits[[mk]])) {
      cat("  ", mk, ": FAILED\n")
      next
    }
    fe <- nlme::fixef(object$lme_fits[[mk]])
    vc <- tryCatch(
      as.numeric(nlme::VarCorr(object$lme_fits[[mk]])[, "StdDev"]),
      error = function(e) rep(NA, 3)
    )
    cat(sprintf("  %-8s  Intercept = %7.2f  Slope = %7.3f  RE SD: %s\n",
                mk, fe[1], fe[2], paste(round(vc, 2), collapse = ", ")))
  }
  cat("\n")

  ## Transition-specific surfaces
  cat("--- Stage 2: Transition-Specific Association Surfaces ---\n")
  cat(sprintf("  %-25s %6s %7s %7s %12s %10s\n",
              "Transition", "N", "Events", "EDF", "Dev.Expl", "Complexity"))
  cat("  ", paste(rep("-", 72), collapse = ""), "\n")

  edf_diag <- edf_diagnostics(object)
  for (i in seq_len(nrow(edf_diag))) {
    row <- edf_diag[i, ]
    cat(sprintf("  %-25s %6d %7d %7.1f %10.1f%% %10s\n",
                row$transition, row$n_obs, row$n_events,
                row$edf, row$deviance_explained * 100,
                row$complexity))
  }
  cat("\n")

  ## EDF interpretation
  cat("--- EDF Interpretation ---\n")
  cat("  EDF ~ 1:      Linear relationship (parametric JM suffices)\n")
  cat("  1 < EDF <= 3: Moderate nonlinearity\n")
  cat("  EDF > 3:      Substantial nonlinearity/interaction\n\n")

  invisible(list(
    edf_diagnostics = edf_diag,
    lme_summary = lapply(object$lme_fits, function(x)
      if (!is.null(x)) nlme::fixef(x) else NULL)
  ))
}


#' Print Method for jmSurface Objects
#'
#' @description Brief overview of a fitted jmSurface model.
#'
#' @param x A \code{"jmSurface"} object.
#' @param ... Ignored.
#' @return The input object \code{x}, returned invisibly. Called for its
#'   side effect of printing a brief model overview to the console.
#' @export
print.jmSurface <- function(x, ...) {

  cat("jmSurface: Joint Model with Semi-Parametric Association Surfaces\n")
  cat("  Patients: ", x$n_patients, " | Biomarkers: ",
      length(x$biomarkers), " | Transitions: ",
      length(x$transitions), "\n")
  cat("  EDF range: ", round(min(x$edf, na.rm = TRUE), 1), " - ",
      round(max(x$edf, na.rm = TRUE), 1), "\n")
  cat("  Transitions: ", paste(x$transitions, collapse = "; "), "\n")

  invisible(x)
}
