% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/hypothesize.R
\name{invert_test}
\alias{invert_test}
\title{Invert a Test into a Confidence Set (Test-Confidence Duality)}
\usage{
invert_test(test_fn, grid, alpha = 0.05)
}
\arguments{
\item{test_fn}{A function that takes a single numeric argument (the
hypothesized null value) and returns a \code{hypothesis_test} object.}

\item{grid}{Numeric vector of candidate null values to test.}

\item{alpha}{Numeric. Significance level (default 0.05). The confidence
level is \eqn{1 - \alpha}.}
}
\value{
An S3 object of class \code{confidence_set} containing:
\describe{
\item{set}{Numeric vector of non-rejected null values}
\item{alpha}{The significance level used}
\item{level}{The confidence level (\eqn{1 - \alpha})}
\item{test_fn}{The input test function}
\item{grid}{The input grid}
}
}
\description{
Takes a test constructor function and returns the confidence set: the set
of null values that are not rejected at level \eqn{\alpha}.
}
\details{
Hypothesis tests and confidence sets are dual: a \eqn{(1-\alpha)}
confidence set contains exactly those parameter values \eqn{\theta_0}
for which the test of \eqn{H_0: \theta = \theta_0} would not reject at
level \eqn{\alpha}. This function makes that duality operational.

\code{invert_test} is the most general confidence set constructor in the
package. Any test — including user-defined tests — can be inverted. The
specialized \code{\link[=confint]{confint()}} methods for \code{wald_test} and \code{z_test} give exact
analytical intervals; \code{invert_test} gives numerical intervals for
arbitrary tests at the cost of a grid search.
}
\section{Higher-Order Function (SICP Principle)}{

This function takes a \strong{function} as input (\code{test_fn}) and returns a
structured result. It demonstrates the power of the \code{hypothesis_test}
abstraction: because all tests implement the same interface (\code{pval()}),
\code{invert_test} can work with any test without knowing its internals.
}

\examples{
# Invert a Wald test to get a confidence interval
cs <- invert_test(
  test_fn = function(theta) wald_test(estimate = 2.5, se = 0.8, null_value = theta),
  grid = seq(0, 5, by = 0.01)
)
cs
lower(cs)
upper(cs)

# Compare with the analytical confint (should agree up to grid resolution)
confint(wald_test(estimate = 2.5, se = 0.8))

# Invert ANY user-defined test — no special support needed
my_test <- function(theta) {
  stat <- (5.0 - theta)^2 / 2
  hypothesis_test(stat = stat,
    p.value = pchisq(stat, df = 1, lower.tail = FALSE), dof = 1)
}
invert_test(my_test, grid = seq(0, 10, by = 0.01))

}
\seealso{
\code{\link[=confint.wald_test]{confint.wald_test()}}, \code{\link[=confint.z_test]{confint.z_test()}} for analytical CIs
}
