/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Gmodule.h" 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5FLprivate.h" 
#include "H5FOprivate.h" 
#include "H5Gpkg.h"      
#include "H5Iprivate.h"  
#include "H5Lprivate.h"  
#include "H5MMprivate.h" 
#include "H5SLprivate.h" 

typedef struct {
    H5G_loc_t *loc; 
} H5G_trav_ins_t;

typedef struct {
    hid_t              gid;      
    H5O_loc_t         *link_loc; 
    H5G_link_iterate_t lnk_op;   
    void              *op_data;  
} H5G_iter_appcall_ud_t;

typedef struct {
    hid_t           gid;           
    H5G_loc_t      *curr_loc;      
    H5_index_t      idx_type;      
    H5_iter_order_t order;         
    H5SL_t         *visited;       
    char           *path;          
    size_t          curr_path_len; 
    size_t          path_buf_size; 
    H5L_iterate2_t  op;            
    void           *op_data;       
} H5G_iter_visit_ud_t;

static herr_t H5G__open_oid(H5G_t *grp);
static herr_t H5G__visit_cb(const H5O_link_t *lnk, void *_udata);
static herr_t H5G__close_cb(void *grp_vol_obj, void **request);

bool H5_PKG_INIT_VAR = false;

H5FL_DEFINE(H5G_t);
H5FL_DEFINE(H5G_shared_t);

H5FL_DEFINE(H5_obj_t);

static const H5I_class_t H5I_GROUP_CLS[1] = {{
    H5I_GROUP,                
    0,                        
    0,                        
    (H5I_free_t)H5G__close_cb 
}};

static bool H5G_top_package_initialize_s = false;

herr_t
H5G_init(void)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)
    

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5G__init_package(void)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    if (H5I_register_type(H5I_GROUP_CLS) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTINIT, FAIL, "unable to initialize interface");

    
    H5G_top_package_initialize_s = true;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

int
H5G_top_term_package(void)
{
    int n = 0;

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    if (H5G_top_package_initialize_s) {
        if (H5I_nmembers(H5I_GROUP) > 0) {
            (void)H5I_clear_type(H5I_GROUP, false, false);
            n++;
        }

        
        if (0 == n)
            H5G_top_package_initialize_s = false;
    } 

    FUNC_LEAVE_NOAPI(n)
} 

int
H5G_term_package(void)
{
    int n = 0;

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    if (H5_PKG_INIT_VAR) {
        
        assert(0 == H5I_nmembers(H5I_GROUP));
        assert(false == H5G_top_package_initialize_s);

        
        n += (H5I_dec_type_ref(H5I_GROUP) > 0);

        
        if (0 == n)
            H5_PKG_INIT_VAR = false;
    } 

    FUNC_LEAVE_NOAPI(n)
} 

static herr_t
H5G__close_cb(void *_grp_vol_obj, void **request)
{
    H5VL_object_t *grp_vol_obj = (H5VL_object_t *)_grp_vol_obj;
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(grp_vol_obj);

    
    if (H5VL_group_close(grp_vol_obj, H5P_DATASET_XFER_DEFAULT, request) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CLOSEERROR, FAIL, "unable to close group");

    
    if (H5VL_free_object(grp_vol_obj) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTDEC, FAIL, "unable to free VOL object");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

H5G_t *
H5G__create_named(const H5G_loc_t *loc, const char *name, hid_t lcpl_id, hid_t gcpl_id)
{
    H5O_obj_create_t ocrt_info;        
    H5G_obj_create_t gcrt_info;        
    H5G_t           *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    
    assert(loc);
    assert(name && *name);
    assert(lcpl_id != H5P_DEFAULT);
    assert(gcpl_id != H5P_DEFAULT);

    
    gcrt_info.gcpl_id    = gcpl_id;
    gcrt_info.cache_type = H5G_NOTHING_CACHED;
    memset(&gcrt_info.cache, 0, sizeof(gcrt_info.cache));

    
    ocrt_info.obj_type = H5O_TYPE_GROUP;
    ocrt_info.crt_info = &gcrt_info;
    ocrt_info.new_obj  = NULL;

    
    if (H5L_link_object(loc, name, &ocrt_info, lcpl_id) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTINIT, NULL, "unable to create and link to group");
    assert(ocrt_info.new_obj);

    
    ret_value = (H5G_t *)ocrt_info.new_obj;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

H5G_t *
H5G__create(H5F_t *file, H5G_obj_create_t *gcrt_info)
{
    H5G_t   *grp       = NULL; 
    unsigned oloc_init = 0;    
    H5G_t   *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    
    assert(file);
    assert(gcrt_info->gcpl_id != H5P_DEFAULT);

    
    if (NULL == (grp = H5FL_CALLOC(H5G_t)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed");
    if (NULL == (grp->shared = H5FL_CALLOC(H5G_shared_t)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed");

    
    if (H5G__obj_create(file, gcrt_info, &(grp->oloc) ) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTINIT, NULL, "unable to create group object header");
    oloc_init = 1; 

    
    if (H5FO_top_incr(grp->oloc.file, grp->oloc.addr) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTINC, NULL, "can't incr object ref. count");
    if (H5FO_insert(grp->oloc.file, grp->oloc.addr, grp->shared, true) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTINSERT, NULL, "can't insert group into list of open objects");

    
    grp->shared->fo_count = 1;

    
    ret_value = grp;

done:
    if (ret_value == NULL) {
        
        if (oloc_init) {
            if (H5O_dec_rc_by_loc(&(grp->oloc)) < 0)
                HDONE_ERROR(H5E_SYM, H5E_CANTDEC, NULL,
                            "unable to decrement refcount on newly created object");
            if (H5O_close(&(grp->oloc), NULL) < 0)
                HDONE_ERROR(H5E_SYM, H5E_CLOSEERROR, NULL, "unable to release object header");
            if (H5O_delete(file, grp->oloc.addr) < 0)
                HDONE_ERROR(H5E_SYM, H5E_CANTDELETE, NULL, "unable to delete object header");
        } 
        if (grp != NULL) {
            if (grp->shared != NULL)
                grp->shared = H5FL_FREE(H5G_shared_t, grp->shared);
            grp = H5FL_FREE(H5G_t, grp);
        } 
    }     

    FUNC_LEAVE_NOAPI(ret_value)
} 

H5G_t *
H5G__open_name(const H5G_loc_t *loc, const char *name)
{
    H5G_t     *grp = NULL;        
    H5G_loc_t  grp_loc;           
    H5G_name_t grp_path;          
    H5O_loc_t  grp_oloc;          
    bool       loc_found = false; 
    H5O_type_t obj_type;          
    H5G_t     *ret_value = NULL;  

    FUNC_ENTER_PACKAGE

    
    assert(loc);
    assert(name);

    
    grp_loc.oloc = &grp_oloc;
    grp_loc.path = &grp_path;
    H5G_loc_reset(&grp_loc);

    
    if (H5G_loc_find(loc, name, &grp_loc ) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_NOTFOUND, NULL, "group not found");
    loc_found = true;

    
    if (H5O_obj_type(&grp_oloc, &obj_type) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTGET, NULL, "can't get object type");
    if (obj_type != H5O_TYPE_GROUP)
        HGOTO_ERROR(H5E_SYM, H5E_BADTYPE, NULL, "not a group");

    
    if (NULL == (grp = H5G_open(&grp_loc)))
        HGOTO_ERROR(H5E_SYM, H5E_CANTOPENOBJ, NULL, "unable to open group");

    
    ret_value = grp;

done:
    if (!ret_value)
        if (loc_found && H5G_loc_free(&grp_loc) < 0)
            HDONE_ERROR(H5E_SYM, H5E_CANTRELEASE, NULL, "can't free location");

    FUNC_LEAVE_NOAPI(ret_value)
} 

H5G_t *
H5G_open(const H5G_loc_t *loc)
{
    H5G_t        *grp = NULL;       
    H5G_shared_t *shared_fo;        
    H5G_t        *ret_value = NULL; 

    FUNC_ENTER_NOAPI(NULL)

    
    assert(loc);

    
    if (NULL == (grp = H5FL_CALLOC(H5G_t)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "can't allocate space for group");

    
    if (H5O_loc_copy_shallow(&(grp->oloc), loc->oloc) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTCOPY, NULL, "can't copy object location");
    if (H5G_name_copy(&(grp->path), loc->path, H5_COPY_SHALLOW) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTCOPY, NULL, "can't copy path");

    
    if ((shared_fo = (H5G_shared_t *)H5FO_opened(grp->oloc.file, grp->oloc.addr)) == NULL) {
        
        if (H5G__open_oid(grp) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_NOTFOUND, NULL, "not found");

        
        if (H5FO_insert(grp->oloc.file, grp->oloc.addr, grp->shared, false) < 0) {
            grp->shared = H5FL_FREE(H5G_shared_t, grp->shared);
            HGOTO_ERROR(H5E_SYM, H5E_CANTINSERT, NULL, "can't insert group into list of open objects");
        } 

        
        if (H5FO_top_incr(grp->oloc.file, grp->oloc.addr) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_CANTINC, NULL, "can't increment object count");

        
        grp->shared->fo_count = 1;
    } 
    else {
        
        grp->shared = shared_fo;

        
        shared_fo->fo_count++;

        
        if (H5FO_top_count(grp->oloc.file, grp->oloc.addr) == 0) {
            
            if (H5O_open(&(grp->oloc)) < 0)
                HGOTO_ERROR(H5E_SYM, H5E_CANTOPENOBJ, NULL, "unable to open object header");
        } 

        
        if (H5FO_top_incr(grp->oloc.file, grp->oloc.addr) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_CANTINC, NULL, "can't increment object count");
    } 

    
    ret_value = grp;

done:
    if (!ret_value && grp) {
        H5O_loc_free(&(grp->oloc));
        H5G_name_free(&(grp->path));
        grp = H5FL_FREE(H5G_t, grp);
    } 

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5G__open_oid(H5G_t *grp)
{
    bool   obj_opened = false;
    htri_t msg_exists;
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(grp);

    
    if (NULL == (grp->shared = H5FL_CALLOC(H5G_shared_t)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "memory allocation failed");

    
    if (H5O_open(&(grp->oloc)) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTOPENOBJ, FAIL, "unable to open group");
    obj_opened = true;

    
    if ((msg_exists = H5O_msg_exists(&(grp->oloc), H5O_STAB_ID)) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "can't check if symbol table message exists");
    if (!msg_exists) {
        if ((msg_exists = H5O_msg_exists(&(grp->oloc), H5O_LINFO_ID)) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "can't check if link info message exists");
        if (!msg_exists)
            HGOTO_ERROR(H5E_SYM, H5E_CANTOPENOBJ, FAIL, "not a group");
    }

done:
    if (ret_value < 0) {
        if (obj_opened)
            H5O_close(&(grp->oloc), NULL);
        if (grp->shared)
            grp->shared = H5FL_FREE(H5G_shared_t, grp->shared);
    } 

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5G_close(H5G_t *grp)
{
    bool   corked;                
    bool   file_closed = true;    
    herr_t ret_value   = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(grp && grp->shared);
    assert(grp->shared->fo_count > 0);

    --grp->shared->fo_count;

    if (0 == grp->shared->fo_count) {
        assert(grp != H5G_rootof(H5G_fileof(grp)));

        
        if (H5AC_cork(grp->oloc.file, grp->oloc.addr, H5AC__GET_CORKED, &corked) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "unable to retrieve an object's cork status");
        if (corked)
            if (H5AC_cork(grp->oloc.file, grp->oloc.addr, H5AC__UNCORK, NULL) < 0)
                HGOTO_ERROR(H5E_SYM, H5E_CANTUNCORK, FAIL, "unable to uncork an object");

        
        if (H5FO_top_decr(grp->oloc.file, grp->oloc.addr) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_CANTRELEASE, FAIL, "can't decrement count for object");
        if (H5FO_delete(grp->oloc.file, grp->oloc.addr) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_CANTRELEASE, FAIL, "can't remove group from list of open objects");
        if (H5O_close(&(grp->oloc), &file_closed) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_CANTINIT, FAIL, "unable to close");

        
        if (!file_closed && H5F_SHARED(grp->oloc.file) && H5F_EVICT_ON_CLOSE(grp->oloc.file)) {
            if (H5AC_flush_tagged_metadata(grp->oloc.file, grp->oloc.addr) < 0)
                HGOTO_ERROR(H5E_CACHE, H5E_CANTFLUSH, FAIL, "unable to flush tagged metadata");
            if (H5AC_evict_tagged_metadata(grp->oloc.file, grp->oloc.addr, false) < 0)
                HGOTO_ERROR(H5E_CACHE, H5E_CANTFLUSH, FAIL, "unable to evict tagged metadata");
        } 

        
        grp->shared = H5FL_FREE(H5G_shared_t, grp->shared);
    }
    else {
        
        if (H5FO_top_decr(grp->oloc.file, grp->oloc.addr) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_CANTRELEASE, FAIL, "can't decrement count for object");

        
        if (H5FO_top_count(grp->oloc.file, grp->oloc.addr) == 0) {
            if (H5O_close(&(grp->oloc), NULL) < 0)
                HGOTO_ERROR(H5E_SYM, H5E_CANTINIT, FAIL, "unable to close");
        } 
        else
            
            if (H5O_loc_free(&(grp->oloc)) < 0)
                HGOTO_ERROR(H5E_SYM, H5E_CANTRELEASE, FAIL, "problem attempting to free location");

        
        if (grp->shared->mounted && grp->shared->fo_count == 1) {
            
            if (H5F_try_close(grp->oloc.file, NULL) < 0)
                HGOTO_ERROR(H5E_FILE, H5E_CANTCLOSEFILE, FAIL, "problem attempting file close");
        } 
    }     

    if (H5G_name_free(&(grp->path)) < 0) {
        grp = H5FL_FREE(H5G_t, grp);
        HGOTO_ERROR(H5E_SYM, H5E_CANTINIT, FAIL, "can't free group entry name");
    } 

    grp = H5FL_FREE(H5G_t, grp);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

H5O_loc_t *
H5G_oloc(H5G_t *grp)
{
    
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    FUNC_LEAVE_NOAPI(grp ? &(grp->oloc) : NULL)
} 

H5G_name_t *
H5G_nameof(H5G_t *grp)
{
    
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    FUNC_LEAVE_NOAPI(grp ? &(grp->path) : NULL)
} 

H5F_t *
H5G_fileof(H5G_t *grp)
{
    
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    assert(grp);

    FUNC_LEAVE_NOAPI(grp->oloc.file)
} 

herr_t
H5G_get_shared_count(H5G_t *grp)
{
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    assert(grp && grp->shared);

    FUNC_LEAVE_NOAPI(grp->shared->fo_count)
} 

herr_t
H5G_mount(H5G_t *grp)
{
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    assert(grp && grp->shared);
    assert(grp->shared->mounted == false);

    
    grp->shared->mounted = true;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

bool
H5G_mounted(H5G_t *grp)
{
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    assert(grp && grp->shared);

    FUNC_LEAVE_NOAPI(grp->shared->mounted)
} 

herr_t
H5G_unmount(H5G_t *grp)
{
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    assert(grp && grp->shared);
    assert(grp->shared->mounted == true);

    
    grp->shared->mounted = false;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5G__iterate_cb(const H5O_link_t *lnk, void *_udata)
{
    H5G_iter_appcall_ud_t *udata     = (H5G_iter_appcall_ud_t *)_udata; 
    herr_t                 ret_value = H5_ITER_ERROR;                   

    FUNC_ENTER_PACKAGE

    
    assert(lnk);
    assert(udata);

    switch (udata->lnk_op.op_type) {
#ifndef H5_NO_DEPRECATED_SYMBOLS
        case H5G_LINK_OP_OLD:
            
            H5_BEFORE_USER_CB(H5_ITER_ERROR)
                {
                    
                    ret_value = (udata->lnk_op.op_func.op_old)(udata->gid, lnk->name, udata->op_data);
                }
            H5_AFTER_USER_CB(H5_ITER_ERROR)
            break;
#endif 

        case H5G_LINK_OP_NEW: {
            H5L_info2_t info; 

            
            if (H5G_link_to_info(udata->link_loc, lnk, &info) < 0)
                HGOTO_ERROR(H5E_SYM, H5E_CANTGET, H5_ITER_ERROR, "unable to get info for link");

            
            H5_BEFORE_USER_CB(H5_ITER_ERROR)
                {
                    
                    ret_value = (udata->lnk_op.op_func.op_new)(udata->gid, lnk->name, &info, udata->op_data);
                }
            H5_AFTER_USER_CB(H5_ITER_ERROR)
        } break;

        default:
            assert(0 && "Unknown link op type?!?");
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5G_iterate(H5G_loc_t *loc, const char *group_name, H5_index_t idx_type, H5_iter_order_t order, hsize_t skip,
            hsize_t *last_lnk, const H5G_link_iterate_t *lnk_op, void *op_data)
{
    hid_t                 gid = H5I_INVALID_HID; 
    H5G_t                *grp = NULL;            
    H5G_iter_appcall_ud_t udata;                 
    herr_t                ret_value = FAIL;      

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(loc);
    assert(group_name);
    assert(last_lnk);
    assert(lnk_op && lnk_op->op_func.op_new);

    
    if (NULL == (grp = H5G__open_name(loc, group_name)))
        HGOTO_ERROR(H5E_SYM, H5E_CANTOPENOBJ, FAIL, "unable to open group");
    if ((gid = H5VL_wrap_register(H5I_GROUP, grp, true)) < 0)
        HGOTO_ERROR(H5E_ID, H5E_CANTREGISTER, FAIL, "unable to register group");

    
    udata.gid      = gid;
    udata.link_loc = &grp->oloc;
    udata.lnk_op   = *lnk_op;
    udata.op_data  = op_data;

    
    if ((ret_value =
             H5G__obj_iterate(&(grp->oloc), idx_type, order, skip, last_lnk, H5G__iterate_cb, &udata)) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_BADITER, FAIL, "error iterating over links");

done:
    
    if (gid != H5I_INVALID_HID) {
        if (H5I_dec_app_ref(gid) < 0)
            HDONE_ERROR(H5E_SYM, H5E_CANTRELEASE, FAIL, "unable to close group");
    }
    else if (grp && H5G_close(grp) < 0)
        HDONE_ERROR(H5E_SYM, H5E_CLOSEERROR, FAIL, "unable to release group");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5G__free_visit_visited(void *item, void H5_ATTR_UNUSED *key, void H5_ATTR_UNUSED *operator_data )
{
    FUNC_ENTER_PACKAGE_NOERR

    item = H5FL_FREE(H5_obj_t, item);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5G__visit_cb(const H5O_link_t *lnk, void *_udata)
{
    H5G_iter_visit_ud_t *udata = (H5G_iter_visit_ud_t *)_udata; 
    H5L_info2_t          info;                                  
    H5G_loc_t            obj_loc;                               
    H5G_name_t           obj_path;                              
    H5O_loc_t            obj_oloc;                              
    bool                 obj_found = false;                     
    size_t old_path_len = udata->curr_path_len; 
    size_t link_name_len;                       
    size_t len_needed;                          
    herr_t ret_value = H5_ITER_CONT;            

    FUNC_ENTER_PACKAGE

    
    assert(lnk);
    assert(udata);

    
    
    link_name_len = strlen(lnk->name);
    len_needed    = udata->curr_path_len + link_name_len + 2;
    if (len_needed > udata->path_buf_size) {
        void *new_path; 

        
        if (NULL == (new_path = H5MM_realloc(udata->path, len_needed)))
            HGOTO_ERROR(H5E_SYM, H5E_NOSPACE, H5_ITER_ERROR, "can't allocate path string");
        udata->path          = (char *)new_path;
        udata->path_buf_size = len_needed;
    } 

    
    assert(udata->path[old_path_len] == '\0');
    strncpy(&(udata->path[old_path_len]), lnk->name, link_name_len + 1);
    udata->curr_path_len += link_name_len;

    
    if (H5G_link_to_info(udata->curr_loc->oloc, lnk, &info) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTGET, H5_ITER_ERROR, "unable to get info for link");

    
    H5_BEFORE_USER_CB(H5_ITER_ERROR)
        {
            
            ret_value = (udata->op)(udata->gid, udata->path, &info, udata->op_data);
        }
    H5_AFTER_USER_CB(H5_ITER_ERROR)

    
    if (ret_value == H5_ITER_CONT && lnk->type == H5L_TYPE_HARD) {
        H5_obj_t obj_pos; 

        
        obj_loc.oloc = &obj_oloc;
        obj_loc.path = &obj_path;
        H5G_loc_reset(&obj_loc);

        
        
        if (H5G_loc_find(udata->curr_loc, lnk->name, &obj_loc ) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_NOTFOUND, H5_ITER_ERROR, "object not found");
        obj_found = true;

        
        H5F_GET_FILENO(obj_oloc.file, obj_pos.fileno);
        obj_pos.addr = obj_oloc.addr;

        
        if (NULL == H5SL_search(udata->visited, &obj_pos)) {
            H5O_type_t otype; 

            
            if (H5O_get_rc_and_type(&obj_oloc, NULL, &otype) < 0)
                HGOTO_ERROR(H5E_SYM, H5E_CANTGET, H5_ITER_ERROR, "unable to get object info");

            
            {
                H5_obj_t *new_node; 

                
                if ((new_node = H5FL_MALLOC(H5_obj_t)) == NULL)
                    HGOTO_ERROR(H5E_SYM, H5E_NOSPACE, H5_ITER_ERROR, "can't allocate object node");

                
                *new_node = obj_pos;

                
                if (H5SL_insert(udata->visited, new_node, new_node) < 0)
                    HGOTO_ERROR(H5E_SYM, H5E_CANTINSERT, H5_ITER_ERROR,
                                "can't insert object node into visited list");
            }

            
            if (otype == H5O_TYPE_GROUP) {
                H5G_loc_t  *old_loc  = udata->curr_loc; 
                H5_index_t  idx_type = udata->idx_type; 
                H5O_linfo_t linfo;                      
                htri_t      linfo_exists;               

                
                assert(udata->path[udata->curr_path_len] == '\0');
                strncpy(&(udata->path[udata->curr_path_len]), "/", (size_t)2);
                udata->curr_path_len++;

                
                if ((linfo_exists = H5G__obj_get_linfo(&obj_oloc, &linfo)) < 0)
                    HGOTO_ERROR(H5E_SYM, H5E_CANTGET, H5_ITER_ERROR, "can't check for link info message");
                if (linfo_exists) {
                    
                    if (idx_type == H5_INDEX_CRT_ORDER) {
                        
                        if (!linfo.track_corder)
                            
                            idx_type = H5_INDEX_NAME;
                    } 
                    else
                        assert(idx_type == H5_INDEX_NAME);
                } 
                else {
                    
                    if (idx_type != H5_INDEX_NAME)
                        
                        idx_type = H5_INDEX_NAME;
                } 

                
                udata->curr_loc = &obj_loc;

                
                ret_value = H5G__obj_iterate(&obj_oloc, idx_type, udata->order, (hsize_t)0, NULL,
                                             H5G__visit_cb, udata);

                
                udata->curr_loc = old_loc;
            } 
        }     
    }         

done:
    
    udata->path[old_path_len] = '\0';
    udata->curr_path_len      = old_path_len;

    
    if (obj_found && H5G_loc_free(&obj_loc) < 0)
        HDONE_ERROR(H5E_SYM, H5E_CANTRELEASE, H5_ITER_ERROR, "can't free location");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5G_visit(H5G_loc_t *loc, const char *group_name, H5_index_t idx_type, H5_iter_order_t order,
          H5L_iterate2_t op, void *op_data)
{
    H5G_iter_visit_ud_t udata;                 
    H5O_linfo_t         linfo;                 
    htri_t              linfo_exists;          
    hid_t               gid = H5I_INVALID_HID; 
    H5G_t              *grp = NULL;            
    H5G_loc_t           start_loc;             
    herr_t              ret_value = FAIL;      

    
    memset(&udata, 0, sizeof(udata));

    FUNC_ENTER_NOAPI(FAIL)

    
    if (!loc)
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "loc parameter cannot be NULL");

    
    if (NULL == (grp = H5G__open_name(loc, group_name)))
        HGOTO_ERROR(H5E_SYM, H5E_CANTOPENOBJ, FAIL, "unable to open group");

    
    if ((gid = H5VL_wrap_register(H5I_GROUP, grp, true)) < 0)
        HGOTO_ERROR(H5E_ID, H5E_CANTREGISTER, FAIL, "unable to register group");

    
    if (H5G_loc(gid, &start_loc) < 0)
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a location");

    
    udata.gid      = gid;
    udata.curr_loc = &start_loc;
    udata.idx_type = idx_type;
    udata.order    = order;
    udata.op       = op;
    udata.op_data  = op_data;

    
    if (NULL == (udata.path = H5MM_strdup("")))
        HGOTO_ERROR(H5E_SYM, H5E_NOSPACE, FAIL, "can't allocate path name buffer");
    udata.path_buf_size = 1;
    udata.curr_path_len = 0;

    
    if ((udata.visited = H5SL_create(H5SL_TYPE_OBJ, NULL)) == NULL)
        HGOTO_ERROR(H5E_SYM, H5E_CANTCREATE, FAIL, "can't create skip list for visited objects");

    
    {
        H5_obj_t *obj_pos; 

        
        if ((obj_pos = H5FL_MALLOC(H5_obj_t)) == NULL)
            HGOTO_ERROR(H5E_SYM, H5E_NOSPACE, FAIL, "can't allocate object node");

        
        H5F_GET_FILENO(grp->oloc.file, obj_pos->fileno);
        obj_pos->addr = grp->oloc.addr;

        
        if (H5SL_insert(udata.visited, obj_pos, obj_pos) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_CANTINSERT, FAIL, "can't insert object node into visited list");
    }

    
    if ((linfo_exists = H5G__obj_get_linfo(&(grp->oloc), &linfo)) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "can't check for link info message");
    if (linfo_exists) {
        
        if (idx_type == H5_INDEX_CRT_ORDER) {
            
            if (!linfo.track_corder)
                
                idx_type = H5_INDEX_NAME;
        } 
        else
            assert(idx_type == H5_INDEX_NAME);
    } 
    else {
        
        if (idx_type != H5_INDEX_NAME)
            
            idx_type = H5_INDEX_NAME;
    } 

    
    if ((ret_value =
             H5G__obj_iterate(&(grp->oloc), idx_type, order, (hsize_t)0, NULL, H5G__visit_cb, &udata)) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_BADITER, FAIL, "can't visit links");

done:
    
    H5MM_xfree(udata.path);
    if (udata.visited)
        H5SL_destroy(udata.visited, H5G__free_visit_visited, NULL);

    
    if (gid != H5I_INVALID_HID) {
        if (H5I_dec_app_ref(gid) < 0)
            HDONE_ERROR(H5E_SYM, H5E_CANTRELEASE, FAIL, "unable to close group");
    }
    else if (grp && H5G_close(grp) < 0)
        HDONE_ERROR(H5E_SYM, H5E_CLOSEERROR, FAIL, "unable to release group");

    FUNC_LEAVE_NOAPI(ret_value)
} 

hid_t
H5G_get_create_plist(const H5G_t *grp)
{
    H5O_linfo_t     linfo; 
    htri_t          ginfo_exists;
    htri_t          linfo_exists;
    htri_t          pline_exists;
    H5P_genplist_t *gcpl_plist;
    H5P_genplist_t *new_plist;
    hid_t           new_gcpl_id = H5I_INVALID_HID;
    hid_t           ret_value   = H5I_INVALID_HID;

    FUNC_ENTER_NOAPI(H5I_INVALID_HID)

    
    if (NULL == (gcpl_plist = (H5P_genplist_t *)H5I_object(H5P_LST_GROUP_CREATE_ID_g)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, H5I_INVALID_HID, "can't get default group creation property list");
    if ((new_gcpl_id = H5P_copy_plist(gcpl_plist, true)) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTGET, H5I_INVALID_HID, "unable to copy the creation property list");
    if (NULL == (new_plist = (H5P_genplist_t *)H5I_object(new_gcpl_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, H5I_INVALID_HID, "can't get property list");

    
    if (H5O_get_create_plist(&grp->oloc, new_plist) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTGET, H5I_INVALID_HID, "can't get object creation info");

    
    if ((ginfo_exists = H5O_msg_exists(&(grp->oloc), H5O_GINFO_ID)) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTINIT, H5I_INVALID_HID, "unable to read object header");
    if (ginfo_exists) {
        H5O_ginfo_t ginfo; 

        
        if (NULL == H5O_msg_read(&(grp->oloc), H5O_GINFO_ID, &ginfo))
            HGOTO_ERROR(H5E_SYM, H5E_BADMESG, H5I_INVALID_HID, "can't get group info");

        
        if (H5P_set(new_plist, H5G_CRT_GROUP_INFO_NAME, &ginfo) < 0)
            HGOTO_ERROR(H5E_PLIST, H5E_CANTSET, H5I_INVALID_HID, "can't set group info");
    } 

    
    if ((linfo_exists = H5G__obj_get_linfo(&(grp->oloc), &linfo)) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTINIT, H5I_INVALID_HID, "unable to read object header");
    if (linfo_exists) {
        
        if (H5P_set(new_plist, H5G_CRT_LINK_INFO_NAME, &linfo) < 0)
            HGOTO_ERROR(H5E_PLIST, H5E_CANTSET, H5I_INVALID_HID, "can't set link info");
    } 

    
    if ((pline_exists = H5O_msg_exists(&(grp->oloc), H5O_PLINE_ID)) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTGET, H5I_INVALID_HID, "unable to read object header");
    if (pline_exists) {
        H5O_pline_t pline; 

        
        if (NULL == H5O_msg_read(&(grp->oloc), H5O_PLINE_ID, &pline))
            HGOTO_ERROR(H5E_SYM, H5E_BADMESG, H5I_INVALID_HID, "can't get link pipeline");

        
        if (H5P_poke(new_plist, H5O_CRT_PIPELINE_NAME, &pline) < 0)
            HGOTO_ERROR(H5E_PLIST, H5E_CANTSET, H5I_INVALID_HID, "can't set link pipeline");
    } 

    
    ret_value = new_gcpl_id;

done:
    if (ret_value < 0) {
        if (new_gcpl_id > 0)
            if (H5I_dec_app_ref(new_gcpl_id) < 0)
                HDONE_ERROR(H5E_SYM, H5E_CANTDEC, H5I_INVALID_HID, "can't free");
    } 

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5G__get_info_by_name(const H5G_loc_t *loc, const char *name, H5G_info_t *grp_info)
{
    H5G_loc_t  grp_loc;             
    H5G_name_t grp_path;            
    H5O_loc_t  grp_oloc;            
    bool       loc_found = false;   
    herr_t     ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(loc);
    assert(grp_info);

    
    grp_loc.oloc = &grp_oloc;
    grp_loc.path = &grp_path;
    H5G_loc_reset(&grp_loc);

    
    if (H5G_loc_find(loc, name, &grp_loc ) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_NOTFOUND, FAIL, "group not found");
    loc_found = true;

    
    if (H5G__obj_info(grp_loc.oloc, grp_info ) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "can't retrieve group info");

done:
    
    if (loc_found && H5G_loc_free(&grp_loc) < 0)
        HDONE_ERROR(H5E_SYM, H5E_CANTRELEASE, FAIL, "can't free location");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5G__get_info_by_idx(const H5G_loc_t *loc, const char *group_name, H5_index_t idx_type, H5_iter_order_t order,
                     hsize_t n, H5G_info_t *grp_info)
{
    H5G_loc_t  grp_loc;             
    H5G_name_t grp_path;            
    H5O_loc_t  grp_oloc;            
    bool       loc_found = false;   
    herr_t     ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(loc);
    assert(grp_info);

    
    grp_loc.oloc = &grp_oloc;
    grp_loc.path = &grp_path;
    H5G_loc_reset(&grp_loc);

    
    if (H5G_loc_find_by_idx(loc, group_name, idx_type, order, n, &grp_loc ) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_NOTFOUND, FAIL, "group not found");
    loc_found = true;

    
    if (H5G__obj_info(grp_loc.oloc, grp_info ) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "can't retrieve group info");

done:
    
    if (loc_found && H5G_loc_free(&grp_loc) < 0)
        HDONE_ERROR(H5E_SYM, H5E_CANTRELEASE, FAIL, "can't free location");

    FUNC_LEAVE_NOAPI(ret_value)
} 

hid_t
H5G_get_gcpl_id(const H5G_obj_create_t *g)
{
    
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    assert(g);

    FUNC_LEAVE_NOAPI(g->gcpl_id);
} 
