/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Fmodule.h" 

#include "H5private.h"   
#include "H5ACprivate.h" 
#include "H5Eprivate.h"  
#include "H5Fpkg.h"      
#include "H5FDprivate.h" 
#include "H5FLprivate.h" 
#include "H5Iprivate.h"  
#include "H5MFprivate.h" 
#include "H5MMprivate.h" 
#include "H5Pprivate.h"  
#include "H5SMprivate.h" 

static herr_t H5F__super_ext_create(H5F_t *f, H5O_loc_t *ext_ptr);
static herr_t H5F__update_super_ext_driver_msg(H5F_t *f);

H5FL_DEFINE(H5F_super_t);

static const unsigned HDF5_superblock_ver_bounds[] = {
    HDF5_SUPERBLOCK_VERSION_DEF,   
    HDF5_SUPERBLOCK_VERSION_2,     
    HDF5_SUPERBLOCK_VERSION_3,     
    HDF5_SUPERBLOCK_VERSION_3,     
    HDF5_SUPERBLOCK_VERSION_3,     
    HDF5_SUPERBLOCK_VERSION_3,     
    HDF5_SUPERBLOCK_VERSION_LATEST 
};

static herr_t
H5F__super_ext_create(H5F_t *f, H5O_loc_t *ext_ptr)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(f->shared);
    assert(f->shared->sblock);
    assert(!H5_addr_defined(f->shared->sblock->ext_addr));
    assert(ext_ptr);

    
    if (f->shared->sblock->super_vers < HDF5_SUPERBLOCK_VERSION_2)
        HGOTO_ERROR(H5E_FILE, H5E_CANTCREATE, FAIL,
                    "superblock extension not permitted with version %u of superblock",
                    f->shared->sblock->super_vers);
    else if (H5_addr_defined(f->shared->sblock->ext_addr))
        HGOTO_ERROR(H5E_FILE, H5E_CANTCREATE, FAIL, "superblock extension already exists?!?!");
    else {
        
        H5O_loc_reset(ext_ptr);
        if (H5O_create(f, (size_t)0, (size_t)1, H5P_GROUP_CREATE_DEFAULT, ext_ptr) < 0)
            HGOTO_ERROR(H5E_FILE, H5E_CANTCREATE, FAIL, "unable to create superblock extension");

        
        f->shared->sblock->ext_addr = ext_ptr->addr;
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5F__super_ext_open(H5F_t *f, haddr_t ext_addr, H5O_loc_t *ext_ptr)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(H5_addr_defined(ext_addr));
    assert(ext_ptr);

    
    H5O_loc_reset(ext_ptr);
    ext_ptr->file = f;
    ext_ptr->addr = ext_addr;

    
    if (H5O_open(ext_ptr) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTOPENOBJ, FAIL, "unable to open superblock extension");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5F__super_ext_close(H5F_t *f, H5O_loc_t *ext_ptr, bool was_created)
{
    H5AC_ring_t orig_ring = H5AC_RING_INV; 
    herr_t      ret_value = SUCCEED;       

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(ext_ptr);

    
    if (was_created) {
        
        H5AC_set_ring(H5AC_RING_SBE, &orig_ring);

        
        if (H5O_link(ext_ptr, 1) < 0)
            HGOTO_ERROR(H5E_FILE, H5E_LINKCOUNT, FAIL, "unable to increment hard link count");

        
        if (H5O_dec_rc_by_loc(ext_ptr) < 0)
            HGOTO_ERROR(H5E_FILE, H5E_CANTDEC, FAIL, "unable to decrement refcount on superblock extension");
    } 

    
    f->nopen_objs++;
    if (H5O_close(ext_ptr, NULL) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTCLOSEOBJ, FAIL, "unable to close superblock extension");
    f->nopen_objs--;

done:
    
    if (orig_ring != H5AC_RING_INV)
        H5AC_set_ring(orig_ring, NULL);

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5F__update_super_ext_driver_msg(H5F_t *f)
{
    H5F_super_t *sblock;              
    herr_t       ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(f->shared);
    sblock = f->shared->sblock;
    assert(sblock);
    assert(sblock->cache_info.type == H5AC_SUPERBLOCK);

    
    if (sblock->super_vers >= HDF5_SUPERBLOCK_VERSION_2) {
        if (H5_addr_defined(sblock->ext_addr)) {
            
            if (!H5F_HAS_FEATURE(f, H5FD_FEAT_IGNORE_DRVRINFO)) {
                size_t driver_size; 

                
                H5_CHECKED_ASSIGN(driver_size, size_t, H5FD_sb_size(f->shared->lf), hsize_t);

                
                if (driver_size > 0) {
                    H5O_drvinfo_t drvinfo;                         
                    uint8_t       dbuf[H5F_MAX_DRVINFOBLOCK_SIZE]; 

                    
                    assert(driver_size <= H5F_MAX_DRVINFOBLOCK_SIZE);

                    
                    if (H5FD_sb_encode(f->shared->lf, drvinfo.name, dbuf) < 0)
                        HGOTO_ERROR(H5E_FILE, H5E_CANTINIT, FAIL, "unable to encode driver information");

                    
                    drvinfo.len = driver_size;
                    drvinfo.buf = dbuf;
                    if (H5F__super_ext_write_msg(f, H5O_DRVINFO_ID, &drvinfo, false, H5O_MSG_NO_FLAGS_SET) <
                        0)
                        HGOTO_ERROR(H5E_FILE, H5E_WRITEERROR, FAIL,
                                    "unable to update driver info header message");
                } 
            }     
        }         
    }             

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5F__super_read(H5F_t *f, H5P_genplist_t *fa_plist, bool initial_read)
{
    H5AC_ring_t               orig_ring = H5AC_RING_INV;
    H5F_super_t              *sblock    = NULL; 
    H5F_superblock_cache_ud_t udata;            
    H5P_genplist_t           *c_plist;          
    H5FD_t                   *file;             
    unsigned sblock_flags = H5AC__NO_FLAGS_SET; 
    haddr_t  super_addr   = HADDR_UNDEF;        
    haddr_t  eof;                               
    unsigned rw_flags;                          
    bool     skip_eof_check = false;            
#ifdef H5_HAVE_PARALLEL
    int mpi_size = 1;
#endif                          
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE_TAG(H5AC__SUPERBLOCK_TAG)

    
    f->shared->drvinfo = NULL;

    
    file = f->shared->lf;

    
#ifdef H5_HAVE_PARALLEL
    if (H5F_HAS_FEATURE(f, H5FD_FEAT_HAS_MPI))
        if ((mpi_size = H5F_mpi_get_size(f)) < 0)
            HGOTO_ERROR(H5E_FILE, H5E_CANTGET, FAIL, "can't retrieve MPI communicator size");

    
    if (mpi_size > 1) {
        MPI_Comm this_comm;
        int      mpi_rank;
        int      mpi_result;

        
        assert(H5F_HAS_FEATURE(f, H5FD_FEAT_HAS_MPI));

        
        if ((mpi_rank = H5F_mpi_get_rank(f)) < 0)
            HGOTO_ERROR(H5E_FILE, H5E_CANTGET, FAIL, "Can't get MPI rank");
        if (MPI_COMM_NULL == (this_comm = H5F_mpi_get_comm(f)))
            HGOTO_ERROR(H5E_FILE, H5E_CANTGET, FAIL, "can't get MPI communicator");

        
        if (0 == mpi_rank) {
            
            
            H5E_PAUSE_ERRORS
                {
                    H5FD_locate_signature(file, &super_addr);
                }
            H5E_RESUME_ERRORS
        } 

        
        if (MPI_SUCCESS != (mpi_result = MPI_Bcast(&super_addr, sizeof(super_addr), MPI_BYTE, 0, this_comm)))
            HMPI_GOTO_ERROR(FAIL, "MPI_Bcast failed", mpi_result)
    } 
    else {
        
#endif 

        if (H5FD_locate_signature(file, &super_addr) < 0)
            HGOTO_ERROR(H5E_FILE, H5E_NOTHDF5, FAIL, "unable to locate file signature");

#ifdef H5_HAVE_PARALLEL
    }  
#endif 
    if (!H5_addr_defined(super_addr))
        HGOTO_ERROR(H5E_FILE, H5E_NOTHDF5, FAIL, "file signature not found");

    
    if (H5_addr_gt(super_addr, 0)) {
        
        if (H5F__set_base_addr(f, super_addr) < 0)
            HGOTO_ERROR(H5E_FILE, H5E_CANTSET, FAIL, "failed to set base address for file driver");
    } 

    

    
    rw_flags = H5AC__FLUSH_LAST_FLAG;
#ifdef H5_HAVE_PARALLEL
    rw_flags |= H5C__FLUSH_COLLECTIVELY_FLAG;
#endif 
    if (!(H5F_INTENT(f) & H5F_ACC_RDWR))
        rw_flags |= H5AC__READ_ONLY_FLAG;

    
    if (NULL == (c_plist = (H5P_genplist_t *)H5I_object(f->shared->fcpl_id)))
        HGOTO_ERROR(H5E_FILE, H5E_BADTYPE, FAIL, "can't get property list");

    
    if (H5F__set_eoa(f, H5FD_MEM_SUPER, (haddr_t)H5F_SUPERBLOCK_SPEC_READ_SIZE) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTSET, FAIL, "set end of space allocation request failed");

    
    udata.f               = f;
    udata.ignore_drvrinfo = H5F_HAS_FEATURE(f, H5FD_FEAT_IGNORE_DRVRINFO);
    udata.sym_leaf_k      = 0;
    if (H5P_get(c_plist, H5F_CRT_BTREE_RANK_NAME, udata.btree_k) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTGET, FAIL, "unable to get rank for btree internal nodes");
    udata.stored_eof       = HADDR_UNDEF;
    udata.drvrinfo_removed = false;

    
    H5AC_set_ring(H5AC_RING_SB, &orig_ring);

    
    if (NULL == (sblock = (H5F_super_t *)H5AC_protect(f, H5AC_SUPERBLOCK, (haddr_t)0, &udata, rw_flags)))
        HGOTO_ERROR(H5E_FILE, H5E_CANTPROTECT, FAIL, "unable to load superblock");

    
    if (H5F_INTENT(f) & H5F_ACC_SWMR_WRITE) {
        if (sblock->super_vers < HDF5_SUPERBLOCK_VERSION_3)
            HGOTO_ERROR(H5E_FILE, H5E_BADVALUE, FAIL, "superblock version for SWMR is less than 3");
        if (sblock->super_vers > HDF5_superblock_ver_bounds[f->shared->high_bound])
            HGOTO_ERROR(H5E_FILE, H5E_BADVALUE, FAIL, "superblock version exceeds high bound");
        f->shared->low_bound = MAX(H5F_LIBVER_V110, f->shared->low_bound);
    }

    
    if (H5AC_pin_protected_entry(sblock) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTPIN, FAIL, "unable to pin superblock");

    
    if (((rw_flags & H5AC__READ_ONLY_FLAG) == 0) && udata.ignore_drvrinfo && udata.drvrinfo_removed) {
        assert(sblock->super_vers < HDF5_SUPERBLOCK_VERSION_2);
        sblock_flags |= H5AC__DIRTIED_FLAG;
    } 

    
    sblock_flags |= H5AC__FLUSH_LAST_FLAG;
#ifdef H5_HAVE_PARALLEL
    sblock_flags |= H5AC__FLUSH_COLLECTIVELY_FLAG;
#endif 

    
    if (!H5_addr_eq(super_addr, sblock->base_addr)) {
        
        
        udata.stored_eof -= (sblock->base_addr - super_addr);

        
        sblock->base_addr = super_addr;

        
        if (H5F__set_base_addr(f, sblock->base_addr) < 0)
            HGOTO_ERROR(H5E_FILE, H5E_CANTSET, FAIL, "failed to set base address for file driver");

        
        if ((rw_flags & H5AC__READ_ONLY_FLAG) == 0)
            sblock_flags |= H5AC__DIRTIED_FLAG;
    } 

    
    if (H5P_set(c_plist, H5F_CRT_SUPER_VERS_NAME, &sblock->super_vers) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTSET, FAIL, "unable to set superblock version");
    if (H5P_set(c_plist, H5F_CRT_ADDR_BYTE_NUM_NAME, &sblock->sizeof_addr) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTSET, FAIL, "unable to set byte number in an address");
    if (H5P_set(c_plist, H5F_CRT_OBJ_BYTE_NUM_NAME, &sblock->sizeof_size) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTSET, FAIL, "unable to set byte number for object size");

    
    if (sblock->super_vers < HDF5_SUPERBLOCK_VERSION_2) {
        
        assert(udata.sym_leaf_k != 0);

        
        if (H5P_set(c_plist, H5F_CRT_SYM_LEAF_NAME, &udata.sym_leaf_k) < 0)
            HGOTO_ERROR(H5E_FILE, H5E_CANTSET, FAIL, "unable to set rank for symbol table leaf nodes");
        sblock->sym_leaf_k = udata.sym_leaf_k;

        
        if (H5P_set(c_plist, H5F_CRT_BTREE_RANK_NAME, udata.btree_k) < 0)
            HGOTO_ERROR(H5E_FILE, H5E_CANTSET, FAIL, "unable to set rank for btree internal nodes");
        H5MM_memcpy(sblock->btree_k, udata.btree_k, sizeof(unsigned) * (size_t)H5B_NUM_BTREE_ID);
    } 
    else {
        
        
        if (H5P_get(c_plist, H5F_CRT_BTREE_RANK_NAME, sblock->btree_k) < 0)
            HGOTO_ERROR(H5E_FILE, H5E_CANTGET, FAIL, "unable to get rank for btree internal nodes");
        if (H5P_get(c_plist, H5F_CRT_SYM_LEAF_NAME, &sblock->sym_leaf_k) < 0)
            HGOTO_ERROR(H5E_FILE, H5E_CANTGET, FAIL, "unable to get rank for btree internal nodes");
    } 

    
    if (H5P_set(c_plist, H5F_CRT_USER_BLOCK_NAME, &sblock->base_addr) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTSET, FAIL, "unable to set userblock size");

    

    
    if (H5P_exist_plist(fa_plist, H5F_ACS_SKIP_EOF_CHECK_NAME) > 0)
        if (H5P_get(fa_plist, H5F_ACS_SKIP_EOF_CHECK_NAME, &skip_eof_check) < 0)
            HGOTO_ERROR(H5E_FILE, H5E_CANTGET, FAIL, "can't get skip EOF check value");

    if (H5F_INTENT(f) & H5F_ACC_SWMR_READ) {
        
        if (sblock->super_vers >= HDF5_SUPERBLOCK_VERSION_3)
            skip_eof_check = true;
    }
    if (!skip_eof_check && initial_read) {
        if (HADDR_UNDEF == (eof = H5FD_get_eof(f->shared->lf, H5FD_MEM_DEFAULT)))
            HGOTO_ERROR(H5E_FILE, H5E_CANTGET, FAIL, "unable to determine file size");

        
        if ((eof + sblock->base_addr) < udata.stored_eof)
            HGOTO_ERROR(H5E_FILE, H5E_TRUNCATED, FAIL,
                        "truncated file: eof = %llu, sblock->base_addr = %llu, stored_eof = %llu",
                        (unsigned long long)eof, (unsigned long long)sblock->base_addr,
                        (unsigned long long)udata.stored_eof);
    }

    

    
    H5AC_set_ring(H5AC_RING_SBE, NULL);

    
    if (H5_addr_defined(sblock->driver_addr)) {
        H5O_drvinfo_t          *drvinfo;             
        H5F_drvrinfo_cache_ud_t drvrinfo_udata;      
        unsigned drvinfo_flags = H5AC__NO_FLAGS_SET; 

        
        assert(sblock->super_vers < HDF5_SUPERBLOCK_VERSION_2);

        
        drvrinfo_udata.f           = f;
        drvrinfo_udata.driver_addr = sblock->driver_addr;

        
        if (H5FD_set_eoa(f->shared->lf, H5FD_MEM_SUPER, sblock->driver_addr + H5F_DRVINFOBLOCK_HDR_SIZE) <
            0) 
            HGOTO_ERROR(H5E_FILE, H5E_CANTINIT, FAIL, "set end of space allocation request failed");

        
        if (NULL == (drvinfo = (H5O_drvinfo_t *)H5AC_protect(f, H5AC_DRVRINFO, sblock->driver_addr,
                                                             &drvrinfo_udata, rw_flags)))
            HGOTO_ERROR(H5E_FILE, H5E_CANTPROTECT, FAIL, "unable to load driver info block");

        

        
        if (((rw_flags & H5AC__READ_ONLY_FLAG) == 0) && H5F_HAS_FEATURE(f, H5FD_FEAT_DIRTY_DRVRINFO_LOAD))
            drvinfo_flags |= H5AC__DIRTIED_FLAG;

        
        drvinfo_flags |= H5AC__PIN_ENTRY_FLAG;

        
        if (H5AC_unprotect(f, H5AC_DRVRINFO, sblock->driver_addr, drvinfo, drvinfo_flags) < 0)
            HGOTO_ERROR(H5E_FILE, H5E_CANTUNPROTECT, FAIL, "unable to release driver info block");

        
        f->shared->drvinfo = drvinfo;
    } 

    
    if (H5F__set_eoa(f, H5FD_MEM_DEFAULT, udata.stored_eof - sblock->base_addr) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTSET, FAIL, "unable to set end-of-address marker for file");

    
    if (H5_addr_defined(sblock->ext_addr)) {
        H5O_loc_t     ext_loc; 
        H5O_btreek_t  btreek;  
        H5O_drvinfo_t drvinfo; 
        size_t        u;       
        htri_t        status;  

        
        if (sblock->super_vers < HDF5_SUPERBLOCK_VERSION_2)
            HGOTO_ERROR(H5E_FILE, H5E_BADVALUE, FAIL,
                        "invalid superblock - extension message should not be defined for version < 2");

        
        if (H5_addr_gt(sblock->ext_addr, udata.stored_eof)) {
            
            if (H5F__set_eoa(f, H5FD_MEM_OHDR, (haddr_t)(sblock->ext_addr + 1024)) < 0)
                HGOTO_ERROR(H5E_FILE, H5E_CANTSET, FAIL, "unable to set end-of-address marker for file");
        } 

        
        if (H5F__super_ext_open(f, sblock->ext_addr, &ext_loc) < 0)
            HGOTO_ERROR(H5E_FILE, H5E_CANTOPENOBJ, FAIL, "unable to open file's superblock extension");

        
        if ((status = H5O_msg_exists(&ext_loc, H5O_DRVINFO_ID)) < 0)
            HGOTO_ERROR(H5E_FILE, H5E_EXISTS, FAIL, "unable to read object header");
        if (status) {
            
            if (!udata.ignore_drvrinfo) {

                
                if (NULL == H5O_msg_read(&ext_loc, H5O_DRVINFO_ID, &drvinfo))
                    HGOTO_ERROR(H5E_FILE, H5E_CANTGET, FAIL, "driver info message not present");

                
                if (H5FD_sb_load(f->shared->lf, drvinfo.name, drvinfo.buf) < 0) {
                    H5O_msg_reset(H5O_DRVINFO_ID, &drvinfo);
                    HGOTO_ERROR(H5E_FILE, H5E_CANTDECODE, FAIL, "unable to decode driver information");
                } 

                
                H5O_msg_reset(H5O_DRVINFO_ID, &drvinfo);

                assert(false == f->shared->drvinfo_sb_msg_exists);
                f->shared->drvinfo_sb_msg_exists = true;
            } 
        }     

        
        if (H5SM_get_info(&ext_loc, c_plist) < 0)
            HGOTO_ERROR(H5E_FILE, H5E_CANTGET, FAIL, "unable to read SOHM table information");

        
        if ((status = H5O_msg_exists(&ext_loc, H5O_BTREEK_ID)) < 0)
            HGOTO_ERROR(H5E_FILE, H5E_EXISTS, FAIL, "unable to read object header");
        if (status) {
            
            if (NULL == H5O_msg_read(&ext_loc, H5O_BTREEK_ID, &btreek))
                HGOTO_ERROR(H5E_FILE, H5E_CANTGET, FAIL, "v1 B-tree 'K' info message not present");

            
            sblock->btree_k[H5B_CHUNK_ID] = btreek.btree_k[H5B_CHUNK_ID];
            sblock->btree_k[H5B_SNODE_ID] = btreek.btree_k[H5B_SNODE_ID];
            sblock->sym_leaf_k            = btreek.sym_leaf_k;

            
            if (H5P_set(c_plist, H5F_CRT_BTREE_RANK_NAME, btreek.btree_k) < 0)
                HGOTO_ERROR(H5E_FILE, H5E_CANTSET, FAIL, "unable to set rank for btree internal nodes");
            if (H5P_set(c_plist, H5F_CRT_SYM_LEAF_NAME, &btreek.sym_leaf_k) < 0)
                HGOTO_ERROR(H5E_FILE, H5E_CANTSET, FAIL, "unable to set rank for symbol table leaf nodes");
        } 

        
        if ((status = H5O_msg_exists(&ext_loc, H5O_FSINFO_ID)) < 0)
            HGOTO_ERROR(H5E_FILE, H5E_EXISTS, FAIL, "unable to read object header");
        if (status) {
            uint8_t flags; 

            
            if (H5O_msg_get_flags(&ext_loc, H5O_FSINFO_ID, &flags) < 0)
                HGOTO_ERROR(H5E_FILE, H5E_CANTGET, FAIL,
                            "unable to message flags for free-space manager info message");

            
            if (!(flags & H5O_MSG_FLAG_WAS_UNKNOWN)) {
                H5O_fsinfo_t fsinfo; 

                memset(&fsinfo, 0, sizeof(H5O_fsinfo_t));

                
                
                if (H5P_exist_plist(fa_plist, H5F_ACS_NULL_FSM_ADDR_NAME) > 0)
                    if (H5P_get(fa_plist, H5F_ACS_NULL_FSM_ADDR_NAME, &f->shared->null_fsm_addr) < 0)
                        HGOTO_ERROR(H5E_FILE, H5E_CANTGET, FAIL,
                                    "can't get clearance for persisting fsm addr");

                
                if (NULL == H5O_msg_read(&ext_loc, H5O_FSINFO_ID, &fsinfo))
                    HGOTO_ERROR(H5E_FILE, H5E_CANTGET, FAIL, "unable to get free-space manager info message");

                
                if (H5O_fsinfo_check_version(H5F_HIGH_BOUND(f), &fsinfo) < 0)
                    HGOTO_ERROR(H5E_FILE, H5E_BADRANGE, FAIL,
                                "File space info message's version out of bounds");

                
                if (f->shared->fs_version != fsinfo.version)
                    f->shared->fs_version = fsinfo.version;
                if (f->shared->fs_strategy != fsinfo.strategy) {
                    f->shared->fs_strategy = fsinfo.strategy;

                    
                    if (H5P_set(c_plist, H5F_CRT_FILE_SPACE_STRATEGY_NAME, &fsinfo.strategy) < 0)
                        HGOTO_ERROR(H5E_FILE, H5E_CANTSET, FAIL, "unable to set file space strategy");
                } 
                if (f->shared->fs_persist != fsinfo.persist) {
                    f->shared->fs_persist = fsinfo.persist;

                    
                    if (H5P_set(c_plist, H5F_CRT_FREE_SPACE_PERSIST_NAME, &fsinfo.persist) < 0)
                        HGOTO_ERROR(H5E_FILE, H5E_CANTSET, FAIL, "unable to set file space strategy");
                } 
                if (f->shared->fs_threshold != fsinfo.threshold) {
                    f->shared->fs_threshold = fsinfo.threshold;

                    
                    if (H5P_set(c_plist, H5F_CRT_FREE_SPACE_THRESHOLD_NAME, &fsinfo.threshold) < 0)
                        HGOTO_ERROR(H5E_FILE, H5E_CANTSET, FAIL, "unable to set file space strategy");
                } 

                if (f->shared->fs_page_size < H5F_FILE_SPACE_PAGE_SIZE_MIN)
                    HGOTO_ERROR(H5E_FILE, H5E_BADVALUE, FAIL, "file space page size too small");
                if (fsinfo.page_size < H5F_FILE_SPACE_PAGE_SIZE_MIN)
                    HGOTO_ERROR(H5E_FILE, H5E_BADVALUE, FAIL, "file space page size too small");

                if (f->shared->fs_page_size != fsinfo.page_size) {
                    f->shared->fs_page_size = fsinfo.page_size;

                    
                    if (H5P_set(c_plist, H5F_CRT_FILE_SPACE_PAGE_SIZE_NAME, &fsinfo.page_size) < 0)
                        HGOTO_ERROR(H5E_FILE, H5E_CANTSET, FAIL, "unable to set file space page size");
                } 
                if (f->shared->pgend_meta_thres != fsinfo.pgend_meta_thres)
                    
                    f->shared->pgend_meta_thres = fsinfo.pgend_meta_thres;

                if (f->shared->eoa_fsm_fsalloc != fsinfo.eoa_pre_fsm_fsalloc)
                    f->shared->eoa_fsm_fsalloc = fsinfo.eoa_pre_fsm_fsalloc;

                
                if (!skip_eof_check && !f->shared->null_fsm_addr)
                    assert((!f->shared->fs_persist) || (f->shared->eoa_fsm_fsalloc != HADDR_UNDEF));

                

                f->shared->fs_addr[0] = HADDR_UNDEF;
                for (u = 1; u < NELMTS(f->shared->fs_addr); u++)
                    f->shared->fs_addr[u] = fsinfo.fs_addr[u - 1];

                
                if (f->shared->fs_persist && f->shared->null_fsm_addr)
                    for (u = 0; u < NELMTS(fsinfo.fs_addr); u++)
                        f->shared->fs_addr[u] = fsinfo.fs_addr[u] = HADDR_UNDEF;

                
                
                if (((fsinfo.mapped || f->shared->null_fsm_addr) && (rw_flags & H5AC__READ_ONLY_FLAG) == 0)) {

                    
#if 1  
                    
                    assert(f->shared->sblock == NULL);
                    f->shared->sblock = sblock;
#endif 

                    if (f->shared->null_fsm_addr) {
                        if (H5F__super_ext_write_msg(f, H5O_FSINFO_ID, &fsinfo, false,
                                                     H5O_MSG_FLAG_MARK_IF_UNKNOWN) < 0)
                            HGOTO_ERROR(H5E_FILE, H5E_WRITEERROR, FAIL,
                                        "error in writing fsinfo message to superblock extension");
                    }
                    else {
                        if (H5F__super_ext_remove_msg(f, H5O_FSINFO_ID) < 0) {
#if 1  
                            f->shared->sblock = NULL;
#endif 

                            HGOTO_ERROR(H5E_FILE, H5E_CANTDELETE, FAIL,
                                        "error in removing message from superblock extension");
                        }

                        if (H5F__super_ext_write_msg(f, H5O_FSINFO_ID, &fsinfo, true,
                                                     H5O_MSG_FLAG_MARK_IF_UNKNOWN) < 0)
                            HGOTO_ERROR(H5E_FILE, H5E_WRITEERROR, FAIL,
                                        "error in writing fsinfo message to superblock extension");
                    }
#if 1  
                    f->shared->sblock = NULL;
#endif 

                } 
            }     
        }         

        
        if ((status = H5O_msg_exists(&ext_loc, H5O_MDCI_MSG_ID)) < 0)
            HGOTO_ERROR(H5E_FILE, H5E_EXISTS, FAIL, "unable to read object header");
        if (status) {
            bool       rw = ((rw_flags & H5AC__READ_ONLY_FLAG) == 0);
            H5O_mdci_t mdci_msg;

            

            
            if (NULL == H5O_msg_read(&ext_loc, H5O_MDCI_MSG_ID, &mdci_msg))
                HGOTO_ERROR(H5E_FILE, H5E_CANTGET, FAIL, "unable to get metadata cache image message");

            
            if (H5AC_load_cache_image_on_next_protect(f, mdci_msg.addr, mdci_msg.size, rw) < 0)
                HGOTO_ERROR(H5E_FILE, H5E_CANTLOAD, FAIL,
                            "call to H5AC_load_cache_image_on_next_protect failed");
        } 

        
        if (H5F__super_ext_close(f, &ext_loc, false) < 0)
            HGOTO_ERROR(H5E_FILE, H5E_CANTCLOSEOBJ, FAIL, "unable to close file's superblock extension");
    } 

    
    
    if (((rw_flags & H5AC__READ_ONLY_FLAG) == 0) && sblock->super_vers >= HDF5_SUPERBLOCK_VERSION_2 &&
        H5_addr_defined(sblock->ext_addr)) {
        
        if (H5F_HAS_FEATURE(f, H5FD_FEAT_DIRTY_DRVRINFO_LOAD)) {
            size_t driver_size; 

            
            H5_CHECKED_ASSIGN(driver_size, size_t, H5FD_sb_size(f->shared->lf), hsize_t);
            if (driver_size > 0) {
                H5O_drvinfo_t drvinfo;                         
                uint8_t       dbuf[H5F_MAX_DRVINFOBLOCK_SIZE]; 

                
                assert(driver_size <= H5F_MAX_DRVINFOBLOCK_SIZE);

                
                if (H5FD_sb_encode(f->shared->lf, drvinfo.name, dbuf) < 0)
                    HGOTO_ERROR(H5E_FILE, H5E_CANTINIT, FAIL, "unable to encode driver information");

                
                drvinfo.len = driver_size;
                drvinfo.buf = dbuf;

                

#if 1  
                
                assert(f->shared->sblock == NULL);
                f->shared->sblock = sblock;
#endif 
                if (H5F__super_ext_write_msg(f, H5O_DRVINFO_ID, &drvinfo, false, H5O_MSG_NO_FLAGS_SET) < 0)
                    HGOTO_ERROR(H5E_FILE, H5E_WRITEERROR, FAIL,
                                "error in writing message to superblock extension");

#if 1  
                f->shared->sblock = NULL;
#endif 

            } 
        }     
        
        else if (H5F_HAS_FEATURE(f, H5FD_FEAT_IGNORE_DRVRINFO)) {
            
            if (H5F__super_ext_remove_msg(f, H5O_DRVINFO_ID) < 0)
                HGOTO_ERROR(H5E_FILE, H5E_CANTRELEASE, FAIL,
                            "error in removing message from superblock extension");

            
            if (!H5_addr_defined(sblock->ext_addr))
                sblock_flags |= H5AC__DIRTIED_FLAG;
        } 
    }     

    
    f->shared->sblock = sblock;

    
    if (H5F__set_paged_aggr(f, (bool)H5F_PAGED_AGGR(f)) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTSET, FAIL, "failed to set paged_aggr status for file driver");

done:
    
    if (orig_ring != H5AC_RING_INV)
        H5AC_set_ring(orig_ring, NULL);

    
    if (sblock && H5AC_unprotect(f, H5AC_SUPERBLOCK, (haddr_t)0, sblock, sblock_flags) < 0)
        HDONE_ERROR(H5E_FILE, H5E_CANTUNPROTECT, FAIL, "unable to close superblock");

    
    if (ret_value < 0) {
        
        if (f->shared->drvinfo) {
            if (H5AC_unpin_entry(f->shared->drvinfo) < 0)
                HDONE_ERROR(H5E_FILE, H5E_CANTUNPIN, FAIL, "unable to unpin driver info");

            
            if (sblock) {
                if (H5AC_expunge_entry(f, H5AC_DRVRINFO, sblock->driver_addr, H5AC__NO_FLAGS_SET) < 0)
                    HDONE_ERROR(H5E_FILE, H5E_CANTEXPUNGE, FAIL, "unable to expunge driver info block");
                f->shared->drvinfo = NULL;
            }
        } 

        
        if (sblock) {
            
            if (H5AC_unpin_entry(sblock) < 0)
                HDONE_ERROR(H5E_FILE, H5E_CANTUNPIN, FAIL, "unable to unpin superblock");

            
            if (H5AC_expunge_entry(f, H5AC_SUPERBLOCK, (haddr_t)0, H5AC__NO_FLAGS_SET) < 0)
                HDONE_ERROR(H5E_FILE, H5E_CANTEXPUNGE, FAIL, "unable to expunge superblock");
            f->shared->sblock = NULL;
        } 
    }     

    FUNC_LEAVE_NOAPI_TAG(ret_value)
} 

herr_t
H5F__super_init(H5F_t *f)
{
    H5F_super_t *sblock    = NULL;  
    bool sblock_in_cache   = false; 
    H5O_drvinfo_t *drvinfo = NULL;  
    bool           drvinfo_in_cache =
        false;             
    H5P_genplist_t *plist; 
    H5AC_ring_t     orig_ring = H5AC_RING_INV;
    hsize_t         userblock_size;      
    hsize_t         superblock_size = 0; 
    haddr_t         superblock_addr = HADDR_UNDEF;
    size_t          driver_size;                              
    unsigned        super_vers = HDF5_SUPERBLOCK_VERSION_DEF; 
    H5O_loc_t       ext_loc;                                  
    bool            need_ext;                                 
    bool            ext_created    = false;                   
    bool   non_default_fs_settings = false;   
    herr_t ret_value               = SUCCEED; 

    FUNC_ENTER_PACKAGE_TAG(H5AC__SUPERBLOCK_TAG)

    
    if (NULL == (sblock = H5FL_CALLOC(H5F_super_t)))
        HGOTO_ERROR(H5E_FILE, H5E_CANTALLOC, FAIL, "memory allocation failed");

    
    sblock->base_addr   = HADDR_UNDEF;
    sblock->ext_addr    = HADDR_UNDEF;
    sblock->driver_addr = HADDR_UNDEF;
    sblock->root_addr   = HADDR_UNDEF;

    
    if (NULL == (plist = (H5P_genplist_t *)H5I_object(f->shared->fcpl_id)))
        HGOTO_ERROR(H5E_FILE, H5E_BADTYPE, FAIL, "not a property list");

    
    if (H5P_get(plist, H5F_CRT_SYM_LEAF_NAME, &sblock->sym_leaf_k) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTGET, FAIL, "can't get byte number for object size");

    
    if (H5P_get(plist, H5F_CRT_BTREE_RANK_NAME, &sblock->btree_k[0]) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTGET, FAIL, "unable to get rank for btree internal nodes");

    
    if (!(f->shared->fs_strategy == H5F_FILE_SPACE_STRATEGY_DEF &&
          f->shared->fs_persist == H5F_FREE_SPACE_PERSIST_DEF &&
          f->shared->fs_threshold == H5F_FREE_SPACE_THRESHOLD_DEF &&
          f->shared->fs_page_size == H5F_FILE_SPACE_PAGE_SIZE_DEF))
        non_default_fs_settings = true;

    

    
    if (H5F_INTENT(f) & H5F_ACC_SWMR_WRITE) {
        super_vers           = HDF5_SUPERBLOCK_VERSION_3;
        f->shared->low_bound = MAX(H5F_LIBVER_V110, f->shared->low_bound);

        
    }
    else if (f->shared->sohm_nindexes > 0)
        super_vers = HDF5_SUPERBLOCK_VERSION_2;
    
    else if (non_default_fs_settings)
        super_vers = HDF5_SUPERBLOCK_VERSION_2;
    
    else if (sblock->btree_k[H5B_CHUNK_ID] != HDF5_BTREE_CHUNK_IK_DEF)
        super_vers = HDF5_SUPERBLOCK_VERSION_1;

    
    super_vers = MAX(super_vers, HDF5_superblock_ver_bounds[f->shared->low_bound]);

    
    if (super_vers > HDF5_superblock_ver_bounds[f->shared->high_bound])
        HGOTO_ERROR(H5E_FILE, H5E_BADVALUE, FAIL, "superblock version out of bounds");

    
    if (super_vers != HDF5_SUPERBLOCK_VERSION_DEF) {
        H5P_genplist_t *c_plist; 

        if (NULL == (c_plist = (H5P_genplist_t *)H5I_object(f->shared->fcpl_id)))
            HGOTO_ERROR(H5E_FILE, H5E_BADTYPE, FAIL, "not property list");
        if (H5P_set(c_plist, H5F_CRT_SUPER_VERS_NAME, &super_vers) < 0)
            HGOTO_ERROR(H5E_FILE, H5E_CANTSET, FAIL, "unable to set superblock version");
    } 

    if (H5FD_set_paged_aggr(f->shared->lf, (bool)H5F_PAGED_AGGR(f)) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTINIT, FAIL, "failed to set paged_aggr status for file driver");

    
    if (H5P_get(plist, H5F_CRT_USER_BLOCK_NAME, &userblock_size) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTGET, FAIL, "unable to get userblock size");

    
    if (userblock_size > 0) {
        
        hsize_t alignment = H5F_PAGED_AGGR(f) ? f->shared->fs_page_size : f->shared->alignment;

        if (userblock_size < alignment)
            HGOTO_ERROR(H5E_FILE, H5E_BADVALUE, FAIL, "userblock size must be > file object alignment");
        if (0 != (userblock_size % alignment))
            HGOTO_ERROR(H5E_FILE, H5E_BADVALUE, FAIL,
                        "userblock size must be an integral multiple of file object alignment");
    } 

    sblock->base_addr    = userblock_size;
    sblock->status_flags = 0;

    
    if (H5F__set_eoa(f, H5FD_MEM_SUPER, userblock_size) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTINIT, FAIL, "unable to set EOA value for userblock");

    
    if (H5F__set_base_addr(f, sblock->base_addr) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTINIT, FAIL, "failed to set base address for file driver");

    
    sblock->super_vers  = super_vers;
    sblock->sizeof_addr = f->shared->sizeof_addr;
    sblock->sizeof_size = f->shared->sizeof_size;

    
    superblock_size = (hsize_t)H5F_SUPERBLOCK_SIZE(sblock);

    
    H5_CHECKED_ASSIGN(driver_size, size_t, H5FD_sb_size(f->shared->lf), hsize_t);

    
    if (driver_size > 0) {
        
        
        if (super_vers < HDF5_SUPERBLOCK_VERSION_2)
            driver_size += H5F_DRVINFOBLOCK_HDR_SIZE;

        
        sblock->driver_addr = superblock_size;
    } 

    
    if (super_vers < HDF5_SUPERBLOCK_VERSION_2)
        superblock_size += driver_size;

    
    H5AC_set_ring(H5AC_RING_SB, &orig_ring);

    
    if (H5AC_insert_entry(f, H5AC_SUPERBLOCK, (haddr_t)0, sblock,
                          H5AC__PIN_ENTRY_FLAG | H5AC__FLUSH_LAST_FLAG | H5AC__FLUSH_COLLECTIVELY_FLAG) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTINS, FAIL, "can't add superblock to cache");
    sblock_in_cache = true;

    
    f->shared->sblock = sblock;

    
    if (HADDR_UNDEF == (superblock_addr = H5MF_alloc(f, H5FD_MEM_SUPER, superblock_size)))
        HGOTO_ERROR(H5E_FILE, H5E_CANTALLOC, FAIL, "file allocation failed for superblock");

    
    f->shared->drvinfo = NULL;

    

    
    if (f->shared->sohm_nindexes > 0) {
        assert(super_vers >= HDF5_SUPERBLOCK_VERSION_2);
        need_ext = true;
    } 
    
    else if (non_default_fs_settings) {
        assert(super_vers >= HDF5_SUPERBLOCK_VERSION_2);
        need_ext = true;
    } 
    
    else if (super_vers >= HDF5_SUPERBLOCK_VERSION_2) {
        
        if (sblock->btree_k[H5B_SNODE_ID] != HDF5_BTREE_SNODE_IK_DEF ||
            sblock->btree_k[H5B_CHUNK_ID] != HDF5_BTREE_CHUNK_IK_DEF ||
            sblock->sym_leaf_k != H5F_CRT_SYM_LEAF_DEF)
            need_ext = true;
        
        else if (driver_size > 0)
            need_ext = true;
        else
            need_ext = false;
    } 
    else
        need_ext = false;

    
    H5AC_set_ring(H5AC_RING_SBE, NULL);

    
    if (need_ext) {
        
        if (H5F__super_ext_create(f, &ext_loc) < 0)
            HGOTO_ERROR(H5E_FILE, H5E_CANTCREATE, FAIL, "unable to create superblock extension");
        ext_created = true;

        
        if (f->shared->sohm_nindexes > 0) {
            
            if (H5SM_init(f, plist, &ext_loc) < 0)
                HGOTO_ERROR(H5E_FILE, H5E_CANTINIT, FAIL, "unable to create SOHM table");
        }

        
        if (sblock->btree_k[H5B_SNODE_ID] != HDF5_BTREE_SNODE_IK_DEF ||
            sblock->btree_k[H5B_CHUNK_ID] != HDF5_BTREE_CHUNK_IK_DEF ||
            sblock->sym_leaf_k != H5F_CRT_SYM_LEAF_DEF) {
            H5O_btreek_t btreek; 

            
            btreek.btree_k[H5B_CHUNK_ID] = sblock->btree_k[H5B_CHUNK_ID];
            btreek.btree_k[H5B_SNODE_ID] = sblock->btree_k[H5B_SNODE_ID];
            btreek.sym_leaf_k            = sblock->sym_leaf_k;
            if (H5O_msg_create(&ext_loc, H5O_BTREEK_ID, H5O_MSG_FLAG_CONSTANT | H5O_MSG_FLAG_DONTSHARE,
                               H5O_UPDATE_TIME, &btreek) < 0)
                HGOTO_ERROR(H5E_FILE, H5E_CANTINIT, FAIL,
                            "unable to update v1 B-tree 'K' value header message");
        } 

        
        if (driver_size > 0) {
            H5O_drvinfo_t info;                            
            uint8_t       dbuf[H5F_MAX_DRVINFOBLOCK_SIZE]; 

            
            assert(driver_size <= H5F_MAX_DRVINFOBLOCK_SIZE);

            
            memset(dbuf, 0, sizeof(dbuf));
            if (H5FD_sb_encode(f->shared->lf, info.name, dbuf) < 0)
                HGOTO_ERROR(H5E_FILE, H5E_CANTINIT, FAIL, "unable to encode driver information");

            
            info.len = driver_size;
            info.buf = dbuf;
            if (H5O_msg_create(&ext_loc, H5O_DRVINFO_ID, H5O_MSG_FLAG_DONTSHARE, H5O_UPDATE_TIME, &info) < 0)
                HGOTO_ERROR(H5E_FILE, H5E_CANTINIT, FAIL, "unable to update driver info header message");

            assert(false == f->shared->drvinfo_sb_msg_exists);
            f->shared->drvinfo_sb_msg_exists = true;
        } 

        
        if (non_default_fs_settings) {
            H5F_mem_page_t ptype;
            H5O_fsinfo_t   fsinfo; 

            
            fsinfo.strategy            = f->shared->fs_strategy;
            fsinfo.persist             = f->shared->fs_persist;
            fsinfo.threshold           = f->shared->fs_threshold;
            fsinfo.page_size           = f->shared->fs_page_size;
            fsinfo.pgend_meta_thres    = f->shared->pgend_meta_thres;
            fsinfo.eoa_pre_fsm_fsalloc = HADDR_UNDEF;
            fsinfo.mapped              = false;

            
            if (H5O_fsinfo_set_version(H5F_LOW_BOUND(f), H5F_HIGH_BOUND(f), &fsinfo) < 0)
                HGOTO_ERROR(H5E_FILE, H5E_CANTSET, FAIL, "can't set version of fsinfo");
            f->shared->fs_version = fsinfo.version;

            for (ptype = H5F_MEM_PAGE_SUPER; ptype < H5F_MEM_PAGE_NTYPES; ptype++)
                fsinfo.fs_addr[ptype - 1] = HADDR_UNDEF;

            if (H5O_msg_create(&ext_loc, H5O_FSINFO_ID, H5O_MSG_FLAG_DONTSHARE | H5O_MSG_FLAG_MARK_IF_UNKNOWN,
                               H5O_UPDATE_TIME, &fsinfo) < 0)
                HGOTO_ERROR(H5E_FILE, H5E_CANTINIT, FAIL, "unable to update free-space info header message");
        } 
    }     
    else {
        
        if (driver_size > 0) {
            
            assert(H5_addr_defined(sblock->driver_addr));

            
            if (NULL == (drvinfo = (H5O_drvinfo_t *)H5MM_calloc(sizeof(H5O_drvinfo_t))))
                HGOTO_ERROR(H5E_FILE, H5E_CANTALLOC, FAIL,
                            "memory allocation failed for driver info message");

            
            
            H5_CHECKED_ASSIGN(drvinfo->len, size_t, H5FD_sb_size(f->shared->lf), hsize_t);

            
            if (H5AC_insert_entry(f, H5AC_DRVRINFO, sblock->driver_addr, drvinfo,
                                  H5AC__PIN_ENTRY_FLAG | H5AC__FLUSH_LAST_FLAG |
                                      H5AC__FLUSH_COLLECTIVELY_FLAG) < 0)
                HGOTO_ERROR(H5E_FILE, H5E_CANTINS, FAIL, "can't add driver info block to cache");
            drvinfo_in_cache   = true;
            f->shared->drvinfo = drvinfo;
        } 
        else
            assert(!H5_addr_defined(sblock->driver_addr));
    } 

done:
    
    if (orig_ring != H5AC_RING_INV)
        H5AC_set_ring(orig_ring, NULL);

    
    if (ext_created && H5F__super_ext_close(f, &ext_loc, ext_created) < 0)
        HDONE_ERROR(H5E_FILE, H5E_CANTRELEASE, FAIL, "unable to close file's superblock extension");

    
    if (ret_value < 0) {
        
        if (drvinfo) {
            
            if (drvinfo_in_cache) {
                
                if (H5AC_unpin_entry(drvinfo) < 0)
                    HDONE_ERROR(H5E_FILE, H5E_CANTUNPIN, FAIL, "unable to unpin driver info");

                
                if (H5AC_expunge_entry(f, H5AC_DRVRINFO, sblock->driver_addr, H5AC__NO_FLAGS_SET) < 0)
                    HDONE_ERROR(H5E_FILE, H5E_CANTEXPUNGE, FAIL, "unable to expunge driver info block");
            } 
            else
                
                H5MM_xfree(drvinfo);
        } 

        
        if (sblock) {
            if (non_default_fs_settings && H5_addr_defined(superblock_addr)) {
                
                if (H5MF_try_close(f) < 0)
                    HDONE_ERROR(H5E_FILE, H5E_CANTFREE, FAIL, "can't close file free space manager");
            }

            
            if (sblock_in_cache) {
                
                if (H5AC_unpin_entry(sblock) < 0)
                    HDONE_ERROR(H5E_FILE, H5E_CANTUNPIN, FAIL, "unable to unpin superblock");

                
                if (H5AC_expunge_entry(f, H5AC_SUPERBLOCK, (haddr_t)0, H5AC__NO_FLAGS_SET) < 0)
                    HDONE_ERROR(H5E_FILE, H5E_CANTEXPUNGE, FAIL, "unable to expunge superblock");
            } 
            else
                
                if (H5F__super_free(sblock) < 0)
                    HDONE_ERROR(H5E_FILE, H5E_CANTFREE, FAIL, "unable to destroy superblock");

            
            f->shared->sblock = NULL;
        } 
    }     

    FUNC_LEAVE_NOAPI_TAG(ret_value)
} 

herr_t
H5F_eoa_dirty(H5F_t *f)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f);
    assert(f->shared);
    assert(f->shared->sblock);

    
    if (H5F_super_dirty(f) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTMARKDIRTY, FAIL, "unable to mark superblock as dirty");

    
    if (f->shared->drvinfo) {
        if (H5AC_mark_entry_dirty(f->shared->drvinfo) < 0)
            HGOTO_ERROR(H5E_FILE, H5E_CANTMARKDIRTY, FAIL, "unable to mark drvinfo as dirty");
    } 
    
    else if (f->shared->drvinfo_sb_msg_exists) {
        if (H5F__update_super_ext_driver_msg(f) < 0)
            HGOTO_ERROR(H5E_FILE, H5E_CANTMARKDIRTY, FAIL, "unable to mark drvinfo message as dirty");
    }

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5F_super_dirty(H5F_t *f)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f);
    assert(f->shared);
    assert(f->shared->sblock);

    
    if (H5AC_mark_entry_dirty(f->shared->sblock) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTMARKDIRTY, FAIL, "unable to mark superblock as dirty");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5F__super_free(H5F_super_t *sblock)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(sblock);

    
    sblock->root_ent = (H5G_entry_t *)H5MM_xfree(sblock->root_ent);

    
    sblock = (H5F_super_t *)H5FL_FREE(H5F_super_t, sblock);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5F__super_size(H5F_t *f, hsize_t *super_size, hsize_t *super_ext_size)
{
    H5AC_ring_t orig_ring = H5AC_RING_INV; 
    herr_t      ret_value = SUCCEED;       

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(f->shared);
    assert(f->shared->sblock);

    
    if (super_size)
        *super_size = (hsize_t)H5F_SUPERBLOCK_SIZE(f->shared->sblock);

    
    if (super_ext_size) {
        if (H5_addr_defined(f->shared->sblock->ext_addr)) {
            H5O_loc_t      ext_loc;  
            H5O_hdr_info_t hdr_info; 

            
            H5O_loc_reset(&ext_loc);
            ext_loc.file = f;
            ext_loc.addr = f->shared->sblock->ext_addr;

            
            H5AC_set_ring(H5AC_RING_SBE, &orig_ring);

            
            if (H5O_get_hdr_info(&ext_loc, &hdr_info) < 0)
                HGOTO_ERROR(H5E_FILE, H5E_CANTGET, FAIL, "unable to retrieve superblock extension info");

            
            *super_ext_size = hdr_info.space.total;
        } 
        else
            
            *super_ext_size = (hsize_t)0;
    } 

done:
    
    if (orig_ring != H5AC_RING_INV)
        H5AC_set_ring(orig_ring, NULL);

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5F__super_ext_write_msg(H5F_t *f, unsigned id, void *mesg, bool may_create, unsigned mesg_flags)
{
    H5AC_ring_t orig_ring   = H5AC_RING_INV; 
    bool        ext_created = false;         
    bool        ext_opened  = false;         
    H5O_loc_t   ext_loc;                     
    htri_t      status;                      
    herr_t      ret_value = SUCCEED;         

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(f->shared);
    assert(f->shared->sblock);

    
    H5AC_set_ring(H5AC_RING_SBE, &orig_ring);

    
    if (H5_addr_defined(f->shared->sblock->ext_addr)) {
        if (H5F__super_ext_open(f, f->shared->sblock->ext_addr, &ext_loc) < 0)
            HGOTO_ERROR(H5E_FILE, H5E_CANTOPENOBJ, FAIL, "unable to open file's superblock extension");
    } 
    else {
        assert(may_create);
        if (H5F__super_ext_create(f, &ext_loc) < 0)
            HGOTO_ERROR(H5E_FILE, H5E_CANTCREATE, FAIL, "unable to create file's superblock extension");
        ext_created = true;
    } 
    assert(H5_addr_defined(ext_loc.addr));
    ext_opened = true;

    
    if ((status = H5O_msg_exists(&ext_loc, id)) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTGET, FAIL,
                    "unable to check object header for message or message exists");

    
    if (may_create) {
        if (status)
            HGOTO_ERROR(H5E_FILE, H5E_CANTGET, FAIL, "Message should not exist");

        
        if (H5O_msg_create(&ext_loc, id, (mesg_flags | H5O_MSG_FLAG_DONTSHARE), H5O_UPDATE_TIME, mesg) < 0)
            HGOTO_ERROR(H5E_FILE, H5E_CANTGET, FAIL, "unable to create the message in object header");
    } 
    else {
        if (!status)
            HGOTO_ERROR(H5E_FILE, H5E_CANTGET, FAIL, "Message should exist");

        
        if (H5O_msg_write(&ext_loc, id, (mesg_flags | H5O_MSG_FLAG_DONTSHARE), H5O_UPDATE_TIME, mesg) < 0)
            HGOTO_ERROR(H5E_FILE, H5E_CANTGET, FAIL, "unable to write the message in object header");
    } 

done:
    
    if (orig_ring != H5AC_RING_INV)
        H5AC_set_ring(orig_ring, NULL);

    
    if (ext_opened && H5F__super_ext_close(f, &ext_loc, ext_created) < 0)
        HDONE_ERROR(H5E_FILE, H5E_CANTRELEASE, FAIL, "unable to close file's superblock extension");

    
    if (ext_created && H5AC_mark_entry_dirty(f->shared->sblock) < 0)
        HDONE_ERROR(H5E_FILE, H5E_CANTMARKDIRTY, FAIL, "unable to mark superblock as dirty");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5F__super_ext_remove_msg(H5F_t *f, unsigned id)
{
    H5AC_ring_t orig_ring = H5AC_RING_INV; 
    H5O_loc_t   ext_loc;                   
    bool        ext_opened = false;        
    int         null_count = 0;            
    htri_t      status;                    
    herr_t      ret_value = SUCCEED;       

    FUNC_ENTER_PACKAGE

    
    assert(H5_addr_defined(f->shared->sblock->ext_addr));

    
    H5AC_set_ring(H5AC_RING_SBE, &orig_ring);

    
    if (H5F__super_ext_open(f, f->shared->sblock->ext_addr, &ext_loc) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTRELEASE, FAIL, "error in starting file's superblock extension");
    ext_opened = true;

    
    if ((status = H5O_msg_exists(&ext_loc, id)) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTGET, FAIL, "unable to check object header for message");
    else if (status) {
        
        H5O_hdr_info_t hdr_info; 

        
        if (H5O_msg_remove(&ext_loc, id, H5O_ALL, true) < 0)
            HGOTO_ERROR(H5E_FILE, H5E_CANTDELETE, FAIL, "unable to delete free-space manager info message");

        
        if (H5O_get_hdr_info(&ext_loc, &hdr_info) < 0)
            HGOTO_ERROR(H5E_FILE, H5E_CANTGET, FAIL, "unable to retrieve superblock extension info");

        
        if (hdr_info.nchunks == 1) {
            if ((null_count = H5O_msg_count(&ext_loc, H5O_NULL_ID)) < 0)
                HGOTO_ERROR(H5E_FILE, H5E_CANTCOUNT, FAIL, "unable to count messages");
            else if ((unsigned)null_count == hdr_info.nmesgs) {
                assert(H5_addr_defined(ext_loc.addr));
                if (H5O_delete(f, ext_loc.addr) < 0)
                    HGOTO_ERROR(H5E_FILE, H5E_CANTCOUNT, FAIL, "unable to count messages");
                f->shared->sblock->ext_addr = HADDR_UNDEF;
            } 
        }     
    }         

done:
    
    if (orig_ring != H5AC_RING_INV)
        H5AC_set_ring(orig_ring, NULL);

    
    if (ext_opened && H5F__super_ext_close(f, &ext_loc, false) < 0)
        HDONE_ERROR(H5E_FILE, H5E_CANTRELEASE, FAIL, "unable to close file's superblock extension");

    FUNC_LEAVE_NOAPI(ret_value)
} 
