/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Bmodule.h" 

#include "H5private.h"   
#include "H5Bpkg.h"      
#include "H5CXprivate.h" 
#include "H5Eprivate.h"  
#include "H5FLprivate.h" 
#include "H5MFprivate.h" 
#include "H5MMprivate.h" 

#define H5B_SIZEOF_HDR(F)                                                                                    \
    (H5_SIZEOF_MAGIC +                                                                  \
     4 +                                                                     \
     2 * H5F_SIZEOF_ADDR(F)) 

#define H5B_INS_UD_T_NULL                                                                                    \
    {                                                                                                        \
        NULL, HADDR_UNDEF, H5AC__NO_FLAGS_SET                                                                \
    }

typedef struct H5B_iter_ud_t {
    H5B_info_t *bt_info; 
    void       *udata;   
} H5B_info_ud_t;

typedef struct H5B_ins_ud_t {
    H5B_t   *bt;          
    haddr_t  addr;        
    unsigned cache_flags; 
} H5B_ins_ud_t;

static herr_t    H5B_find_helper(H5F_t *f, const H5B_class_t *type, haddr_t addr, int exp_level, bool *found,
                                 void *udata);
static H5B_ins_t H5B__insert_helper(H5F_t *f, H5B_ins_ud_t *bt_ud, const H5B_class_t *type, uint8_t *lt_key,
                                    bool *lt_key_changed, uint8_t *md_key, void *udata, uint8_t *rt_key,
                                    bool *rt_key_changed, H5B_ins_ud_t *split_bt_ud );
static herr_t H5B__insert_child(H5B_t *bt, unsigned *bt_flags, unsigned idx, haddr_t child, H5B_ins_t anchor,
                                const void *md_key);
static herr_t H5B__split(H5F_t *f, H5B_ins_ud_t *bt_ud, unsigned idx, void *udata,
                         H5B_ins_ud_t *split_bt_ud );
static H5B_t *H5B__copy(const H5B_t *old_bt);

bool H5_PKG_INIT_VAR = false;

H5FL_SEQ_DEFINE(haddr_t);

H5FL_BLK_DEFINE(native_block);

H5FL_DEFINE(H5B_t);

H5FL_DEFINE_STATIC(H5B_shared_t);

H5FL_BLK_DEFINE_STATIC(page);

H5FL_SEQ_DEFINE_STATIC(size_t);

herr_t
H5B_create(H5F_t *f, const H5B_class_t *type, void *udata, haddr_t *addr_p )
{
    H5B_t        *bt        = NULL;
    H5B_shared_t *shared    = NULL; 
    herr_t        ret_value = SUCCEED;

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f);
    assert(type);
    assert(addr_p);

    
    if (NULL == (bt = H5FL_CALLOC(H5B_t)))
        HGOTO_ERROR(H5E_BTREE, H5E_CANTALLOC, FAIL, "memory allocation failed for B-tree root node");
    memset(&bt->cache_info, 0, sizeof(H5AC_info_t));
    bt->level     = 0;
    bt->left      = HADDR_UNDEF;
    bt->right     = HADDR_UNDEF;
    bt->nchildren = 0;
    if (NULL == (bt->rc_shared = (type->get_shared)(f, udata)))
        HGOTO_ERROR(H5E_BTREE, H5E_CANTGET, FAIL, "can't retrieve B-tree node buffer");
    H5UC_INC(bt->rc_shared);
    shared = (H5B_shared_t *)H5UC_GET_OBJ(bt->rc_shared);
    assert(shared);
    if (NULL == (bt->native = H5FL_BLK_MALLOC(native_block, shared->sizeof_keys)) ||
        NULL == (bt->child = H5FL_SEQ_MALLOC(haddr_t, (size_t)shared->two_k)))
        HGOTO_ERROR(H5E_BTREE, H5E_CANTALLOC, FAIL, "memory allocation failed for B-tree root node");
    if (HADDR_UNDEF == (*addr_p = H5MF_alloc(f, H5FD_MEM_BTREE, (hsize_t)shared->sizeof_rnode)))
        HGOTO_ERROR(H5E_BTREE, H5E_CANTALLOC, FAIL, "file allocation failed for B-tree root node");

    
    if (H5AC_insert_entry(f, H5AC_BT, *addr_p, bt, H5AC__NO_FLAGS_SET) < 0)
        HGOTO_ERROR(H5E_BTREE, H5E_CANTINS, FAIL, "can't add B-tree root node to cache");

done:
    if (ret_value < 0) {
        if (shared && shared->sizeof_rnode > 0) {
            H5_CHECK_OVERFLOW(shared->sizeof_rnode, size_t, hsize_t);
            (void)H5MF_xfree(f, H5FD_MEM_BTREE, *addr_p, (hsize_t)shared->sizeof_rnode);
        } 
        if (bt)
            
            if (H5B__node_dest(bt) < 0)
                HDONE_ERROR(H5E_BTREE, H5E_CANTRELEASE, FAIL, "unable to destroy B-tree node");
    } 

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5B_find(H5F_t *f, const H5B_class_t *type, haddr_t addr, bool *found, void *udata)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f);
    assert(type);
    assert(type->decode);
    assert(type->cmp3);
    assert(type->found);
    assert(H5_addr_defined(addr));

    if ((ret_value = H5B_find_helper(f, type, addr, H5B_UNKNOWN_NODELEVEL, found, udata)) < 0)
        HGOTO_ERROR(H5E_BTREE, H5E_NOTFOUND, FAIL, "can't lookup key");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5B_find_helper(H5F_t *f, const H5B_class_t *type, haddr_t addr, int exp_level, bool *found, void *udata)
{
    H5B_t         *bt = NULL;
    H5UC_t        *rc_shared;           
    H5B_shared_t  *shared;              
    H5B_cache_ud_t cache_udata;         
    unsigned       idx = 0, lt = 0, rt; 
    int            cmp       = 1;       
    herr_t         ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f);
    assert(type);
    assert(type->decode);
    assert(type->cmp3);
    assert(type->found);
    assert(H5_addr_defined(addr));

    
    if (NULL == (rc_shared = (type->get_shared)(f, udata)))
        HGOTO_ERROR(H5E_BTREE, H5E_CANTGET, FAIL, "can't retrieve B-tree's shared ref. count object");
    shared = (H5B_shared_t *)H5UC_GET_OBJ(rc_shared);
    assert(shared);

    
    cache_udata.f         = f;
    cache_udata.type      = type;
    cache_udata.rc_shared = rc_shared;
    cache_udata.exp_level = exp_level;
    if (NULL == (bt = (H5B_t *)H5AC_protect(f, H5AC_BT, addr, &cache_udata, H5AC__READ_ONLY_FLAG)))
        HGOTO_ERROR(H5E_BTREE, H5E_CANTPROTECT, FAIL, "unable to load B-tree node");

    rt = bt->nchildren;
    while (lt < rt && cmp) {
        idx = (lt + rt) / 2;
        
        if ((cmp = (type->cmp3)(H5B_NKEY(bt, shared, idx), udata, H5B_NKEY(bt, shared, (idx + 1)))) < 0)
            rt = idx;
        else
            lt = idx + 1;
    } 

    
    if (cmp)
        *found = false;
    else {
        
        assert(idx < bt->nchildren);

        if (bt->level > 0) {
            
            if (bt->child[idx] == addr)
                HGOTO_ERROR(H5E_BTREE, H5E_BADVALUE, FAIL, "cyclic B-tree detected");

            if ((ret_value = H5B_find_helper(f, type, bt->child[idx], (int)(bt->level - 1), found, udata)) <
                0)
                HGOTO_ERROR(H5E_BTREE, H5E_NOTFOUND, FAIL, "can't lookup key in subtree");
        } 
        else {
            if ((ret_value = (type->found)(f, bt->child[idx], H5B_NKEY(bt, shared, idx), found, udata)) < 0)
                HGOTO_ERROR(H5E_BTREE, H5E_NOTFOUND, FAIL, "can't lookup key in leaf node");
        } 
    }     

done:
    if (bt && H5AC_unprotect(f, H5AC_BT, addr, bt, H5AC__NO_FLAGS_SET) < 0)
        HDONE_ERROR(H5E_BTREE, H5E_CANTUNPROTECT, FAIL, "unable to release node");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5B__split(H5F_t *f, H5B_ins_ud_t *bt_ud, unsigned idx, void *udata, H5B_ins_ud_t *split_bt_ud )
{
    H5B_shared_t  *shared;              
    H5B_cache_ud_t cache_udata;         
    unsigned       nleft, nright;       
    double         split_ratios[3];     
    herr_t         ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(bt_ud);
    assert(bt_ud->bt);
    assert(H5_addr_defined(bt_ud->addr));
    assert(split_bt_ud);
    assert(!split_bt_ud->bt);

    
    shared = (H5B_shared_t *)H5UC_GET_OBJ(bt_ud->bt->rc_shared);
    assert(shared);
    assert(bt_ud->bt->nchildren == shared->two_k);

    
    if (H5CX_get_btree_split_ratios(split_ratios) < 0)
        HGOTO_ERROR(H5E_BTREE, H5E_CANTGET, FAIL, "can't retrieve B-tree split ratios");

    
    if (!H5_addr_defined(bt_ud->bt->right))
        nleft = (unsigned)((double)shared->two_k * split_ratios[2]); 
    else if (!H5_addr_defined(bt_ud->bt->left))
        nleft = (unsigned)((double)shared->two_k * split_ratios[0]); 
    else
        nleft = (unsigned)((double)shared->two_k * split_ratios[1]); 

    
    if (idx < nleft && nleft == shared->two_k)
        --nleft;
    else if (idx >= nleft && 0 == nleft)
        nleft++;
    nright = shared->two_k - nleft;

    
    if (H5B_create(f, shared->type, udata, &split_bt_ud->addr ) < 0)
        HGOTO_ERROR(H5E_BTREE, H5E_CANTINIT, FAIL, "unable to create B-tree");
    cache_udata.f         = f;
    cache_udata.type      = shared->type;
    cache_udata.rc_shared = bt_ud->bt->rc_shared;
    cache_udata.exp_level = H5B_UNKNOWN_NODELEVEL;
    if (NULL == (split_bt_ud->bt =
                     (H5B_t *)H5AC_protect(f, H5AC_BT, split_bt_ud->addr, &cache_udata, H5AC__NO_FLAGS_SET)))
        HGOTO_ERROR(H5E_BTREE, H5E_CANTPROTECT, FAIL, "unable to protect B-tree");
    split_bt_ud->bt->level = bt_ud->bt->level;

    

    split_bt_ud->cache_flags = H5AC__DIRTIED_FLAG;
    H5MM_memcpy(split_bt_ud->bt->native, bt_ud->bt->native + nleft * shared->type->sizeof_nkey,
                (nright + 1) * shared->type->sizeof_nkey);
    H5MM_memcpy(split_bt_ud->bt->child, &bt_ud->bt->child[nleft], nright * sizeof(haddr_t));

    split_bt_ud->bt->nchildren = nright;

    
    bt_ud->cache_flags |= H5AC__DIRTIED_FLAG;
    bt_ud->bt->nchildren = nleft;

    
    split_bt_ud->bt->left  = bt_ud->addr;
    split_bt_ud->bt->right = bt_ud->bt->right;

    if (H5_addr_defined(bt_ud->bt->right)) {
        H5B_t *tmp_bt;

        if (NULL ==
            (tmp_bt = (H5B_t *)H5AC_protect(f, H5AC_BT, bt_ud->bt->right, &cache_udata, H5AC__NO_FLAGS_SET)))
            HGOTO_ERROR(H5E_BTREE, H5E_CANTPROTECT, FAIL, "unable to load right sibling");

        tmp_bt->left = split_bt_ud->addr;

        if (H5AC_unprotect(f, H5AC_BT, bt_ud->bt->right, tmp_bt, H5AC__DIRTIED_FLAG) < 0)
            HGOTO_ERROR(H5E_BTREE, H5E_CANTUNPROTECT, FAIL, "unable to release B-tree node");
    } 

    bt_ud->bt->right = split_bt_ud->addr;
    assert(bt_ud->cache_flags & H5AC__DIRTIED_FLAG);

done:
    if (ret_value < 0) {
        if (split_bt_ud->bt &&
            H5AC_unprotect(f, H5AC_BT, split_bt_ud->addr, split_bt_ud->bt, split_bt_ud->cache_flags) < 0)
            HDONE_ERROR(H5E_BTREE, H5E_CANTUNPROTECT, FAIL, "unable to release B-tree node");
        split_bt_ud->bt          = NULL;
        split_bt_ud->addr        = HADDR_UNDEF;
        split_bt_ud->cache_flags = H5AC__NO_FLAGS_SET;
    } 

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5B_insert(H5F_t *f, const H5B_class_t *type, haddr_t addr, void *udata)
{
    
    uint64_t _lt_key[128], _md_key[128], _rt_key[128];
    uint8_t *lt_key = (uint8_t *)_lt_key;
    uint8_t *md_key = (uint8_t *)_md_key;
    uint8_t *rt_key = (uint8_t *)_rt_key;

    bool           lt_key_changed = false, rt_key_changed = false;
    haddr_t        old_root_addr = HADDR_UNDEF;
    unsigned       level;
    H5B_ins_ud_t   bt_ud       = H5B_INS_UD_T_NULL; 
    H5B_ins_ud_t   split_bt_ud = H5B_INS_UD_T_NULL; 
    H5B_t         *new_root_bt = NULL;              
    H5UC_t        *rc_shared;                       
    H5B_shared_t  *shared;                          
    H5B_cache_ud_t cache_udata;                     
    H5B_ins_t      my_ins    = H5B_INS_ERROR;
    herr_t         ret_value = SUCCEED;

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f);
    assert(type);
    assert(type->sizeof_nkey <= sizeof _lt_key);
    assert(H5_addr_defined(addr));

    
    if (NULL == (rc_shared = (type->get_shared)(f, udata)))
        HGOTO_ERROR(H5E_BTREE, H5E_CANTGET, FAIL, "can't retrieve B-tree's shared ref. count object");
    shared = (H5B_shared_t *)H5UC_GET_OBJ(rc_shared);
    assert(shared);

    
    cache_udata.f         = f;
    cache_udata.type      = type;
    cache_udata.rc_shared = rc_shared;
    cache_udata.exp_level = H5B_UNKNOWN_NODELEVEL;
    bt_ud.addr            = addr;
    if (NULL == (bt_ud.bt = (H5B_t *)H5AC_protect(f, H5AC_BT, addr, &cache_udata, H5AC__NO_FLAGS_SET)))
        HGOTO_ERROR(H5E_BTREE, H5E_CANTPROTECT, FAIL, "unable to locate root of B-tree");

    
    if ((int)(my_ins = H5B__insert_helper(f, &bt_ud, type, lt_key, &lt_key_changed, md_key, udata, rt_key,
                                          &rt_key_changed, &split_bt_ud )) < 0)
        HGOTO_ERROR(H5E_BTREE, H5E_CANTINSERT, FAIL, "unable to insert key");

    
    if (H5B_INS_NOOP == my_ins) {
        
        assert(!split_bt_ud.bt);
        HGOTO_DONE(SUCCEED);
    } 
    assert(H5B_INS_RIGHT == my_ins);
    assert(split_bt_ud.bt);
    assert(H5_addr_defined(split_bt_ud.addr));

    
    level = bt_ud.bt->level;

    
    if (!lt_key_changed)
        H5MM_memcpy(lt_key, H5B_NKEY(bt_ud.bt, shared, 0), type->sizeof_nkey);
    if (!rt_key_changed)
        H5MM_memcpy(rt_key, H5B_NKEY(split_bt_ud.bt, shared, split_bt_ud.bt->nchildren), type->sizeof_nkey);

    
    H5_CHECK_OVERFLOW(shared->sizeof_rnode, size_t, hsize_t);
    if (HADDR_UNDEF == (old_root_addr = H5MF_alloc(f, H5FD_MEM_BTREE, (hsize_t)shared->sizeof_rnode)))
        HGOTO_ERROR(H5E_BTREE, H5E_CANTALLOC, FAIL, "unable to allocate file space to move root");

    

    
    if (NULL == (new_root_bt = H5B__copy(bt_ud.bt)))
        HGOTO_ERROR(H5E_BTREE, H5E_CANTCOPY, FAIL, "unable to copy old root");

    
    if (H5AC_unprotect(f, H5AC_BT, bt_ud.addr, bt_ud.bt, H5AC__DIRTIED_FLAG) < 0)
        HGOTO_ERROR(H5E_BTREE, H5E_CANTUNPROTECT, FAIL, "unable to release old root");
    bt_ud.bt = NULL; 

    
    if (H5AC_move_entry(f, H5AC_BT, bt_ud.addr, old_root_addr) < 0)
        HGOTO_ERROR(H5E_BTREE, H5E_CANTMOVE, FAIL, "unable to move B-tree root node");
    bt_ud.addr = old_root_addr;

    
    split_bt_ud.bt->left = bt_ud.addr;
    split_bt_ud.cache_flags |= H5AC__DIRTIED_FLAG;

    
    new_root_bt->left  = HADDR_UNDEF;
    new_root_bt->right = HADDR_UNDEF;

    
    new_root_bt->level     = level + 1;
    new_root_bt->nchildren = 2;

    new_root_bt->child[0] = bt_ud.addr;
    H5MM_memcpy(H5B_NKEY(new_root_bt, shared, 0), lt_key, shared->type->sizeof_nkey);

    new_root_bt->child[1] = split_bt_ud.addr;
    H5MM_memcpy(H5B_NKEY(new_root_bt, shared, 1), md_key, shared->type->sizeof_nkey);
    H5MM_memcpy(H5B_NKEY(new_root_bt, shared, 2), rt_key, shared->type->sizeof_nkey);

    
    if (H5AC_insert_entry(f, H5AC_BT, addr, new_root_bt, H5AC__NO_FLAGS_SET) < 0)
        HGOTO_ERROR(H5E_BTREE, H5E_CANTINS, FAIL, "unable to add old B-tree root node to cache");

done:
    if (ret_value < 0)
        if (new_root_bt && H5B__node_dest(new_root_bt) < 0)
            HDONE_ERROR(H5E_BTREE, H5E_CANTRELEASE, FAIL, "unable to free B-tree root node");

    if (bt_ud.bt)
        if (H5AC_unprotect(f, H5AC_BT, bt_ud.addr, bt_ud.bt, bt_ud.cache_flags) < 0)
            HDONE_ERROR(H5E_BTREE, H5E_CANTUNPROTECT, FAIL, "unable to unprotect old root");

    if (split_bt_ud.bt)
        if (H5AC_unprotect(f, H5AC_BT, split_bt_ud.addr, split_bt_ud.bt, split_bt_ud.cache_flags) < 0)
            HDONE_ERROR(H5E_BTREE, H5E_CANTUNPROTECT, FAIL, "unable to unprotect new child");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5B__insert_child(H5B_t *bt, unsigned *bt_flags, unsigned idx, haddr_t child, H5B_ins_t anchor,
                  const void *md_key)
{
    H5B_shared_t *shared; 
    uint8_t      *base;   

    FUNC_ENTER_PACKAGE_NOERR

    assert(bt);
    assert(bt_flags);
    assert(H5_addr_defined(child));
    shared = (H5B_shared_t *)H5UC_GET_OBJ(bt->rc_shared);
    assert(shared);
    assert(bt->nchildren < shared->two_k);

    
    base = H5B_NKEY(bt, shared, (idx + 1));
    if ((idx + 1) == bt->nchildren) {
        
        H5MM_memcpy(base + shared->type->sizeof_nkey, base,
                    shared->type->sizeof_nkey); 
        H5MM_memcpy(base, md_key, shared->type->sizeof_nkey);

        
        if (H5B_INS_RIGHT == anchor)
            idx++; 
        else
            
            bt->child[idx + 1] = bt->child[idx];
    } 
    else {
        
        memmove(base + shared->type->sizeof_nkey, base, (bt->nchildren - idx) * shared->type->sizeof_nkey);
        H5MM_memcpy(base, md_key, shared->type->sizeof_nkey);

        
        if (H5B_INS_RIGHT == anchor)
            idx++;

        
        memmove(bt->child + idx + 1, bt->child + idx, (bt->nchildren - idx) * sizeof(haddr_t));
    } 

    bt->child[idx] = child;
    bt->nchildren += 1;

    
    *bt_flags |= H5AC__DIRTIED_FLAG;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static H5B_ins_t
H5B__insert_helper(H5F_t *f, H5B_ins_ud_t *bt_ud, const H5B_class_t *type, uint8_t *lt_key,
                   bool *lt_key_changed, uint8_t *md_key, void *udata, uint8_t *rt_key, bool *rt_key_changed,
                   H5B_ins_ud_t *split_bt_ud )
{
    H5B_t         *bt;                                  
    H5UC_t        *rc_shared;                           
    H5B_shared_t  *shared;                              
    H5B_cache_ud_t cache_udata;                         
    unsigned       lt = 0, idx = 0, rt;                 
    int            cmp             = -1;                
    H5B_ins_ud_t   child_bt_ud     = H5B_INS_UD_T_NULL; 
    H5B_ins_ud_t   new_child_bt_ud = H5B_INS_UD_T_NULL; 
    H5B_ins_t      my_ins          = H5B_INS_ERROR;
    H5B_ins_t      ret_value       = H5B_INS_ERROR; 

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(bt_ud);
    assert(bt_ud->bt);
    assert(H5_addr_defined(bt_ud->addr));
    assert(type);
    assert(type->decode);
    assert(type->cmp3);
    assert(type->new_node);
    assert(lt_key);
    assert(lt_key_changed);
    assert(rt_key);
    assert(rt_key_changed);
    assert(split_bt_ud);
    assert(!split_bt_ud->bt);
    assert(!H5_addr_defined(split_bt_ud->addr));
    assert(split_bt_ud->cache_flags == H5AC__NO_FLAGS_SET);

    bt = bt_ud->bt;

    *lt_key_changed = false;
    *rt_key_changed = false;

    
    if (NULL == (rc_shared = (type->get_shared)(f, udata)))
        HGOTO_ERROR(H5E_BTREE, H5E_CANTGET, H5B_INS_ERROR,
                    "can't retrieve B-tree's shared ref. count object");
    shared = (H5B_shared_t *)H5UC_GET_OBJ(rc_shared);
    assert(shared);

    
    rt = bt->nchildren;

    while (lt < rt && cmp) {
        idx = (lt + rt) / 2;
        if ((cmp = (type->cmp3)(H5B_NKEY(bt, shared, idx), udata, H5B_NKEY(bt, shared, idx + 1))) < 0)
            rt = idx;
        else
            lt = idx + 1;
    } 

    
    cache_udata.f         = f;
    cache_udata.type      = type;
    cache_udata.rc_shared = rc_shared;
    cache_udata.exp_level = H5B_UNKNOWN_NODELEVEL;

    if (0 == bt->nchildren) {
        
        assert(0 == bt->level);
        if ((type->new_node)(f, H5B_INS_FIRST, H5B_NKEY(bt, shared, 0), udata, H5B_NKEY(bt, shared, 1),
                             bt->child + 0 ) < 0)
            HGOTO_ERROR(H5E_BTREE, H5E_CANTINIT, H5B_INS_ERROR, "unable to create leaf node");
        bt->nchildren = 1;
        bt_ud->cache_flags |= H5AC__DIRTIED_FLAG;
        idx = 0;

        if (type->follow_min) {
            if ((int)(my_ins = (type->insert)(f, bt->child[idx], H5B_NKEY(bt, shared, idx), lt_key_changed,
                                              md_key, udata, H5B_NKEY(bt, shared, idx + 1), rt_key_changed,
                                              &new_child_bt_ud.addr )) < 0)
                HGOTO_ERROR(H5E_BTREE, H5E_CANTINSERT, H5B_INS_ERROR, "unable to insert first leaf node");
        } 
        else
            my_ins = H5B_INS_NOOP;
    }
    else if (cmp < 0 && idx == 0) {
        if (bt->level > 0) {
            
            child_bt_ud.addr = bt->child[idx];
            if (NULL == (child_bt_ud.bt = (H5B_t *)H5AC_protect(f, H5AC_BT, child_bt_ud.addr, &cache_udata,
                                                                H5AC__NO_FLAGS_SET)))
                HGOTO_ERROR(H5E_BTREE, H5E_CANTPROTECT, H5B_INS_ERROR, "unable to load node");

            if ((int)(my_ins = H5B__insert_helper(
                          f, &child_bt_ud, type, H5B_NKEY(bt, shared, idx), lt_key_changed, md_key, udata,
                          H5B_NKEY(bt, shared, idx + 1), rt_key_changed, &new_child_bt_ud )) < 0)
                HGOTO_ERROR(H5E_BTREE, H5E_CANTINSERT, H5B_INS_ERROR, "can't insert minimum subtree");
        }
        else if (type->follow_min) {
            
            if ((int)(my_ins = (type->insert)(f, bt->child[idx], H5B_NKEY(bt, shared, idx), lt_key_changed,
                                              md_key, udata, H5B_NKEY(bt, shared, idx + 1), rt_key_changed,
                                              &new_child_bt_ud.addr )) < 0)
                HGOTO_ERROR(H5E_BTREE, H5E_CANTINSERT, H5B_INS_ERROR, "can't insert minimum leaf node");
        }
        else {
            
            my_ins = H5B_INS_LEFT;
            H5MM_memcpy(md_key, H5B_NKEY(bt, shared, idx), type->sizeof_nkey);
            if ((type->new_node)(f, H5B_INS_LEFT, H5B_NKEY(bt, shared, idx), udata, md_key,
                                 &new_child_bt_ud.addr ) < 0)
                HGOTO_ERROR(H5E_BTREE, H5E_CANTINSERT, H5B_INS_ERROR, "can't insert minimum leaf node");
            *lt_key_changed = true;
        } 

#ifdef H5_STRICT_FORMAT_CHECKS
        
        if (H5_addr_defined(bt->left))
            HGOTO_ERROR(H5E_BTREE, H5E_BADVALUE, H5B_INS_ERROR, "internal error: likely corrupt key values");
#endif 
    }
    else if (cmp > 0 && idx + 1 >= bt->nchildren) {
        if (bt->level > 0) {
            
            idx              = bt->nchildren - 1;
            child_bt_ud.addr = bt->child[idx];
            if (NULL == (child_bt_ud.bt = (H5B_t *)H5AC_protect(f, H5AC_BT, child_bt_ud.addr, &cache_udata,
                                                                H5AC__NO_FLAGS_SET)))
                HGOTO_ERROR(H5E_BTREE, H5E_CANTPROTECT, H5B_INS_ERROR, "unable to load node");

            if ((int)(my_ins = H5B__insert_helper(
                          f, &child_bt_ud, type, H5B_NKEY(bt, shared, idx), lt_key_changed, md_key, udata,
                          H5B_NKEY(bt, shared, idx + 1), rt_key_changed, &new_child_bt_ud )) < 0)
                HGOTO_ERROR(H5E_BTREE, H5E_CANTINSERT, H5B_INS_ERROR, "can't insert maximum subtree");
        }
        else if (type->follow_max) {
            
            idx = bt->nchildren - 1;
            if ((int)(my_ins = (type->insert)(f, bt->child[idx], H5B_NKEY(bt, shared, idx), lt_key_changed,
                                              md_key, udata, H5B_NKEY(bt, shared, idx + 1), rt_key_changed,
                                              &new_child_bt_ud.addr )) < 0)
                HGOTO_ERROR(H5E_BTREE, H5E_CANTINSERT, H5B_INS_ERROR, "can't insert maximum leaf node");
        }
        else {
            
            idx    = bt->nchildren - 1;
            my_ins = H5B_INS_RIGHT;
            H5MM_memcpy(md_key, H5B_NKEY(bt, shared, idx + 1), type->sizeof_nkey);
            if ((type->new_node)(f, H5B_INS_RIGHT, md_key, udata, H5B_NKEY(bt, shared, idx + 1),
                                 &new_child_bt_ud.addr ) < 0)
                HGOTO_ERROR(H5E_BTREE, H5E_CANTINSERT, H5B_INS_ERROR, "can't insert maximum leaf node");
            *rt_key_changed = true;
        } 

#ifdef H5_STRICT_FORMAT_CHECKS
        
        if (H5_addr_defined(bt->right))
            HGOTO_ERROR(H5E_BTREE, H5E_BADVALUE, H5B_INS_ERROR, "internal error: likely corrupt key values");
#endif 
    }
    else if (cmp) {
        
        HGOTO_ERROR(H5E_BTREE, H5E_CANTINSERT, H5B_INS_ERROR,
                    "internal error: could not determine which branch to follow out of this node");
    }
    else if (bt->level > 0) {
        
        assert(idx < bt->nchildren);
        child_bt_ud.addr = bt->child[idx];
        if (NULL == (child_bt_ud.bt = (H5B_t *)H5AC_protect(f, H5AC_BT, child_bt_ud.addr, &cache_udata,
                                                            H5AC__NO_FLAGS_SET)))
            HGOTO_ERROR(H5E_BTREE, H5E_CANTPROTECT, H5B_INS_ERROR, "unable to load node");

        if ((int)(my_ins = H5B__insert_helper(f, &child_bt_ud, type, H5B_NKEY(bt, shared, idx),
                                              lt_key_changed, md_key, udata, H5B_NKEY(bt, shared, idx + 1),
                                              rt_key_changed, &new_child_bt_ud )) < 0)
            HGOTO_ERROR(H5E_BTREE, H5E_CANTINSERT, H5B_INS_ERROR, "can't insert subtree");
    }
    else {
        
        assert(idx < bt->nchildren);
        if ((int)(my_ins = (type->insert)(f, bt->child[idx], H5B_NKEY(bt, shared, idx), lt_key_changed,
                                          md_key, udata, H5B_NKEY(bt, shared, idx + 1), rt_key_changed,
                                          &new_child_bt_ud.addr )) < 0)
            HGOTO_ERROR(H5E_BTREE, H5E_CANTINSERT, H5B_INS_ERROR, "can't insert leaf node");
    }
    assert((int)my_ins >= 0);

    
    if (*lt_key_changed) {
        bt_ud->cache_flags |= H5AC__DIRTIED_FLAG;
        if (idx > 0) {
            assert(type->critical_key == H5B_LEFT);
            assert(!(H5B_INS_LEFT == my_ins || H5B_INS_RIGHT == my_ins));
            *lt_key_changed = false;
        } 
        else
            H5MM_memcpy(lt_key, H5B_NKEY(bt, shared, idx), type->sizeof_nkey);
    } 
    if (*rt_key_changed) {
        bt_ud->cache_flags |= H5AC__DIRTIED_FLAG;
        if (idx + 1 < bt->nchildren) {
            assert(type->critical_key == H5B_RIGHT);
            assert(!(H5B_INS_LEFT == my_ins || H5B_INS_RIGHT == my_ins));
            *rt_key_changed = false;
        } 
        else
            H5MM_memcpy(rt_key, H5B_NKEY(bt, shared, idx + 1), type->sizeof_nkey);
    } 

    
    assert(!(bt->level == 0) != !(child_bt_ud.bt));
    if (H5B_INS_CHANGE == my_ins) {
        
        assert(!child_bt_ud.bt);
        assert(bt->level == 0);
        bt->child[idx] = new_child_bt_ud.addr;
        bt_ud->cache_flags |= H5AC__DIRTIED_FLAG;
    }
    else if (H5B_INS_LEFT == my_ins || H5B_INS_RIGHT == my_ins) {
        unsigned *tmp_bt_flags_ptr = NULL;
        H5B_t    *tmp_bt;

        
        if (bt->nchildren == shared->two_k) {
            if (H5B__split(f, bt_ud, idx, udata, split_bt_ud ) < 0)
                HGOTO_ERROR(H5E_BTREE, H5E_CANTSPLIT, H5B_INS_ERROR, "unable to split node");
            if (idx < bt->nchildren) {
                tmp_bt           = bt;
                tmp_bt_flags_ptr = &bt_ud->cache_flags;
            }
            else {
                idx -= bt->nchildren;
                tmp_bt           = split_bt_ud->bt;
                tmp_bt_flags_ptr = &split_bt_ud->cache_flags;
            }
        } 
        else {
            tmp_bt           = bt;
            tmp_bt_flags_ptr = &bt_ud->cache_flags;
        } 

        
        if (H5B__insert_child(tmp_bt, tmp_bt_flags_ptr, idx, new_child_bt_ud.addr, my_ins, md_key) < 0)
            HGOTO_ERROR(H5E_BTREE, H5E_CANTINSERT, H5B_INS_ERROR, "can't insert child");
    } 

    
    if (split_bt_ud->bt) {
        H5MM_memcpy(md_key, H5B_NKEY(split_bt_ud->bt, shared, 0), type->sizeof_nkey);
        ret_value = H5B_INS_RIGHT;
    }
    else
        ret_value = H5B_INS_NOOP;

done:
    if (child_bt_ud.bt)
        if (H5AC_unprotect(f, H5AC_BT, child_bt_ud.addr, child_bt_ud.bt, child_bt_ud.cache_flags) < 0)
            HDONE_ERROR(H5E_BTREE, H5E_CANTUNPROTECT, H5B_INS_ERROR, "unable to unprotect child");

    if (new_child_bt_ud.bt)
        if (H5AC_unprotect(f, H5AC_BT, new_child_bt_ud.addr, new_child_bt_ud.bt,
                           new_child_bt_ud.cache_flags) < 0)
            HDONE_ERROR(H5E_BTREE, H5E_CANTUNPROTECT, H5B_INS_ERROR, "unable to unprotect new child");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5B__iterate_helper(H5F_t *f, const H5B_class_t *type, haddr_t addr, int exp_level, H5B_operator_t op,
                    void *udata)
{
    H5B_t         *bt = NULL;                
    H5UC_t        *rc_shared;                
    H5B_shared_t  *shared;                   
    H5B_cache_ud_t cache_udata;              
    unsigned       u;                        
    herr_t         ret_value = H5_ITER_CONT; 

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(type);
    assert(H5_addr_defined(addr));
    assert(op);
    assert(udata);

    
    if (NULL == (rc_shared = (type->get_shared)(f, udata)))
        HGOTO_ERROR(H5E_BTREE, H5E_CANTGET, FAIL, "can't retrieve B-tree's shared ref. count object");
    shared = (H5B_shared_t *)H5UC_GET_OBJ(rc_shared);
    assert(shared);

    
    cache_udata.f         = f;
    cache_udata.type      = type;
    cache_udata.rc_shared = rc_shared;
    cache_udata.exp_level = exp_level;
    if (NULL == (bt = (H5B_t *)H5AC_protect(f, H5AC_BT, addr, &cache_udata, H5AC__READ_ONLY_FLAG)))
        HGOTO_ERROR(H5E_BTREE, H5E_CANTPROTECT, H5_ITER_ERROR, "unable to load B-tree node");

    
    for (u = 0; u < bt->nchildren && ret_value == H5_ITER_CONT; u++) {
        if (bt->level > 0)
            ret_value = H5B__iterate_helper(f, type, bt->child[u], (int)(bt->level - 1), op, udata);
        else
            ret_value = (*op)(f, H5B_NKEY(bt, shared, u), bt->child[u], H5B_NKEY(bt, shared, u + 1), udata);
        if (ret_value < 0)
            HERROR(H5E_BTREE, H5E_BADITER, "B-tree iteration failed");
    } 

done:
    if (bt && H5AC_unprotect(f, H5AC_BT, addr, bt, H5AC__NO_FLAGS_SET) < 0)
        HDONE_ERROR(H5E_BTREE, H5E_CANTUNPROTECT, H5_ITER_ERROR, "unable to release B-tree node");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5B_iterate(H5F_t *f, const H5B_class_t *type, haddr_t addr, H5B_operator_t op, void *udata)
{
    herr_t ret_value = FAIL; 

    FUNC_ENTER_NOAPI_NOERR

    
    assert(f);
    assert(type);
    assert(H5_addr_defined(addr));
    assert(op);
    assert(udata);

    
    if ((ret_value = H5B__iterate_helper(f, type, addr, H5B_UNKNOWN_NODELEVEL, op, udata)) < 0)
        HERROR(H5E_BTREE, H5E_BADITER, "B-tree iteration failed");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static H5B_ins_t
H5B__remove_helper(H5F_t *f, haddr_t addr, const H5B_class_t *type, int level, uint8_t *lt_key ,
                   bool *lt_key_changed , void *udata, uint8_t *rt_key ,
                   bool *rt_key_changed )
{
    H5B_t         *bt = NULL, *sibling = NULL;
    unsigned       bt_flags = H5AC__NO_FLAGS_SET;
    H5UC_t        *rc_shared;           
    H5B_shared_t  *shared;              
    H5B_cache_ud_t cache_udata;         
    unsigned       idx = 0, lt = 0, rt; 
    int            cmp       = 1;       
    H5B_ins_t      ret_value = H5B_INS_ERROR;

    FUNC_ENTER_PACKAGE

    assert(f);
    assert(H5_addr_defined(addr));
    assert(type);
    assert(type->decode);
    assert(type->cmp3);
    assert(lt_key && lt_key_changed);
    assert(udata);
    assert(rt_key && rt_key_changed);

    
    if (NULL == (rc_shared = (type->get_shared)(f, udata)))
        HGOTO_ERROR(H5E_BTREE, H5E_CANTGET, H5B_INS_ERROR,
                    "can't retrieve B-tree's shared ref. count object");
    shared = (H5B_shared_t *)H5UC_GET_OBJ(rc_shared);
    assert(shared);

    
    cache_udata.f         = f;
    cache_udata.type      = type;
    cache_udata.rc_shared = rc_shared;
    cache_udata.exp_level = H5B_UNKNOWN_NODELEVEL;
    if (NULL == (bt = (H5B_t *)H5AC_protect(f, H5AC_BT, addr, &cache_udata, H5AC__NO_FLAGS_SET)))
        HGOTO_ERROR(H5E_BTREE, H5E_CANTPROTECT, H5B_INS_ERROR, "unable to load B-tree node");

    rt = bt->nchildren;
    while (lt < rt && cmp) {
        idx = (lt + rt) / 2;
        if ((cmp = (type->cmp3)(H5B_NKEY(bt, shared, idx), udata, H5B_NKEY(bt, shared, idx + 1))) < 0)
            rt = idx;
        else
            lt = idx + 1;
    } 
    if (cmp)
        HGOTO_ERROR(H5E_BTREE, H5E_NOTFOUND, H5B_INS_ERROR, "B-tree key not found");

    
    assert(idx < bt->nchildren);
    if (bt->level > 0) {
        
        if ((int)(ret_value =
                      H5B__remove_helper(f, bt->child[idx], type, level + 1,
                                         H5B_NKEY(bt, shared, idx) , lt_key_changed , udata,
                                         H5B_NKEY(bt, shared, idx + 1) , rt_key_changed )) < 0)
            HGOTO_ERROR(H5E_BTREE, H5E_CANTREMOVE, H5B_INS_ERROR, "key not found in subtree");
    }
    else if (type->remove) {
        
        if ((int)(ret_value = (type->remove)(f, bt->child[idx], H5B_NKEY(bt, shared, idx), lt_key_changed,
                                             udata, H5B_NKEY(bt, shared, idx + 1), rt_key_changed)) < 0)
            HGOTO_ERROR(H5E_BTREE, H5E_CANTREMOVE, H5B_INS_ERROR, "key not found in leaf node");
    }
    else {
        
        *lt_key_changed = false;
        *rt_key_changed = false;
        ret_value       = H5B_INS_REMOVE;
    }

    
    if (*lt_key_changed) {
        assert(type->critical_key == H5B_LEFT);
        bt_flags |= H5AC__DIRTIED_FLAG;

        if (idx > 0)
            
            *lt_key_changed = false;
        else
            H5MM_memcpy(lt_key, H5B_NKEY(bt, shared, idx), type->sizeof_nkey);
    } 
    if (*rt_key_changed) {
        assert(type->critical_key == H5B_RIGHT);
        bt_flags |= H5AC__DIRTIED_FLAG;
        if (idx + 1 < bt->nchildren)
            
            *rt_key_changed = false;
        else
            H5MM_memcpy(rt_key, H5B_NKEY(bt, shared, idx + 1), type->sizeof_nkey);
    } 

    
    if (H5B_INS_REMOVE == ret_value) {
        
        assert(!(*lt_key_changed));
        assert(!(*rt_key_changed));

        if (1 == bt->nchildren) {
            
            
            if (level > 0) {
                
                if (H5_addr_defined(bt->left)) {
                    if (NULL == (sibling = (H5B_t *)H5AC_protect(f, H5AC_BT, bt->left, &cache_udata,
                                                                 H5AC__NO_FLAGS_SET)))
                        HGOTO_ERROR(H5E_BTREE, H5E_CANTPROTECT, H5B_INS_ERROR,
                                    "unable to load node from tree");

                    
                    if (type->critical_key == H5B_LEFT)
                        H5MM_memcpy(H5B_NKEY(sibling, shared, sibling->nchildren), H5B_NKEY(bt, shared, 1),
                                    type->sizeof_nkey);

                    sibling->right = bt->right;

                    if (H5AC_unprotect(f, H5AC_BT, bt->left, sibling, H5AC__DIRTIED_FLAG) < 0)
                        HGOTO_ERROR(H5E_BTREE, H5E_CANTUNPROTECT, H5B_INS_ERROR,
                                    "unable to release node from tree");
                    sibling = NULL; 
                }                   
                if (H5_addr_defined(bt->right)) {
                    if (NULL == (sibling = (H5B_t *)H5AC_protect(f, H5AC_BT, bt->right, &cache_udata,
                                                                 H5AC__NO_FLAGS_SET)))
                        HGOTO_ERROR(H5E_BTREE, H5E_CANTPROTECT, H5B_INS_ERROR,
                                    "unable to unlink node from tree");

                    
                    if (type->critical_key == H5B_RIGHT)
                        H5MM_memcpy(H5B_NKEY(sibling, shared, 0), H5B_NKEY(bt, shared, 0), type->sizeof_nkey);

                    sibling->left = bt->left;

                    if (H5AC_unprotect(f, H5AC_BT, bt->right, sibling, H5AC__DIRTIED_FLAG) < 0)
                        HGOTO_ERROR(H5E_BTREE, H5E_CANTUNPROTECT, H5B_INS_ERROR,
                                    "unable to release node from tree");
                    sibling = NULL; 
                }                   

                
                bt->left      = HADDR_UNDEF;
                bt->right     = HADDR_UNDEF;
                bt->nchildren = 0;

                
                bt_flags |= H5AC__DIRTIED_FLAG | H5AC__FREE_FILE_SPACE_FLAG;
                H5_CHECK_OVERFLOW(shared->sizeof_rnode, size_t, hsize_t);
                if (H5AC_unprotect(f, H5AC_BT, addr, bt, bt_flags | H5AC__DELETED_FLAG) < 0) {
                    bt       = NULL;
                    bt_flags = H5AC__NO_FLAGS_SET;
                    HGOTO_ERROR(H5E_BTREE, H5E_CANTUNPROTECT, H5B_INS_ERROR, "unable to free B-tree node");
                } 
                bt       = NULL;
                bt_flags = H5AC__NO_FLAGS_SET;
            }
            else {
                
                bt->nchildren = 0;
                bt->level     = 0;
                bt_flags |= H5AC__DIRTIED_FLAG;
            } 
        }
        else if (0 == idx) {
            
            if (type->critical_key == H5B_LEFT) {
                
                memmove(H5B_NKEY(bt, shared, 0), H5B_NKEY(bt, shared, 1), bt->nchildren * type->sizeof_nkey);
                H5MM_memcpy(lt_key, H5B_NKEY(bt, shared, 0), type->sizeof_nkey);
                *lt_key_changed = true;
            }
            else
                
                memmove(H5B_NKEY(bt, shared, 1), H5B_NKEY(bt, shared, 2),
                        (bt->nchildren - 1) * type->sizeof_nkey);

            memmove(bt->child, bt->child + 1, (bt->nchildren - 1) * sizeof(haddr_t));

            bt->nchildren -= 1;
            bt_flags |= H5AC__DIRTIED_FLAG;
            ret_value = H5B_INS_NOOP;
        }
        else if (idx + 1 == bt->nchildren) {
            
            if (type->critical_key == H5B_LEFT)
                
                memmove(H5B_NKEY(bt, shared, bt->nchildren - 1), H5B_NKEY(bt, shared, bt->nchildren),
                        type->sizeof_nkey);
            else {
                
                H5MM_memcpy(rt_key, H5B_NKEY(bt, shared, bt->nchildren - 1), type->sizeof_nkey);
                *rt_key_changed = true;
            } 

            bt->nchildren -= 1;
            bt_flags |= H5AC__DIRTIED_FLAG;
            ret_value = H5B_INS_NOOP;
        }
        else {
            
            if (type->critical_key == H5B_LEFT)
                memmove(H5B_NKEY(bt, shared, idx), H5B_NKEY(bt, shared, idx + 1),
                        (bt->nchildren - idx) * type->sizeof_nkey);
            else
                memmove(H5B_NKEY(bt, shared, idx + 1), H5B_NKEY(bt, shared, idx + 2),
                        (bt->nchildren - 1 - idx) * type->sizeof_nkey);

            memmove(bt->child + idx, bt->child + idx + 1, (bt->nchildren - 1 - idx) * sizeof(haddr_t));

            bt->nchildren -= 1;
            bt_flags |= H5AC__DIRTIED_FLAG;
            ret_value = H5B_INS_NOOP;
        } 
    }
    else 
        ret_value = H5B_INS_NOOP;

    
    if (*lt_key_changed && bt != NULL && H5_addr_defined(bt->left)) {
        assert(type->critical_key == H5B_LEFT);
        assert(level > 0);

        
        if (NULL == (sibling = (H5B_t *)H5AC_protect(f, H5AC_BT, bt->left, &cache_udata, H5AC__NO_FLAGS_SET)))
            HGOTO_ERROR(H5E_BTREE, H5E_CANTPROTECT, H5B_INS_ERROR, "unable to protect node");

        H5MM_memcpy(H5B_NKEY(sibling, shared, sibling->nchildren), H5B_NKEY(bt, shared, 0),
                    type->sizeof_nkey);

        if (H5AC_unprotect(f, H5AC_BT, bt->left, sibling, H5AC__DIRTIED_FLAG) < 0)
            HGOTO_ERROR(H5E_BTREE, H5E_CANTUNPROTECT, H5B_INS_ERROR, "unable to release node from tree");
        sibling = NULL; 
    }                   
    else if (*rt_key_changed && bt != NULL && H5_addr_defined(bt->right)) {
        assert(type->critical_key == H5B_RIGHT);
        assert(level > 0);

        
        if (NULL ==
            (sibling = (H5B_t *)H5AC_protect(f, H5AC_BT, bt->right, &cache_udata, H5AC__NO_FLAGS_SET)))
            HGOTO_ERROR(H5E_BTREE, H5E_CANTPROTECT, H5B_INS_ERROR, "unable to protect node");

        H5MM_memcpy(H5B_NKEY(sibling, shared, 0), H5B_NKEY(bt, shared, bt->nchildren), type->sizeof_nkey);

        if (H5AC_unprotect(f, H5AC_BT, bt->right, sibling, H5AC__DIRTIED_FLAG) < 0)
            HGOTO_ERROR(H5E_BTREE, H5E_CANTUNPROTECT, H5B_INS_ERROR, "unable to release node from tree");
        sibling = NULL; 
    }                   

done:
    if (bt && H5AC_unprotect(f, H5AC_BT, addr, bt, bt_flags) < 0)
        HDONE_ERROR(H5E_BTREE, H5E_CANTUNPROTECT, H5B_INS_ERROR, "unable to release node");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5B_remove(H5F_t *f, const H5B_class_t *type, haddr_t addr, void *udata)
{
    
    uint64_t _lt_key[128], _rt_key[128];
    uint8_t *lt_key         = (uint8_t *)_lt_key; 
    uint8_t *rt_key         = (uint8_t *)_rt_key; 
    bool     lt_key_changed = false;              
    bool     rt_key_changed = false;              
    herr_t   ret_value      = SUCCEED;            

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f);
    assert(type);
    assert(type->sizeof_nkey <= sizeof _lt_key);
    assert(H5_addr_defined(addr));

    
    if (H5B_INS_ERROR ==
        H5B__remove_helper(f, addr, type, 0, lt_key, &lt_key_changed, udata, rt_key, &rt_key_changed))
        HGOTO_ERROR(H5E_BTREE, H5E_CANTREMOVE, FAIL, "unable to remove entry from B-tree");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5B_delete(H5F_t *f, const H5B_class_t *type, haddr_t addr, void *udata)
{
    H5B_t         *bt = NULL;           
    H5UC_t        *rc_shared;           
    H5B_shared_t  *shared;              
    H5B_cache_ud_t cache_udata;         
    unsigned       u;                   
    herr_t         ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f);
    assert(type);
    assert(H5_addr_defined(addr));

    
    if (NULL == (rc_shared = (type->get_shared)(f, udata)))
        HGOTO_ERROR(H5E_BTREE, H5E_CANTGET, FAIL, "can't retrieve B-tree's shared ref. count object");
    shared = (H5B_shared_t *)H5UC_GET_OBJ(rc_shared);
    assert(shared);

    
    cache_udata.f         = f;
    cache_udata.type      = type;
    cache_udata.rc_shared = rc_shared;
    cache_udata.exp_level = H5B_UNKNOWN_NODELEVEL;
    if (NULL == (bt = (H5B_t *)H5AC_protect(f, H5AC_BT, addr, &cache_udata, H5AC__NO_FLAGS_SET)))
        HGOTO_ERROR(H5E_BTREE, H5E_CANTPROTECT, FAIL, "unable to load B-tree node");

    
    if (bt->level > 0) {
        
        for (u = 0; u < bt->nchildren; u++)
            if (H5B_delete(f, type, bt->child[u], udata) < 0)
                HGOTO_ERROR(H5E_BTREE, H5E_CANTDELETE, FAIL, "unable to delete B-tree node");

    } 
    else {
        bool lt_key_changed, rt_key_changed; 

        
        if (type->remove) {
            
            for (u = 0; u < bt->nchildren; u++) {
                
                if ((type->remove)(f, bt->child[u], H5B_NKEY(bt, shared, u), &lt_key_changed, udata,
                                   H5B_NKEY(bt, shared, u + 1), &rt_key_changed) < H5B_INS_NOOP)
                    HGOTO_ERROR(H5E_BTREE, H5E_CANTREMOVE, FAIL, "can't remove B-tree node");
            } 
        }     
    }         

done:
    if (bt && H5AC_unprotect(f, H5AC_BT, addr, bt, H5AC__DELETED_FLAG | H5AC__FREE_FILE_SPACE_FLAG) < 0)
        HDONE_ERROR(H5E_BTREE, H5E_CANTUNPROTECT, FAIL, "unable to release B-tree node in cache");

    FUNC_LEAVE_NOAPI(ret_value)
} 

H5B_shared_t *
H5B_shared_new(const H5F_t *f, const H5B_class_t *type, size_t sizeof_rkey)
{
    H5B_shared_t *shared = NULL;    
    size_t        u;                
    H5B_shared_t *ret_value = NULL; 

    FUNC_ENTER_NOAPI(NULL)

    
    assert(type);

    
    if (NULL == (shared = H5FL_CALLOC(H5B_shared_t)))
        HGOTO_ERROR(H5E_BTREE, H5E_CANTALLOC, NULL, "memory allocation failed for shared B-tree info");

    
    shared->type        = type;
    shared->two_k       = 2 * H5F_KVALUE(f, type);
    shared->sizeof_addr = H5F_SIZEOF_ADDR(f);
    shared->sizeof_len  = H5F_SIZEOF_SIZE(f);
    shared->sizeof_rkey = sizeof_rkey;
    assert(shared->sizeof_rkey);
    shared->sizeof_keys  = (shared->two_k + 1) * type->sizeof_nkey;
    shared->sizeof_rnode = ((size_t)H5B_SIZEOF_HDR(f) +                 
                            shared->two_k * H5F_SIZEOF_ADDR(f) +        
                            (shared->two_k + 1) * shared->sizeof_rkey); 
    assert(shared->sizeof_rnode);

    
    if (NULL == (shared->page = H5FL_BLK_MALLOC(page, shared->sizeof_rnode)))
        HGOTO_ERROR(H5E_BTREE, H5E_CANTALLOC, NULL, "memory allocation failed for B-tree page");
    memset(shared->page, 0, shared->sizeof_rnode);

    if (NULL == (shared->nkey = H5FL_SEQ_MALLOC(size_t, (size_t)(shared->two_k + 1))))
        HGOTO_ERROR(H5E_BTREE, H5E_CANTALLOC, NULL, "memory allocation failed for B-tree native keys");

    
    for (u = 0; u < (shared->two_k + 1); u++)
        shared->nkey[u] = u * type->sizeof_nkey;

    
    ret_value = shared;

done:
    if (NULL == ret_value)
        if (shared) {
            if (shared->page)
                shared->page = H5FL_BLK_FREE(page, shared->page);
            if (shared->nkey)
                shared->nkey = H5FL_SEQ_FREE(size_t, shared->nkey);
            shared = H5FL_FREE(H5B_shared_t, shared);
        } 

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5B_shared_free(void *_shared)
{
    H5B_shared_t *shared = (H5B_shared_t *)_shared;

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    shared->page = H5FL_BLK_FREE(page, shared->page);

    
    shared->nkey = H5FL_SEQ_FREE(size_t, shared->nkey);

    
    shared = H5FL_FREE(H5B_shared_t, shared);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static H5B_t *
H5B__copy(const H5B_t *old_bt)
{
    H5B_t        *new_node = NULL;
    H5B_shared_t *shared;           
    H5B_t        *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    
    assert(old_bt);
    shared = (H5B_shared_t *)H5UC_GET_OBJ(old_bt->rc_shared);
    assert(shared);

    
    if (NULL == (new_node = H5FL_MALLOC(H5B_t)))
        HGOTO_ERROR(H5E_BTREE, H5E_CANTALLOC, NULL, "memory allocation failed for B-tree root node");

    
    H5MM_memcpy(new_node, old_bt, sizeof(H5B_t));

    
    memset(&new_node->cache_info, 0, sizeof(H5AC_info_t));

    if (NULL == (new_node->native = H5FL_BLK_MALLOC(native_block, shared->sizeof_keys)) ||
        NULL == (new_node->child = H5FL_SEQ_MALLOC(haddr_t, (size_t)shared->two_k)))
        HGOTO_ERROR(H5E_BTREE, H5E_CANTALLOC, NULL, "memory allocation failed for B-tree root node");

    
    H5MM_memcpy(new_node->native, old_bt->native, shared->sizeof_keys);
    H5MM_memcpy(new_node->child, old_bt->child, (sizeof(haddr_t) * shared->two_k));

    
    H5UC_INC(new_node->rc_shared);

    
    ret_value = new_node;

done:
    if (NULL == ret_value) {
        if (new_node) {
            new_node->native = H5FL_BLK_FREE(native_block, new_node->native);
            new_node->child  = H5FL_SEQ_FREE(haddr_t, new_node->child);
            new_node         = H5FL_FREE(H5B_t, new_node);
        } 
    }     

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5B__get_info_helper(H5F_t *f, const H5B_class_t *type, haddr_t addr, const H5B_info_ud_t *info_udata)
{
    H5B_t         *bt = NULL;           
    H5UC_t        *rc_shared;           
    H5B_shared_t  *shared;              
    H5B_cache_ud_t cache_udata;         
    unsigned       level;               
    size_t         sizeof_rnode;        
    haddr_t        next_addr;           
    haddr_t        left_child;          
    herr_t         ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(type);
    assert(H5_addr_defined(addr));
    assert(info_udata);
    assert(info_udata->bt_info);
    assert(info_udata->udata);

    
    if (NULL == (rc_shared = (type->get_shared)(f, info_udata->udata)))
        HGOTO_ERROR(H5E_BTREE, H5E_CANTGET, FAIL, "can't retrieve B-tree's shared ref. count object");
    shared = (H5B_shared_t *)H5UC_GET_OBJ(rc_shared);
    assert(shared);

    
    sizeof_rnode = shared->sizeof_rnode;

    
    cache_udata.f         = f;
    cache_udata.type      = type;
    cache_udata.rc_shared = rc_shared;
    cache_udata.exp_level = H5B_UNKNOWN_NODELEVEL;
    if (NULL == (bt = (H5B_t *)H5AC_protect(f, H5AC_BT, addr, &cache_udata, H5AC__READ_ONLY_FLAG)))
        HGOTO_ERROR(H5E_BTREE, H5E_CANTPROTECT, FAIL, "unable to load B-tree node");

    
    left_child = bt->child[0];
    next_addr  = bt->right;
    level      = bt->level;

    
    info_udata->bt_info->size += sizeof_rnode;
    info_udata->bt_info->num_nodes++;

    
    if (H5AC_unprotect(f, H5AC_BT, addr, bt, H5AC__NO_FLAGS_SET) < 0)
        HGOTO_ERROR(H5E_BTREE, H5E_CANTUNPROTECT, FAIL, "unable to release B-tree node");
    bt = NULL;

    
    while (H5_addr_defined(next_addr)) {
        
        addr = next_addr;
        if (NULL == (bt = (H5B_t *)H5AC_protect(f, H5AC_BT, addr, &cache_udata, H5AC__READ_ONLY_FLAG)))
            HGOTO_ERROR(H5E_BTREE, H5E_CANTPROTECT, FAIL, "B-tree node");

        
        next_addr = bt->right;

        
        info_udata->bt_info->size += sizeof_rnode;
        info_udata->bt_info->num_nodes++;

        
        if (H5AC_unprotect(f, H5AC_BT, addr, bt, H5AC__NO_FLAGS_SET) < 0)
            HGOTO_ERROR(H5E_BTREE, H5E_CANTUNPROTECT, FAIL, "unable to release B-tree node");
        bt = NULL;
    } 

    
    if (level > 0) {
        
        if (H5B__get_info_helper(f, type, left_child, info_udata) < 0)
            HGOTO_ERROR(H5E_BTREE, H5E_BADITER, FAIL, "unable to list B-tree node");
    } 

done:
    if (bt && H5AC_unprotect(f, H5AC_BT, addr, bt, H5AC__NO_FLAGS_SET) < 0)
        HDONE_ERROR(H5E_BTREE, H5E_CANTUNPROTECT, FAIL, "unable to release B-tree node");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5B_get_info(H5F_t *f, const H5B_class_t *type, haddr_t addr, H5B_info_t *bt_info, H5B_operator_t op,
             void *udata)
{
    H5B_info_ud_t info_udata;          
    herr_t        ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f);
    assert(type);
    assert(bt_info);
    assert(H5_addr_defined(addr));
    assert(udata);

    
    memset(bt_info, 0, sizeof(*bt_info));

    
    info_udata.bt_info = bt_info;
    info_udata.udata   = udata;

    
    if (H5B__get_info_helper(f, type, addr, &info_udata) < 0)
        HGOTO_ERROR(H5E_BTREE, H5E_BADITER, FAIL, "B-tree iteration failed");

    
    
    if (op)
        if ((ret_value = H5B__iterate_helper(f, type, addr, H5B_UNKNOWN_NODELEVEL, op, udata)) < 0)
            HERROR(H5E_BTREE, H5E_BADITER, "B-tree iteration failed");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5B_valid(H5F_t *f, const H5B_class_t *type, haddr_t addr)
{
    H5B_t         *bt = NULL;           
    H5UC_t        *rc_shared;           
    H5B_cache_ud_t cache_udata;         
    herr_t         ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f);
    assert(type);

    if (!H5_addr_defined(addr))
        HGOTO_ERROR(H5E_BTREE, H5E_BADVALUE, FAIL, "address is undefined");

    
    if (NULL == (rc_shared = (type->get_shared)(f, NULL)))
        HGOTO_ERROR(H5E_BTREE, H5E_CANTGET, FAIL, "can't retrieve B-tree's shared ref. count object");
    assert(H5UC_GET_OBJ(rc_shared) != NULL);

    
    cache_udata.f         = f;
    cache_udata.type      = type;
    cache_udata.rc_shared = rc_shared;
    cache_udata.exp_level = H5B_UNKNOWN_NODELEVEL;
    if (NULL == (bt = (H5B_t *)H5AC_protect(f, H5AC_BT, addr, &cache_udata, H5AC__READ_ONLY_FLAG)))
        HGOTO_ERROR(H5E_BTREE, H5E_CANTPROTECT, FAIL, "unable to protect B-tree node");

done:
    
    if (bt && H5AC_unprotect(f, H5AC_BT, addr, bt, H5AC__NO_FLAGS_SET) < 0)
        HDONE_ERROR(H5E_BTREE, H5E_CANTUNPROTECT, FAIL, "unable to release B-tree node");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5B__node_dest(H5B_t *bt)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(bt);
    assert(bt->rc_shared);

    bt->child  = H5FL_SEQ_FREE(haddr_t, bt->child);
    bt->native = H5FL_BLK_FREE(native_block, bt->native);
    H5UC_DEC(bt->rc_shared);
    bt = H5FL_FREE(H5B_t, bt);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 
