% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ekw.R
\name{pekw}
\alias{pekw}
\title{Cumulative Distribution Function (CDF) of the EKw Distribution}
\usage{
pekw(q, alpha = 1, beta = 1, lambda = 1, lower.tail = TRUE, log.p = FALSE)
}
\arguments{
\item{q}{Vector of quantiles (values generally between 0 and 1).}

\item{alpha}{Shape parameter \code{alpha} > 0. Can be a scalar or a vector.
Default: 1.0.}

\item{beta}{Shape parameter \code{beta} > 0. Can be a scalar or a vector.
Default: 1.0.}

\item{lambda}{Shape parameter \code{lambda} > 0 (exponent parameter).
Can be a scalar or a vector. Default: 1.0.}

\item{lower.tail}{Logical; if \code{TRUE} (default), probabilities are
\eqn{P(X \le q)}, otherwise, \eqn{P(X > q)}.}

\item{log.p}{Logical; if \code{TRUE}, probabilities \eqn{p} are given as
\eqn{\log(p)}. Default: \code{FALSE}.}
}
\value{
A vector of probabilities, \eqn{F(q)}, or their logarithms/complements
depending on \code{lower.tail} and \code{log.p}. The length of the result
is determined by the recycling rule applied to the arguments (\code{q},
\code{alpha}, \code{beta}, \code{lambda}). Returns \code{0} (or \code{-Inf}
if \code{log.p = TRUE}) for \code{q <= 0} and \code{1} (or \code{0} if
\code{log.p = TRUE}) for \code{q >= 1}. Returns \code{NaN} for invalid
parameters.
}
\description{
Computes the cumulative distribution function (CDF), \eqn{P(X \le q)}, for the
Exponentiated Kumaraswamy (EKw) distribution with parameters \code{alpha}
(\eqn{\alpha}), \code{beta} (\eqn{\beta}), and \code{lambda} (\eqn{\lambda}).
This distribution is defined on the interval (0, 1) and is a special case
of the Generalized Kumaraswamy (GKw) distribution where \eqn{\gamma = 1}
and \eqn{\delta = 0}.
}
\details{
The Exponentiated Kumaraswamy (EKw) distribution is a special case of the
five-parameter Generalized Kumaraswamy distribution (\code{\link{pgkw}})
obtained by setting parameters \eqn{\gamma = 1} and \eqn{\delta = 0}.

The CDF of the GKw distribution is \eqn{F_{GKw}(q) = I_{y(q)}(\gamma, \delta+1)},
where \eqn{y(q) = [1-(1-q^{\alpha})^{\beta}]^{\lambda}} and \eqn{I_x(a,b)}
is the regularized incomplete beta function (\code{\link[stats]{pbeta}}).
Setting \eqn{\gamma=1} and \eqn{\delta=0} gives \eqn{I_{y(q)}(1, 1)}. Since
\eqn{I_x(1, 1) = x}, the CDF simplifies to \eqn{y(q)}:
\deqn{
F(q; \alpha, \beta, \lambda) = \bigl[1 - (1 - q^\alpha)^\beta \bigr]^\lambda
}
for \eqn{0 < q < 1}.
The implementation uses this closed-form expression for efficiency and handles
\code{lower.tail} and \code{log.p} arguments appropriately.
}
\examples{
\donttest{
# Example values
q_vals <- c(0.2, 0.5, 0.8)
alpha_par <- 2.0
beta_par <- 3.0
lambda_par <- 1.5

# Calculate CDF P(X <= q)
probs <- pekw(q_vals, alpha_par, beta_par, lambda_par)
print(probs)

# Calculate upper tail P(X > q)
probs_upper <- pekw(q_vals, alpha_par, beta_par, lambda_par,
  lower.tail = FALSE
)
print(probs_upper)
# Check: probs + probs_upper should be 1
print(probs + probs_upper)

# Calculate log CDF
logs <- pekw(q_vals, alpha_par, beta_par, lambda_par, log.p = TRUE)
print(logs)
# Check: should match log(probs)
print(log(probs))

# Compare with pgkw setting gamma = 1, delta = 0
probs_gkw <- pgkw(q_vals, alpha_par, beta_par,
  gamma = 1.0, delta = 0.0,
  lambda = lambda_par
)
print(paste("Max difference:", max(abs(probs - probs_gkw)))) # Should be near zero

# Plot the CDF for different lambda values
curve_q <- seq(0.01, 0.99, length.out = 200)
curve_p1 <- pekw(curve_q, alpha = 2, beta = 3, lambda = 0.5)
curve_p2 <- pekw(curve_q, alpha = 2, beta = 3, lambda = 1.0) # standard Kw
curve_p3 <- pekw(curve_q, alpha = 2, beta = 3, lambda = 2.0)

plot(curve_q, curve_p2,
  type = "l", main = "EKw CDF Examples (alpha=2, beta=3)",
  xlab = "q", ylab = "F(q)", col = "red", ylim = c(0, 1)
)
lines(curve_q, curve_p1, col = "blue")
lines(curve_q, curve_p3, col = "green")
legend("bottomright",
  legend = c("lambda=0.5", "lambda=1.0 (Kw)", "lambda=2.0"),
  col = c("blue", "red", "green"), lty = 1, bty = "n"
)
}

}
\references{
Nadarajah, S., Cordeiro, G. M., & Ortega, E. M. (2012). The exponentiated
Kumaraswamy distribution. \emph{Journal of the Franklin Institute}, \emph{349}(3),

Cordeiro, G. M., & de Castro, M. (2011). A new family of generalized
distributions. \emph{Journal of Statistical Computation and Simulation},

Kumaraswamy, P. (1980). A generalized probability density function for
double-bounded random processes. \emph{Journal of Hydrology}, \emph{46}(1-2), 79-88.
}
\seealso{
\code{\link{pgkw}} (parent distribution CDF),
\code{\link{dekw}}, \code{\link{qekw}}, \code{\link{rekw}} (other EKw functions),
}
\author{
Lopes, J. E.
}
\keyword{cumulative}
\keyword{distribution}
