% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/geo_join.R
\name{geo_join}
\alias{geo_join}
\alias{geo_inner_join}
\alias{geo_left_join}
\alias{geo_right_join}
\alias{geo_full_join}
\alias{geo_semi_join}
\alias{geo_anti_join}
\title{Join two tables based on a geo distance of longitudes and latitudes}
\usage{
geo_join(
  x,
  y,
  by = NULL,
  max_dist,
  method = c("haversine", "geo", "cosine", "meeus", "vincentysphere",
    "vincentyellipsoid"),
  unit = c("miles", "km"),
  mode = "inner",
  distance_col = NULL,
  ...
)

geo_inner_join(
  x,
  y,
  by = NULL,
  method = "haversine",
  max_dist = 1,
  distance_col = NULL,
  ...
)

geo_left_join(
  x,
  y,
  by = NULL,
  method = "haversine",
  max_dist = 1,
  distance_col = NULL,
  ...
)

geo_right_join(
  x,
  y,
  by = NULL,
  method = "haversine",
  max_dist = 1,
  distance_col = NULL,
  ...
)

geo_full_join(
  x,
  y,
  by = NULL,
  method = "haversine",
  max_dist = 1,
  distance_col = NULL,
  ...
)

geo_semi_join(
  x,
  y,
  by = NULL,
  method = "haversine",
  max_dist = 1,
  distance_col = NULL,
  ...
)

geo_anti_join(
  x,
  y,
  by = NULL,
  method = "haversine",
  max_dist = 1,
  distance_col = NULL,
  ...
)
}
\arguments{
\item{x}{A tbl}

\item{y}{A tbl}

\item{by}{Columns by which to join the two tables}

\item{max_dist}{Maximum distance to use for joining}

\item{method}{Method to use for computing distance: one of "haversine"
(default), "geo", "cosine", "meeus", "vincentysphere", "vincentyellipsoid"}

\item{unit}{Unit of distance for threshold (default "miles")}

\item{mode}{One of "inner", "left", "right", "full" "semi", or "anti"}

\item{distance_col}{If given, will add a column with this name containing the
geographical distance between the two}

\item{...}{Extra arguments passed on to the distance method}
}
\description{
This allows joining based on combinations of longitudes and latitudes. If you
are using a distance metric that is *not* based on latitude and longitude,
use \code{\link{distance_join}} instead. Distances are calculated based on
the \code{distHaversine}, \code{distGeo}, \code{distCosine}, etc methods in
the geosphere package.
}
\details{
"Haversine" was chosen as default since in some tests it is
  approximately the fastest method. Note that by far the slowest method is
  vincentyellipsoid, and on fuzzy joins should only be used when there are
  very few pairs and accuracy is imperative.

  If you need to use a custom geo method, you may want to write it directly
  with the \code{multi_by} and \code{multi_match_fun} arguments to
  \code{fuzzy_join}.
}
\examples{

library(dplyr)
data("state")

# find pairs of US states whose centers are within
# 200 miles of each other
states <- tibble::tibble(state = state.name,
                 longitude = state.center$x,
                 latitude = state.center$y)

s1 <- rename(states, state1 = state)
s2 <- rename(states, state2 = state)

pairs <- s1 \%>\%
  geo_inner_join(s2, max_dist = 200) \%>\%
  filter(state1 != state2)

pairs

# plot them
if (requireNamespace("ggplot2", quietly = TRUE)) {
  ggplot2::ggplot(pairs, ggplot2::aes(x = longitude.x, y = latitude.x,
                                      xend = longitude.y, yend = latitude.y)) +
    ggplot2::geom_segment(color = "red") +
    ggplot2::theme_void()
}

# also get distances
s1 \%>\%
  geo_inner_join(s2, max_dist = 200, distance_col = "distance")


}
