% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/notice.R
\name{calcite_notice}
\alias{calcite_notice}
\title{Create a Calcite Notice Component}
\usage{
calcite_notice(
  ...,
  title = NULL,
  message = NULL,
  link = NULL,
  actions_end = NULL,
  id = NULL,
  open = NULL,
  closable = NULL,
  icon = NULL,
  icon_flip_rtl = NULL,
  kind = NULL,
  scale = NULL,
  width = NULL
)
}
\arguments{
\item{...}{Additional content passed to the component}

\item{title}{Content for the title slot}

\item{message}{Content for the message slot}

\item{link}{Content for the link slot (e.g. a \code{calcite_link()})}

\item{actions_end}{Content for the actions-end slot}

\item{id}{Component ID (required for Shiny reactivity)}

\item{open}{Whether the notice is visible (default: FALSE)}

\item{closable}{Whether to show a close button (default: FALSE)}

\item{icon}{Show default icon (TRUE) or specific icon name (string)}

\item{icon_flip_rtl}{Flip icon in RTL languages (default: FALSE)}

\item{kind}{Type of notice: "brand", "danger", "info", "success", or "warning"}

\item{scale}{Size of the component: "s" (small), "m" (medium), or "l" (large)}

\item{width}{Width behavior: "auto" or "full" (note: "half" is deprecated)}
}
\value{
An object of class \code{calcite_component}
}
\description{
Creates a notice component for displaying informative, contextually relevant
messages. Best for persistent information that can be open at page load or
displayed as a result of user action.
}
\details{
\subsection{Shiny Integration}{

The notice emits events when opened or closed, making it easy to track state
and respond to user dismissals.

\strong{Available properties in \code{input$id}:}
\itemize{
\item \verb{$open} - Whether the notice is currently visible
\item \verb{$closable} - Whether the notice can be closed
\item \verb{$kind} - The type of notice
\item Other component properties
}

\strong{Basic usage:}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{calcite_notice(
  id = "my_notice",
  open = TRUE,
  closable = TRUE,
  kind = "success",
  icon = TRUE,
  title = "Success!",
  message = "Your changes have been saved."
)

# In server - detect when user closes the notice
observeEvent(input$my_notice$open, \{
  if (!input$my_notice$open) \{
    message("User dismissed the notice")
  \}
\})
}\if{html}{\out{</div>}}

\strong{Show/hide from server:}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{# Show a notice
update_calcite("my_notice", open = TRUE)

# Hide a notice
update_calcite("my_notice", open = FALSE)
}\if{html}{\out{</div>}}
}

\subsection{Slots}{

The notice supports several slots for organizing content:
\itemize{
\item \strong{title}: The notice title
\item \strong{message}: The notice message
\item \strong{link}: A calcite-action for links like "Read more"
\item \strong{actions-end}: Additional actions (recommended: 2 or fewer)
}
}

\subsection{Best Practices}{
\itemize{
\item Use for informative, contextually relevant information
\item Can be open at page load or shown based on user action
\item Can be persistent or closable
\item Use appropriate \code{kind} to convey message severity
}
}
}
\examples{
# Basic notice
calcite_notice(
  open = TRUE,
  icon = TRUE,
  closable = TRUE,
  title = "New feature available",
  message = "Check out the reporting dashboard"
)

# Notice with specific icon and kind
calcite_notice(
  open = TRUE,
  kind = "danger",
  icon = "exclamation-mark-triangle",
  title = "Error in form",
  message = "Please correct the highlighted fields"
)

# Notice with action link
calcite_notice(
  open = TRUE,
  icon = "layers-reference",
  title = "Try this trick",
  message = "Select multiple layers at once",
  link = calcite_link(text = "Read more", href = "#")
)

# Shiny example with server control
if (interactive()) {
  library(shiny)

  ui <- calcite_shell(
    calcite_panel(
      heading = "Notice Demo",

      calcite_notice(
        id = "my_notice",
        open = FALSE,
        closable = TRUE,
        kind = "success",
        icon = TRUE,
        title = "Success!",
        message = "Your action completed successfully"
      ),

      calcite_button(
        id = "show_notice",
        "Show Notice"
      ),

      verbatimTextOutput("notice_status")
    )
  )

  server <- function(input, output, session) {
    observeEvent(input$show_notice$clicks, {
      update_calcite("my_notice", open = TRUE)
    })

    output$notice_status <- renderPrint({
      input$my_notice
    })
  }

  shinyApp(ui, server)
}
}
\references{
\href{https://developers.arcgis.com/calcite-design-system/components/notice/}{Official Documentation}
}
