#' Plot 2D trajectories as variable-width lines
#'
#' Creates a 2D line plot showing research trajectories over time, with highlighted
#' trajectories displayed as variable-width lines and optional background trajectories
#' shown in lowlight style. Edge widths grow along each highlighted trajectory based
#' on cumulative paper counts, and labels are placed at trajectory endpoints.
#'
#' @param traj_data List containing trajectory data generated by 
#'   `detect_main_trajectories()` with components:
#'   - `graph`: igraph object containing nodes and edges across years
#'   - `trajectories`: tibble of all candidate trajectories (traj_id + nodes list)
#' @param traj_filtered Filtered trajectories tibble from `filter_trajectories()` 
#'   containing the subset to emphasize. Must contain columns:
#'   - `traj_id`: trajectory identifiers
#'   - `nodes`: list of character vectors (ordered by time or orderable)
#' @param title Plot title (default: "Main trajectories")
#' @param width_range Range for edge widths of highlighted trajectories 
#'   (default: c(0.8, 6.0)). Width at each segment is scaled by cumulative 
#'   paper count up to the next node.
#' @param use_raw_papers Whether to use raw paper counts for width scaling 
#'   (default: FALSE). If TRUE, uses raw `quantity_papers`; if FALSE, uses 
#'   weighted size: `quantity_papers * prop_tracked_intra_group`.
#' @param label_nudge_x Horizontal nudge for trajectory end labels to prevent 
#'   overlap with nodes (default: 0.30)
#' @param label_size Text size for trajectory end labels (default: 4)
#' @param show_only_highlighted Whether to show only highlighted trajectories 
#'   (default: FALSE). If TRUE, hides all non-highlighted trajectory lines; 
#'   if FALSE, draws lowlight background.
#' @param lowlight_width Line width for lowlight (background) edges 
#'   (default: 0.9)
#' @param lowlight_alpha Transparency for lowlight edges (default: 0.22; 
#'   smaller values = more transparent)
#' @param lowlight_color Color for lowlight edges (default: "#9AA5B1" - neutral gray)
#'
#' @return A ggplot object displaying the trajectory network
#'
#' @details
#' This function visualizes research trajectories as variable-width lines:
#' - **Highlighted trajectories** (`traj_filtered`) are colored lines with widths 
#'   proportional to cumulative paper counts (raw or weighted)
#' - **Background trajectories** (when `show_only_highlighted = FALSE`) are shown 
#'   as thin, transparent lines
#' - **Trajectory labels** are placed at the end of each highlighted trajectory
#' - The **x-axis represents publication years** using a Sugiyama layout
#' - The **y-axis shows vertical positions** from the layout (no intrinsic meaning)
#' - Colors are assigned only to highlighted trajectories present in the plot
#' 
#' When `traj_data$trajectories` is available and `show_only_highlighted = FALSE`,
#' the lowlight layer shows only edges that belong to any trajectory but not the 
#' highlighted set. Otherwise, it shows the entire graph minus highlighted edges.
#'
#' @examples
#' \dontrun{
#' # Detect main trajectories first
#' traj_data <- detect_main_trajectories(your_graph_data)
#' 
#' # Filter trajectories of interest
#' filtered_traj <- filter_trajectories(traj_data$trajectories, 
#'                                      min_papers = 10)
#' 
#' # Create the plot
#' plot_group_trajectories_lines_2d(
#'   traj_data = traj_data,
#'   traj_filtered = filtered_traj,
#'   title = "Key Research Trajectories",
#'   width_range = c(1, 8),
#'   show_only_highlighted = FALSE
#' )
#' }
#'
#' @export
#' @importFrom igraph vcount ecount E V ends
#' @importFrom dplyr filter
#' @importFrom ggplot2 ggplot labs aes element_rect element_text element_blank 
#'   scale_x_continuous scale_y_continuous theme expansion geom_label
#' @importFrom ggraph ggraph geom_edge_link scale_edge_colour_manual 
#'   scale_edge_width_identity
#' @importFrom RColorBrewer brewer.pal
#' @importFrom scales hue_pal
#' @importFrom stats setNames
#' @importFrom tibble tibble
plot_group_trajectories_lines_2d <- function(
  traj_data,
  traj_filtered,
  title = "Main trajectories",
  width_range = c(0.8, 6.0),
  use_raw_papers = FALSE,
  label_nudge_x = 0.30,
  label_size = 4,
  show_only_highlighted = FALSE,
  lowlight_width = 0.9,
  lowlight_alpha = 0.22,
  lowlight_color = "#9AA5B1"
) {
  # Input validation
  if (!is.list(traj_data) || !all(c("graph", "trajectories") %in% names(traj_data))) {
    stop("traj_data must be a list with 'graph' and 'trajectories' components", call. = FALSE)
  }
  
  g <- traj_data$graph
  tr_all <- traj_data$trajectories
  tr_highlight <- traj_filtered
  
  if (igraph::vcount(g) == 0) return(ggplot2::ggplot() + ggplot2::labs(title = "Empty graph"))

  # 1) Tag highlighted edges (adds E(g)$traj_id and E(g)$traj_width)
  g2 <- assign_traj_edge_widths(g, tr_highlight, width_range, use_raw_papers)

  # 2) Optionally tag edges that belong to ANY trajectory (for cleaner lowlight)
  igraph::E(g2)$in_any_traj <- FALSE
  if (!is.null(tr_all) && nrow(tr_all)) {
    for (i in seq_len(nrow(tr_all))) {
      pn <- tr_all$nodes[[i]]
      if (length(pn) >= 2) {
        pn <- pn[order(.extract_year(pn), pn)]
        eids <- igraph::E(g2, path = pn)
        igraph::E(g2)[eids]$in_any_traj <- TRUE
      }
    }
  }

  # 3) Layout & axis ticks
  layinfo <- mk_layout_and_year_scale(g2); lay <- layinfo$lay
  if (igraph::ecount(g2) > 0 && length(igraph::E(g2)[!is.na(traj_id)]) > 0) {
    e_sel    <- igraph::E(g2)[!is.na(traj_id)]
    ends_mat <- igraph::ends(g2, e_sel, names = TRUE)
    x_pos    <- c(layinfo$vertex_x[ends_mat[, 1]], layinfo$vertex_x[ends_mat[, 2]])
    x_pos    <- sort(unique(as.integer(x_pos)))
    breaks   <- x_pos
    labels   <- as.integer(layinfo$x_to_year[as.character(x_pos)])
  } else {
    breaks <- layinfo$all_breaks
    labels <- layinfo$all_labels
  }

  # 4) Colour palette locked to the highlighted traj_ids present
  levs <- sort(unique(igraph::E(g2)$traj_id[!is.na(igraph::E(g2)$traj_id)]))
  nlev <- length(levs)
  pal  <- if (nlev <= 8) RColorBrewer::brewer.pal(max(3, nlev), "Set2")[seq_len(nlev)] else scales::hue_pal()(nlev)
  values_map <- stats::setNames(pal, levs)
  igraph::E(g2)$traj_id <- factor(igraph::E(g2)$traj_id, levels = levs)

  # 5) End labels for highlighted
  label_positions <- tibble::tibble()
  if (nlev > 0 && nrow(tr_highlight)) {
    get_last <- function(nodes) { 
      ord <- order(.extract_year(nodes), nodes)
      nodes[ord][length(nodes)] 
    }
    lnodes <- vapply(tr_highlight$nodes, get_last, FUN.VALUE = character(1))
    tids   <- tr_highlight$traj_id
    keep   <- tids %in% levs
    vidx   <- match(lnodes[keep], lay$name)
    label_positions <- tibble::tibble(
      x = lay$x[vidx] + label_nudge_x,
      y = lay$y[vidx],
      label = tids[keep]
    ) |>
      dplyr::filter(is.finite(.data$x), is.finite(.data$y))
  }

  p <- ggraph::ggraph(lay)

  # 6) Background edges (lowlight)
  if (!show_only_highlighted) {
    if (!is.null(tr_all) && nrow(tr_all)) {
      # only edges that belong to any trajectory, but are NOT in the highlighted set
      p <- p +
        ggraph::geom_edge_link(
          ggplot2::aes(filter = .data$in_any_traj & is.na(.data$traj_id)),
          linewidth = lowlight_width, alpha = lowlight_alpha, colour = lowlight_color,
          lineend = "round", show.legend = FALSE
        )
    } else {
      # whole graph minus highlighted
      p <- p +
        ggraph::geom_edge_link(
          ggplot2::aes(filter = is.na(.data$traj_id)),
          linewidth = lowlight_width, alpha = lowlight_alpha, colour = lowlight_color,
          lineend = "round", show.legend = FALSE
        )
    }
  }

  # 7) Highlighted trajectories — use edge_* aesthetics & scales
  p <- p +
    ggraph::geom_edge_link(
      ggplot2::aes(edge_colour = .data$traj_id, edge_width = .data$traj_width, 
                   filter = !is.na(.data$traj_id)),
      lineend = "round", show.legend = FALSE
    ) +
    ggraph::scale_edge_colour_manual(values = values_map, limits = levs, breaks = levs, drop = FALSE) +
    ggraph::scale_edge_width_identity(guide = "none")

  # 8) End labels
  if (nrow(label_positions) > 0) {
    p <- p +
      ggplot2::geom_label(
        data = label_positions,
        ggplot2::aes(x = .data$x, y = .data$y, label = .data$label),
        fill = "gray40", color = "white",
        size = label_size + 0.5, fontface = "bold",
        label.size = 0, alpha = 0.9, hjust = 0,
        show.legend = FALSE
      )
  }

  # 9) Axes & theme
  p +
    ggplot2::scale_x_continuous(breaks = breaks, labels = labels, 
                               expand = ggplot2::expansion(mult = c(0.02, 0.10))) +
    ggplot2::scale_y_continuous(expand = ggplot2::expansion(mult = 0.10)) +
    ggplot2::theme(
      axis.text.x = ggplot2::element_text(angle = 90, vjust = 0.5, hjust = 1, size = 18),
      axis.text.y = ggplot2::element_blank(),
      axis.ticks.y = ggplot2::element_blank(),
      panel.background = ggplot2::element_rect(fill = "white", colour = "white"),
      legend.position = "none",
      plot.title = ggplot2::element_text(size = 24)
    ) +
    ggplot2::labs(title = title, x = "Publication year", y = NULL)
}
