% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ccf_function.R
\name{CCF}
\alias{CCF}
\title{CCF: Copula Control Function}
\usage{
CCF(formula, data)
}
\arguments{
\item{formula}{A formula describing the model to be fitted. The details of
model specification are given under ``Details''.}

\item{data}{a data frame, list, or environment containing the variables in
the model.}
}
\value{
A list of class "\code{ccf}" containing the following components:
\item{ccf}{a matrix of the first-stage residuals as copula control functions.}
\item{pstar}{a matrix representing \eqn{P^*}}
\item{wstar}{a matrix representing \eqn{W^*}}
}
\description{
\code{CCF()} computes copula control functions (CCFs) that can be
added in the outcome model as control variables to correct for endogeneity.
which returns \eqn{P^*}, \eqn{W^*}, and the first-stage residuals.
}
\details{
The \code{formula} argument is either in the 1-bar form \code{Y ~ X | P} or the 2-bar form \code{Y ~ X | P | W}, where
\code{X} respresents the explanatory variable(s) in the \code{Y} model, \code{P} represents the continuous
endogenous regressors, and \code{W} represents the exogenous regressors.  If \code{X} contains no
exogenous regressors, then the 2sCOPE model reduces to the simpler model in Park and Gupta (2012)
and returns \eqn{P^*} (the copula transformation of \code{P}) as CCF and \eqn{W^*} (the copula transformation of \code{W}) as null.
When the structural outcome model includes an intercept, copula transformations of regressors in \code{P} and \code{W}  use the
optimized algorithm (Equation 9 in Qian, Koschmann, and Xie, 2025)  to avoid estimation bias.

The function CCF() will compute copula control function for each endogenous regressor specified in \code{P}.
Only first-order terms of endogenous regressors need to be included in \code{P}, even when the structural outcome model
contains higher-order terms of endogenous regressors. This is because including copula control functions for the
first-order endogenous regressors is sufficient to control for endogeneity, while adding control functions for
higher-order endogenous terms---such as interactions among endogenous regressors, interactions between endogenous and
exogenous regressors, or squared endogenous regressors---is unnecessary and can substantially degrade the performance
of copula correction (Qian, Koschmann, and Xie, 2025). This parsimonious treatment of higher-order endogenous
regressors is a merit of copula correction.

Thus, if \code{X} contains no higher-order terms of endogenous regressors, the simpler 1-bar form \code{Y ~ X | P}
can be used, and \code{CCF()} treats all regressors in \code{X} except those in \code{P} as exogenous.
When \code{X} includes higher-order endogenous terms, the 2-bar form  \code{Y ~ X | P | W} should be used to explicitly specify
the exogenous regressors in \code{W} and ensure that the higher-order endogenous terms are not treated as exogenous variables.
}
\examples{
data("diapers") #load data

### Specify logPrice as endogenous using the 1-bar option
#run the copula control function
ccf_1bar <- CCF(logVol ~ logPrice+Fshare+week+Q2+Q3+Q4|logPrice,data=diapers)
#print the first 5 elements of the first-stage residuals
head(ccf_1bar$ccf, 5)
head(ccf_1bar$pstar, 5) #print the first 5 elements of P*
head(ccf_1bar$wstar, 5) #print the first 5 elements of W*

### Specify logPrice as endogenous and the rest of the variables as exogenous
#using the 2-bar option, which will produce the same results,
ccf_2bar <- CCF(logVol ~ logPrice+Fshare+week+Q2+Q3+Q4|logPrice|
    Fshare+week+Q2+Q3+Q4, data = diapers) #run the copula control function
head(ccf_2bar$ccf, 5) #print first 5 elements of the 1st-stage resid
head(ccf_2bar$pstar, 5) #print first 5 elements of P*
head(ccf_2bar$wstar, 5) #print first 5 elements of W*

### Run Park & Gupta (2012) by specifying logPrice as the only regressor,
### which is endogenous.

#run the copula control function
ccf_pg <- CCF(logVol ~ logPrice|logPrice, data = diapers)
head(ccf_pg$ccf, 5) #print first 5 elements of the 1st-stage resid
head(ccf_pg$pstar, 5) #print first 5 elements of P*
head(ccf_pg$wstar, 5) #print first 5 elements of W*
# notice how the 1st-stage residuals and P* are equivalent, and wstar is NULL

}
\references{
Qian, Y., Koschmann, A., & Xie, H. (2025).
\emph{EXPRESS: A Practical Guide to Endogeneity Correction Using Copulas.}
Journal of Marketing. \url{doi:10.1177/00222429251410844}\cr

Park, S., & Gupta, S. (2012).
\emph{Handling endogenous regressors by joint estimation using copulas.}
Marketing Science, 31(4), 567-586.\cr

Yang, F., Qian, Y., & Xie, H. (2025).
\emph{Addressing Endogeneity Using a Two-Stage Copula Generated Regressor Approach.}
Journal of Marketing Research, 62(4), 601-623.
\url{doi:10.1177/00222437241296453}\cr
}
