# =============================================================================
# MODULE 0: ROBUST UTILITY FUNCTIONS
# =============================================================================

#' Null-coalescing operator
#'
#' Null-coalescing operator (internal)
#' @noRd
`%||%` <- function(a, b) if (length(a) == 0 || is.null(a)) b else a

#' Robust Scale Estimation
#'
#' MAD-based scale estimation with fallback chain: MAD -> IQR/1.349 -> SD.
#'
#' @param x Numeric vector
#' @return Robust scale estimate, or NA_real_ if fewer than 2 values
#' @export
#' @examples
#' robust_scale(rnorm(50))
#' robust_scale(c(1, 2, 3, 100))
robust_scale <- function(x) {
  x <- x[is.finite(x)]
  if (length(x) < 2) return(NA_real_)
  m <- stats::mad(x, constant = 1.4826)
  if (m == 0) m <- stats::IQR(x) / 1.349
  if (is.na(m) || m == 0) m <- stats::sd(x)
  if (is.na(m) || m == 0) m <- .Machine$double.eps
  m
}

#' Safe Standard Deviation
#'
#' Returns SD, guaranteed to be positive (returns machine epsilon if zero/NA).
#'
#' @param x Numeric vector
#' @param na.rm Logical; should NA values be removed?
#' @return Standard deviation, guaranteed positive
#' @export
#' @examples
#' safe_sd(c(5, 5, 5))
safe_sd <- function(x, na.rm = TRUE) {
  s <- stats::sd(x, na.rm = na.rm)
  if (is.na(s) || s == 0) .Machine$double.eps else s
}

#' Safe Ratio
#'
#' Computes b / (a + b) with guards for NA and near-zero denominators.
#'
#' @param a First value
#' @param b Second value
#' @return Ratio b/(a+b), or 0.5 if degenerate, or NA_real_ if inputs invalid
#' @noRd
safe_ratio <- function(a, b) {
  if (is.na(a) || is.na(b) || !is.finite(a) || !is.finite(b))
    return(NA_real_)
  denom <- a + b
  if (!is.finite(denom) || denom <= .Machine$double.eps) return(0.5)
  b / denom
}
