# =============================================================================
# MODULE 6: SEPARATION QUALITY ASSESSMENT
# =============================================================================

#' Cohen's d Effect Size
#'
#' Computes Cohen's d between two distributions.
#'
#' @param x First distribution (numeric vector)
#' @param y Second distribution (numeric vector)
#' @return Cohen's d (positive = x > y in expected direction)
#' @export
#' @examples
#' cohens_d(rnorm(30, 10, 2), rnorm(30, 5, 2))
cohens_d <- function(x, y) {
  x <- x[is.finite(x)]
  y <- y[is.finite(y)]
  if (length(x) < 2 || length(y) < 2) return(NA_real_)
  nx <- length(x)
  ny <- length(y)
  pooled_sd <- sqrt(((nx - 1) * stats::var(x) + (ny - 1) * stats::var(y)) / (nx + ny - 2))
  if (!is.finite(pooled_sd) || pooled_sd <= .Machine$double.eps) return(NA_real_)
  (mean(x) - mean(y)) / pooled_sd
}

#' Assess Separation Quality
#'
#' Evaluates PC/NC separation and recommends classification profile.
#' Cohen's d thresholds are heuristic, calibrated against three RT-QuIC datasets.
#'
#' @param pc_mp Positive control MP values
#' @param nc_mp Negative control MP values
#' @param pc_ttt Positive control TTT values
#' @param nc_ttt Negative control TTT values
#' @return List with d_mp, d_ttt, d_combined, regime, and recommended_profile
#' @export
#' @examples
#' pc_mp <- rnorm(8, 100, 10)
#' nc_mp <- rnorm(8, 20, 5)
#' pc_ttt <- rnorm(8, 8, 1)
#' nc_ttt <- rnorm(8, 72, 5)
#' assess_separation(pc_mp, nc_mp, pc_ttt, nc_ttt)
assess_separation <- function(pc_mp, nc_mp, pc_ttt, nc_ttt) {
  d_mp <- cohens_d(pc_mp, nc_mp)
  d_ttt <- cohens_d(nc_ttt, pc_ttt)  # NC should be higher TTT
  d_vals <- abs(c(d_mp %||% NA_real_, d_ttt %||% NA_real_))
  d_vals <- d_vals[is.finite(d_vals)]
  d_combined <- if (length(d_vals) > 0) stats::median(d_vals) else NA_real_


  # Profile thresholds are heuristic, calibrated against 3 RT-QuIC datasets.
  # d < 1.5: matrix interference or assay failure
  # d 1.5-3.0: moderate separation (environmental/spiked samples)
  # d > 3.0: strong separation (clean assay conditions)
  if (is.na(d_combined) || d_combined < 1.5) {
    profile <- "matrix_robust"
    regime <- "poor_separation"
  } else if (d_combined < 3.0) {
    profile <- "sensitive"
    regime <- "moderate_separation"
  } else {
    profile <- "standard"
    regime <- "strong_separation"
  }

  list(
    d_mp = d_mp,
    d_ttt = d_ttt,
    d_combined = d_combined,
    regime = regime,
    recommended_profile = profile
  )
}
