% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/wms-all.R
\encoding{UTF-8}
\name{catr_wms_get_layer}
\alias{catr_wms_get_layer}
\title{WMS INSPIRE: Download map images}
\usage{
catr_wms_get_layer(
  x,
  srs = NULL,
  what = c("building", "buildingpart", "parcel", "zoning", "address", "admboundary",
    "admunit"),
  styles = "default",
  update_cache = FALSE,
  cache_dir = NULL,
  verbose = FALSE,
  crop = FALSE,
  options = NULL,
  ...
)
}
\arguments{
\item{x}{See \strong{Bounding box}. It could be:
\itemize{
\item A numeric vector of length 4 with the coordinates that defines
the bounding box: \code{c(xmin, ymin, xmax, ymax)}
\item A \code{sf/sfc} object, as provided by the \CRANpkg{sf} package.
}}

\item{srs}{SRS/CRS to use on the query. To check the admitted values check
\link{catr_srs_values}, specifically the \code{wfs_service} column. See
\strong{Bounding box}.}

\item{what, styles}{Layer and style of the WMS layer to be downloaded. See
\strong{Layers and styles}.}

\item{update_cache}{logical. Should the cached file be refreshed? Default is
\code{FALSE}. When set to \code{TRUE} it would force a new download.}

\item{cache_dir}{A path to a cache directory. On \code{NULL} the function
would store the cached files on a temporary dir (See \code{\link[base:tempfile]{base::tempdir()}}).}

\item{verbose}{logical. If \code{TRUE} displays informational messages.}

\item{crop}{logical. If \code{TRUE}, the results will be cropped to the specified
\code{x} extent. If \code{x} is an \code{\link[sf:sf]{sf}} object with one \code{POINT},
\code{crop} is set to \code{FALSE}. See \code{\link[terra:crop]{terra::crop()}}.}

\item{options}{A named list containing additional options to pass to the
query.}

\item{...}{
  Arguments passed on to \code{\link[mapSpain:esp_get_tiles]{mapSpain::esp_get_tiles}}
  \describe{
    \item{\code{res}}{character string or number. Only valid for WMS providers.
Resolution (in pixels) of the final tile.}
    \item{\code{bbox_expand}}{number. Expansion percentage of the bounding box of \code{x}.}
    \item{\code{transparent}}{logical. Provides transparent background, if supported.}
    \item{\code{mask}}{logical. \code{TRUE} if the result should be masked to \code{x}. See
\code{\link[terra:mask]{terra::mask()}}.}
  }}
}
\value{
A \code{\link[terra:rast]{SpatRaster}} is returned, with 3 (RGB) or 4 (RGBA) layers,
see \code{\link[terra:RGB]{terra::RGB()}}.
}
\description{
Get geotagged images from the Spanish Cadastre. This function is a wrapper of
\code{\link[mapSpain:esp_get_tiles]{mapSpain::esp_get_tiles()}}.
}
\section{Bounding box}{

When \code{x} is a numeric vector, make sure that the \code{srs} matches the
coordinate values. When \code{x} is a \code{\link[sf:sf]{sf}} object, the value
\code{srs} is ignored.

The query is performed using \href{https://epsg.io/3857}{EPSG:3857} (Web Mercator)
and the tile is projected back to the SRS of \code{x}. In case that the tile
looks deformed, try either providing \code{x} or specify the SRS of the requested
tile via the \code{srs} argument, that (ideally) would need to match the SRS of
\code{x}. See \strong{Examples}.
}

\section{Layers and styles}{

\subsection{Layers}{

The argument \code{what} defines the layer to be extracted. The equivalence with
the
\href{https://www.catastro.hacienda.gob.es/webinspire/documentos/inspire-WMS.pdf}{API Docs}
reference is:
\itemize{
\item \code{"parcel"}: CP.CadastralParcel
\item \code{"zoning"}: CP.CadastralZoning
\item \code{"building"}: BU.Building
\item \code{"buildingpart"}: BU.BuildingPart
\item \code{"address"}: AD.Address
\item \code{"admboundary"}: AU.AdministrativeBoundary
\item \code{"admunit"}: AU.AdministrativeUnit
}
}

\subsection{Styles}{

The WMS service provides different styles on each layer (\code{what} argument).
Some of the styles available are:
\itemize{
\item \code{"parcel"}: styles : \code{"BoundariesOnly"}, \code{"ReferencePointOnly"},
\code{"ELFCadastre"}.
\item \code{"zoning"}: styles : \code{"BoundariesOnly"}, \code{"ELFCadastre"}.
\item \code{"building"}, \code{"buildingpart"}: \code{"ELFCadastre"}
\item \code{"address"}: \code{"Number.ELFCadastre"}
\item \code{"admboundary"}, \code{"admunit"}: \code{"ELFCadastre"}
}

Check the \href{https://www.catastro.hacienda.gob.es/webinspire/documentos/inspire-WMS.pdf}{API Docs}
for more information.
}
}

\examples{
\dontshow{if (run_example()) withAutoprint(\{ # examplesIf}
\donttest{

# With a bbox

pict <- catr_wms_get_layer(
  c(222500, 4019500, 223700, 4020700),
  srs = 25830,
  what = "parcel"
)

library(mapSpain)
library(ggplot2)
library(tidyterra)

ggplot() +
  geom_spatraster_rgb(data = pict)

# With a spatial object

parcels <- catr_wfs_get_parcels_neigh_parcel("3662303TF3136B", srs = 25830)

# Use styles

parcels_img <- catr_wms_get_layer(parcels,
  what = "buildingpart",
  srs = 25830, # As parcels object
  bbox_expand = 0.3,
  styles = "ELFCadastre"
)

ggplot() +
  geom_sf(data = parcels, fill = "blue", alpha = 0.5) +
  geom_spatraster_rgb(data = parcels_img)
}
\dontshow{\}) # examplesIf}
}
\seealso{
\code{\link[mapSpain:esp_get_tiles]{mapSpain::esp_get_tiles()}} and \code{\link[terra:RGB]{terra::RGB()}}. For plotting see
\code{\link[terra:plotRGB]{terra::plotRGB()}} and \code{\link[tidyterra:geom_spatraster_rgb]{tidyterra::geom_spatraster_rgb()}}.

INSPIRE API functions:
\code{\link{catr_atom_get_address}()},
\code{\link{catr_atom_get_address_db_all}()},
\code{\link{catr_atom_get_buildings}()},
\code{\link{catr_atom_get_buildings_db_all}()},
\code{\link{catr_atom_get_parcels}()},
\code{\link{catr_atom_get_parcels_db_all}()},
\code{\link{catr_wfs_get_address_bbox}()},
\code{\link{catr_wfs_get_buildings_bbox}()},
\code{\link{catr_wfs_get_parcels_bbox}()},
\code{\link{inspire_wfs_get}()}

Other spatial: 
\code{\link{catr_atom_get_address}()},
\code{\link{catr_atom_get_buildings}()},
\code{\link{catr_atom_get_parcels}()},
\code{\link{catr_wfs_get_address_bbox}()},
\code{\link{catr_wfs_get_buildings_bbox}()},
\code{\link{catr_wfs_get_parcels_bbox}()}
}
\concept{INSPIRE}
\concept{WMS}
\concept{spatial}
