#' Generate Samples from Target Distributions
#'
#' This function draws samples directly from the target probability distributions 
#' specified in a \code{ccm_sample} object. These samples serve as a "ground truth" 
#' to evaluate whether the MCMC chain has converged to the intended target.
#'
#' @param object An object of class \code{ccm_sample} generated by \code{\link{sample_ccm}}.
#' @param n_sim Integer. The number of independent samples to draw from the 
#'   theoretical target distributions. Default is equal to the number of CCM samples.
#'
#' @return The input \code{ccm_sample} object with the \code{theoretical} slot 
#'   populated. This slot contains a data frame of statistics sampled directly 
#'   from the target distributions.
#'
#' @details 
#' This function performs direct i.i.d. sampling (e.g., using \code{rpois}, 
#' \code{rnorm}, etc.) based on the parameters stored in the \code{ccm_sample} 
#' object. It does not use MCMC. The resulting samples are used by 
#' \code{\link{plot.ccm_sample}} when \code{include_theoretical = TRUE} is specified.
#'
#' @examples
#' # 1. Generate MCMC samples
#' ccm_sample <- sample_ccm(
#'   network_stats = "edges",
#'   prob_distr = "poisson",
#'   prob_distr_params = list(list(350)),
#'   population = 50 
#' )
#' 
#' # 2. Generate theoretical samples for comparison
#' ccm_sample <- sample_theoretical(ccm_sample, n_sim = 1000)
#' 
#' # 3. Compare MCMC to theoretical target
#' plot(ccm_sample, stats = "edges", type = "hist", include_theoretical = TRUE)
#'
#' @export

sample_theoretical <- function(
    object,
    n_sim = nrow(object$mcmc_stats)
) {
  stat <- object$network_stats

  #---------------------------
  # Network Property: edges
  #---------------------------
  if (length(stat) == 1 && stat == "edges") {
    return(CCM_theoretical_check_edges(object,
                                       n_sim))
  }
  
  #---------------------------
  # Network Property: Density
  #---------------------------
  if (length(stat) == 1 && stat == "density") {
    return(CCM_theoretical_check_density(object,
                                       n_sim))
  }
  
  #---------------------------
  # Network Property: Mixing
  #---------------------------
  if (length(stat) == 1 && stat == "mixing") {
    return(CCM_theoretical_check_mixing(object,
                                       n_sim))
  }
  
  #---------------------------
  # Network Property: Degree
  #---------------------------
  if (length(stat) == 1 && (stat == "degree" || stat == "degreedist")) {
    return(CCM_theoretical_check_degree(object,
                                        n_sim))
  }
  
  if ((length(stat) == 1 && stat == "degmix") || (length(stat) == 1 && stat == "degmixing"))  {
    return(CCM_theoretical_check_degmix(object,
                                        n_sim))
  }
  
  if ((length(stat) == 1 && stat == "degmix_clustering") || (length(stat) == 2 && stat[1] == "degmixing" && stat[2] == "triangles")) {
    return(CCM_theoretical_check_degmixclustering(object,
                                                  n_sim))
  }
  
  if ((length(stat) == 2 && stat[1] == "degreedist" && stat[2] == "mixing")) {
    return(CCM_theoretical_check_degree_mixing(object,
                                                  n_sim))
  }
  
  stop("Theoretical distribution not implemented for this statistic.")
}
