% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/misc_and_utility.R
\name{umx_make_TwinData}
\alias{umx_make_TwinData}
\title{Simulate twin data with control over A, C, and E parameters, as well as moderation of A.}
\usage{
umx_make_TwinData(nMZpairs, nDZpairs = nMZpairs, AA = NULL,
  CC = NULL, EE = NULL, varNames = "var", mean = 0, sd = 1,
  nThresh = NULL, sum2one = TRUE, seed = NULL, empirical = FALSE,
  MZr = NULL, DZr = MZr, Amod = NULL, Cmod = NULL, Emod = NULL)
}
\arguments{
\item{nMZpairs}{Number of MZ pairs to simulate}

\item{nDZpairs}{Number of DZ pairs to simulate (defaults to nMZpairs)}

\item{AA}{value for A variance. NOTE: See options for use in GxE and Bivariate GxE}

\item{CC}{value for C variance.}

\item{EE}{value for E variance.}

\item{varNames}{name for variables (defaults to 'var')}

\item{mean}{mean for traits (default = 0) (not applied to moderated cases)}

\item{sd}{sd of traits (default = 1) (not applied to moderated cases)}

\item{nThresh}{If supplied, use as thresholds and return mxFactor output? (default is not to)}

\item{sum2one}{Whether to enforce AA + CC + EE summing the one (default = TRUE)}

\item{seed}{Allows user to set.seed() if wanting reproducible dataset}

\item{empirical}{Passed to mvrnorm}

\item{MZr}{If MZr and DZr are set (default = NULL), the function returns dataframes of the request n and correlation.}

\item{DZr}{NULL}

\item{Amod}{Used for Bivariate GxE data: list(Beta_a1 = .025, Beta_a2 = .025)}

\item{Cmod}{Used for Bivariate GxE data: list(Beta_c1 = .025, Beta_c2 = .025)}

\item{Emod}{Used for Bivariate GxE data: list(Beta_e1 = .025, Beta_e2 = .025)}
}
\value{
\itemize{
\item list of mzData and dzData dataframes containing T1 and T2 plus, if needed M1 and M2 (moderator values)
}
}
\description{
Makes MZ and DZ twin data, optionally with moderated A. By default, the three variance components must sum to 1.

See examples for how to use this: it is pretty flexible).

If you provide 2 varNames, they will be used for twin 1 and twin 2. If you provide one, it will be expanded to var_T1 and var_T2

You supply the number of pairs of each zygosity that wish to simulate (nMZpairs, nDZpairs), along with the values of AA, CC,and EE.

\emph{Note}, if you want a power calculator, see \href{https://www.people.vcu.edu/~bverhulst/power/power.html}{here}.

\strong{Shortcuts}

You can omit nDZpairs. You can also give any 2 of A, C, or E and the function will add the value which makes the ACE total = 1.

\strong{Moderation}
\strong{Univariate GxE Data}
AA can take a list c(avg = .5, min = 0, max = 1). If specified will act like a moderated heritability, with average value = avg, and swinging
down to min and up to max across 3 SDs of the moderator.

\strong{Bivariate GxE Data}

To simulate data with a moderator that is not shared by both twins.
Moderated heritability is specified via the bivariate relationship (AA, CC, EE) and two moderators in each component.
AA   = list(a11 = .4, a12 = .1, a22 = .15)
CC   = list(c11 = .2, c12 = .1, c22 = .10)
EE   = list(e11 = .4, e12 = .3, e22 = .25)
Amod = list(Beta_a1 = .025, Beta_a2 = .025)
Cmod = list(Beta_c1 = .025, Beta_c2 = .025)
Emod = list(Beta_e1 = .025, Beta_e2 = .025)
}
\examples{
# =====================================================================
# = Basic Example, with all elements of std univariate data specified =
# =====================================================================
tmp = umx_make_TwinData(nMZpairs = 10000, AA = .30, CC = .00, EE = .70)
# Show list of 2 data sets
str(tmp)
# = How to consume the built datasets =
mzData = tmp[[1]];
dzData = tmp[[2]];
cov(mzData); cov(dzData)
umxAPA(mzData)
str(mzData); str(dzData); 

# Prefer to work in path coefficient values? (little a?)
tmp = umx_make_TwinData(200, AA = .6^2, CC = .2^2)
# Check the correlations
umxAPA(tmp[[1]]); umxAPA(tmp[[2]])

# =============
# = Shortcuts =
# =============

# Omit nDZpairs (equal numbers of both by default)
tmp = umx_make_TwinData(nMZpairs = 100, nDZpairs = 100, AA = .36, CC = .04, EE = .60)
tmp = umx_make_TwinData(100, AA = 0.5, CC = 0.3) # omit any one of A, C, or E (sums to 1)
cov(tmp[[1]])
# Not limited to unit variance
tmp = umx_make_TwinData(100, AA = 3, CC = 2, EE = 3, sum2one = FALSE) 
cov(tmp[[1]])

# =====================
# = Moderator Example =
# =====================

x = umx_make_TwinData(100, AA = c(avg = .7, min = 0, max = 1), CC = .55, EE = .63)
str(x)

# =====================
# = Threshold Example =
# =====================
tmp = umx_make_TwinData(100, AA = .6, CC = .2, nThresh = 3)
str(tmp)
umxAPA(tmp[[1]]); umxAPA(tmp[[2]])


# ========================
# = Just use MZr and DZr =
# ========================
tmp = umx_make_TwinData(100, MZr = .86, DZr= .60, varNames = "IQ")
umxAPA(tmp[[1]]); umxAPA(tmp[[2]])

# Bivariate GxSES example (see umxGxEbiv)

AA   = list(a11 = .4, a12 = .1, a22 = .15)
CC   = list(c11 = .2, c12 = .1, c22 = .10)
EE   = list(e11 = .4, e12 = .3, e22 = .25)
Amod = list(Beta_a1 = .025, Beta_a2 = .025)
Cmod = list(Beta_c1 = .025, Beta_c2 = .025)
Emod = list(Beta_e1 = .025, Beta_e2 = .025)
tmp = umx_make_TwinData(5000, AA =AA, CC = CC, EE = EE, Amod = Amod, Cmod =Cmod, Emod =Emod)
# List of 2
#  $ mzData:'data.frame':	5000 obs. of  6 variables:
#   ..$ defM_T1: num [1:5000] -1.424 -1.606 -0.749 -0.358 -0.208 ...
#   ..$ defM_T2: num [1:5000] -1.703 -1.125 -1.136 0.366 -0.864 ...
#   ..$ M_T1   : num [1:5000] -1.424 -1.606 -0.749 -0.358 -0.208 ...
#   ..$ var_T1 : num [1:5000] -1.1197 -0.3074 -0.4156 -0.4149 0.0739 ...
#   ..$ M_T2   : num [1:5000] -1.703 -1.125 -1.136 0.366 -0.864 ...
#   ..$ var_T2 : num [1:5000] 0.381 -1.275 -1.114 1.297 -1.53 ...
#  $ dzData:'data.frame':	5000 obs. of  6 variables:
#   ..$ defM_T1: num [1:5000] 0.253 -0.402 0.545 -1.494 -0.278 ...
#   ..$ defM_T2: num [1:5000] 1.7587 0.3025 -0.3864 0.0737 0.514 ...
#   ..$ M_T1   : num [1:5000] 0.253 -0.402 0.545 -1.494 -0.278 ...
#   ..$ var_T1 : num [1:5000] -0.835 -0.305 -0.299 -1.576 -0.26 ...
#   ..$ M_T2   : num [1:5000] 1.7587 0.3025 -0.3864 0.0737 0.514 ...
#   ..$ var_T2 : num [1:5000] -0.418 0.678 -0.78 -0.312 -0.272 ...

# TODO tmx example showing how moderation of A introduces heteroskedasticity in a regression model.
# More residual variance at one extreme of the x axis (moderator) 
# m1 = lm(var_T1~ M_T1, data = x); 
# x = rbind(tmp[[1]], tmp[[2]])
# plot(residuals(m1)~ x$M_T1, data=x)
}
\references{
\itemize{
\item \url{https://github.com/tbates/umx}, \url{https://tbates.github.io}
}
}
\seealso{
\itemize{
\item \code{\link{umx_make_TwinData}}, \code{\link{umxGxEbiv}}, \code{\link{umxACE}}, \code{\link{umxGxE}}
}

Other Twin Data functions: \code{\link{umx_long2wide}},
  \code{\link{umx_residualize}},
  \code{\link{umx_scale_wide_twin_data}},
  \code{\link{umx_wide2long}}, \code{\link{umx}}

Other Data Functions: \code{\link{umxCovData}},
  \code{\link{umxHetCor}},
  \code{\link{umxPadAndPruneForDefVars}},
  \code{\link{umx_as_numeric}}, \code{\link{umx_cov2raw}},
  \code{\link{umx_lower2full}},
  \code{\link{umx_make_MR_data}},
  \code{\link{umx_make_bin_cont_pair_data}},
  \code{\link{umx_make_fake_data}},
  \code{\link{umx_polychoric}},
  \code{\link{umx_polypairwise}},
  \code{\link{umx_polytriowise}},
  \code{\link{umx_read_lower}}, \code{\link{umx_rename}},
  \code{\link{umx_reorder}}, \code{\link{umx_stack}},
  \code{\link{umx_swap_a_block}}, \code{\link{umx}}
}
\concept{Data Functions}
\concept{Twin Data functions}
