% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/plot-acf_diagnostics.R
\name{plot_acf_diagnostics}
\alias{plot_acf_diagnostics}
\title{Visualize the ACF, PACF, and CCFs for One or More Time Series}
\usage{
plot_acf_diagnostics(
  .data,
  .date_var,
  .value,
  ...,
  .lags = 0:20,
  .facet_ncol = 1,
  .facet_scales = "fixed",
  .line_color = "#2c3e50",
  .line_size = 0.5,
  .line_alpha = 1,
  .point_color = "#2c3e50",
  .point_size = 1,
  .point_alpha = 1,
  .hline_color = "#2c3e50",
  .title = "Lag Diagnostics",
  .x_lab = "Lag",
  .y_lab = "Correlation",
  .interactive = TRUE,
  .plotly_slider = FALSE
)
}
\arguments{
\item{.data}{A data frame or tibble with numeric features (values) in descending
chronological order}

\item{.date_var}{A column containing either date or date-time values}

\item{.value}{A numeric column with a value to have ACF and PACF calculations
performed.}

\item{...}{Additional features to perform Lag Cross Correlations (CCFs)
versus the \code{.value}. Useful for evaluating external lagged regressors.}

\item{.lags}{A seqence of one or more lags to evaluate.}

\item{.facet_ncol}{Facets: Number of facet columns. Has no effect if using \code{grouped_df}.}

\item{.facet_scales}{Facets: Options include "fixed", "free", "free_y", "free_x"}

\item{.line_color}{Line color. Use keyword: "scale_color" to change the color by the facet.}

\item{.line_size}{Line size}

\item{.line_alpha}{Line opacity. Adjust the transparency of the line. Range: (0, 1)}

\item{.point_color}{Point color. Use keyword: "scale_color" to change the color by the facet.}

\item{.point_size}{Point size}

\item{.point_alpha}{Opacity. Adjust the transparency of the points. Range: (0, 1)}

\item{.hline_color}{Color for the y-intercept = 0 line.}

\item{.title}{Title for the plot}

\item{.x_lab}{X-axis label for the plot}

\item{.y_lab}{Y-axis label for the plot}

\item{.interactive}{Returns either a static (\code{ggplot2}) visualization or an interactive (\code{plotly}) visualization}

\item{.plotly_slider}{If TRUE, returns a plotly x-axis range slider.}
}
\value{
A static \code{ggplot2} plot or an interactive \code{plotly} plot
}
\description{
Returns the \strong{ACF and PACF of a target} and
optionally \strong{CCF's of one or more lagged predictors} in interactive \code{plotly} plots. Scales
to multiple time series with \code{group_by()}.
}
\details{
\strong{Simplified ACF, PACF, & CCF}

We are often interested in all 3 of these functions. Why not get all 3 at once?
Now you can.
\itemize{
\item \strong{ACF} - Autocorrelation between a target variable and lagged versions of itself
\item \strong{PACF} - Partial Autocorrelation removes the dependence of lags on
other lags highlighting key seasonalities.
\item \strong{CCF} - Shows how lagged predictors can be used for prediction of a target
variable.
}

\strong{Scales to Multiple Time Series with Groupes}

The \code{plot_acf_diagnostics()} works with \code{grouped_df}'s, meaning you can
group your time series by one or more categorical columns with \code{dplyr::group_by()}
and then apply \code{plot_acf_diagnostics()} to return group-wise lag diagnostics.

\strong{Special Note on Dots (...)}

Unlike other plotting utilities, the \code{...} arguments is NOT used for
group-wise analysis. Rather, it's used for processing Cross Correlations (CCFs).

Use \code{dplyr::group_by()} for processing multiple time series groups.
}
\examples{
library(tidyverse)
library(timetk)


# Apply Transformations
# - Differencing transformation to identify ARIMA & SARIMA Orders
m4_hourly \%>\%
    group_by(id) \%>\%
    plot_acf_diagnostics(
        date, value,               # ACF & PACF
        .lags = 0:(24*7),          # 7-Days of hourly lags
        .interactive = FALSE
    )

# Apply Transformations
# - Differencing transformation to identify ARIMA & SARIMA Orders
m4_hourly \%>\%
    group_by(id) \%>\%
    plot_acf_diagnostics(
        date,
        diff_vec(value, lag = 1), # Difference the value column
        .lags        = 0:(24*7),   # 7-Days of hourly lags
        .interactive = FALSE
    ) +
    ggtitle("ACF Diagnostics",  subtitle = "1st Difference")

# CCFs Too!
walmart_sales_weekly \%>\%
    select(id, Date, Weekly_Sales, Temperature, Fuel_Price) \%>\%
    group_by(id) \%>\%
    plot_acf_diagnostics(
        Date, Weekly_Sales,        # ACF & PACF
        Temperature, Fuel_Price,   # CCFs
        .lags        = 0:(4*3),    # 3 months of weekly lags
        .interactive = FALSE
    )

}
\seealso{
\itemize{
\item \strong{Visualizing ACF, PACF, & CCF:} \code{\link[=plot_acf_diagnostics]{plot_acf_diagnostics()}}
\item \strong{Visualizing Seasonality:} \code{\link[=plot_seasonal_diagnostics]{plot_seasonal_diagnostics()}}
\item \strong{Visualizing Time Series:} \code{\link[=plot_time_series]{plot_time_series()}}
}
}
