% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/raptor.R
\name{raptor}
\alias{raptor}
\title{Calculate travel times from one stop to all reachable stops}
\usage{
raptor(stop_times, transfers, from_stop_ids, departure_time_range = 3600,
  max_transfers = NULL, keep = "all")
}
\arguments{
\item{stop_times}{A (prepared) stop_times table from a gtfs feed. Prepared means
that all stop time rows before the desired journey departure time
should be removed. The table should also only include departures
happening on one day. Use \code{filter_stop_times} for easier preparation.}

\item{transfers}{Transfers table from a gtfs feed. In general no preparation is needed.}

\item{from_stop_ids}{Atomic char vector with stop_ids from where a journey should start}

\item{departure_time_range}{All departures from the first departure of
stop_times (not necessarily a from_stop) within
\code{departure_time_range} (in seconds) are considered.}

\item{max_transfers}{Maximum number of transfers allowed, no limit (NULL) as default.}

\item{keep}{One of c("all", "shortest", "earliest"). By default \code{all} journeys
arriving at a stop are returned. With \code{shortest} the
journey with shortest travel time is returned. With \code{earliest} the
journey arriving at a stop the earliest is returned.}
}
\value{
By default a table with travel times to all stop_ids reachable from
\code{from_stop_ids} and their corresponding journey departure and arrival times.
}
\description{
\code{raptor} finds the minimal travel time and/or earliest arrival time for all
stops in \code{stop_times} with journeys departing from \code{from_stop_ids} within
\code{departure_time_range}.
}
\details{
With a modified \href{https://www.microsoft.com/en-us/research/publication/round-based-public-transit-routing}{Round-Based Public Transit Routing Algorithm}
(RAPTOR) using data.table earliest arrival times for all stops are calculated. If two
journeys arrive at the same time, the one with the later departure time and thus shorter
travel time is kept. By default, all journeys within \code{departure_time_range} that arrive
at a stop are returned in a table. Journeys are defined by a departure stop_id, a
departure, arrival and travel time. Note that the exact journeys (with each intermediate
stop and route id for example) is \emph{not} returned.

For most cases \code{stop_times} needs to be filtered as it should only contain
trips happening on a single day and departures later than a given journey
start time (see \code{filter_stop_times}).

The algorithm scans all trips until it exceeds \code{max_transfers} or all trips
in \code{stop_times} have been visited.
}
\examples{
\donttest{
nyc_path <- system.file("extdata", "google_transit_nyc_subway.zip", package = "tidytransit")
nyc <- read_gtfs(nyc_path, local=TRUE)

# you can use initial walk times to different stops in walking distance (arbitrary example values)
stop_ids_harlem_st <- c("301", "301N", "301S")
stop_ids_155_st <- c("A11", "A11N", "A11S", "D12", "D12N", "D12S")
walk_times <- data.frame(stop_id = c(stop_ids_harlem_st, stop_ids_155_st), 
                         walk_time = c(rep(600, 3), rep(410, 6)), stringsAsFactors = F)

# Use journeys departing after 7 AM with arrival time before 11 AM on 26th of June
stop_times <- filter_stop_times(nyc, "2018-06-26", 7*3600, 9*3600)

# calculate all journeys departing from Harlem St or 155 St between 7:00 and 7:30
rptr <- raptor(stop_times, nyc$transfers, walk_times$stop_id, departure_time_range = 1800,
               keep = "all")

# add walk times to travel times
rptr <- left_join(rptr, walk_times, by=c("journey_departure_stop_id" = "stop_id"))
rptr$travel_time_incl_walk <- rptr$travel_time + rptr$walk_time

# get minimal travel times (with walk times) for all stop_ids 
shortest_travel_times <- setDT(rptr)[order(travel_time_incl_walk)][, .SD[1], by = "stop_id"]
hist(shortest_travel_times$travel_time, breaks = 360)
}
}
