% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/taskscheduleR.R
\name{taskscheduler_create}
\alias{taskscheduler_create}
\title{Schedule an R script with the Windows task scheduler.}
\usage{
taskscheduler_create(
  taskname = basename(rscript),
  rscript,
  schedule = c("ONCE", "MONTHLY", "WEEKLY", "DAILY", "HOURLY", "MINUTE", "ONLOGON",
    "ONIDLE"),
  starttime = format(Sys.time() + 62, "\%H:\%M"),
  startdate = format(Sys.Date(), "\%d/\%m/\%Y"),
  days = c("*", "MON", "TUE", "WED", "THU", "FRI", "SAT", "SUN", 1:31),
  months = c("*", "JAN", "FEB", "MAR", "APR", "MAY", "JUN", "JUL", "AUG", "SEP", "OCT",
    "NOV", "DEC"),
  modifier,
  idletime = 60L,
  Rexe = file.path(Sys.getenv("R_HOME"), "bin", "Rscript.exe"),
  rscript_args = "",
  rscript_options = "",
  schtasks_extra = "",
  debug = FALSE
)
}
\arguments{
\item{taskname}{a character string with the name of the task. Defaults to the filename. Should not contain any spaces.}

\item{rscript}{the full path to the .R script with the R code to execute. Should not contain any spaces.}

\item{schedule}{when to schedule the \code{rscript}. 
Either one of 'ONCE', 'MONTHLY', 'WEEKLY', 'DAILY', 'HOURLY', 'MINUTE', 'ONLOGON', 'ONIDLE'.}

\item{starttime}{a timepoint in HH:mm format indicating when to run the script. Defaults to within 62 seconds.}

\item{startdate}{a date that specifies the first date on which to run the task.
Only applicable if schedule is of type 'MONTHLY', 'WEEKLY', 'DAILY', 'HOURLY', 'MINUTE'. Defaults to today in '\%d/\%m/\%Y' format. Change to your locale format if needed.}

\item{days}{character string with days on which to run the script if schedule is 'WEEKLY' or 'MONTHLY'. Possible values
are * (all days). For weekly: 'MON', 'TUE', 'WED', 'THU', 'FRI', 'SAT', 'SUN' or a vector of these in your locale.
For monthly: 1:31 or a vector of these.}

\item{months}{character string with months on which to run the script if schedule is 'MONTHLY'. Possible values
are * (all months) or 'JAN', 'FEB', 'MAR', 'APR', 'MAY', 'JUN', 'JUL', 'AUG', 'SEP', 'OCT', 'NOV', 'DEC' or a vector of these in your locale.}

\item{modifier}{a modifier to apply. See the docs/schtasks.pdf}

\item{idletime}{integer containing a value that specifies the amount of idle time to wait before 
running a scheduled ONIDLE task. The valid range is 1 - 999 minutes.}

\item{Rexe}{path to Rscript.exe which will be used to run the script. Defaults to Rscript at the bin folder of R_HOME.}

\item{rscript_args}{character string with further arguments passed on to Rscript. See args in \code{\link{Rscript}}.}

\item{rscript_options}{character string with further options passed on to Rscript. See options in \code{\link{Rscript}}.}

\item{schtasks_extra}{character string with further schtasks arguments. See the inst/docs/schtasks.pdf}

\item{debug}{logical to print the system call to screen}
}
\value{
the system call to schtasks /Create
}
\description{
Schedule an R script with the Windows task scheduler. E.g. daily, weekly, once, at startup, ...
More information about the scheduling format can be found in the docs/schtasks.pdf file inside this package.
The rscript file will be scheduled with Rscript.exe and the log of the run will be put in the .log file which can be found in the same directory
as the location of the rscript
}
\examples{
myscript <- system.file("extdata", "helloworld.R", package = "taskscheduleR")
cat(readLines(myscript), sep = "\n")

\dontrun{
## Run script once at a specific timepoint (within 62 seconds)
runon <- format(Sys.time() + 62, "\%H:\%M")
taskscheduler_create(taskname = "myfancyscript", rscript = myscript, 
 schedule = "ONCE", starttime = runon)
 
## Run every day at the same time on 09:10, starting from tomorrow on
## Mark: change the format of startdate to your locale if needed (e.g. US: \%m/\%d/\%Y)
taskscheduler_create(taskname = "myfancyscriptdaily", rscript = myscript, 
 schedule = "DAILY", starttime = "09:10", startdate = format(Sys.Date()+1, "\%d/\%m/\%Y"))
 
## Run every week on Sunday at 09:10
taskscheduler_create(taskname = "myfancyscript_sun", rscript = myscript, 
  schedule = "WEEKLY", starttime = "09:10", days = 'SUN')

## Run every 5 minutes, starting from 10:40
taskscheduler_create(taskname = "myfancyscript_5min", rscript = myscript,
  schedule = "MINUTE", starttime = "10:40", modifier = 5)
  
## Run every minute, giving some command line arguments which can be used in the script itself
taskscheduler_create(taskname = "myfancyscript_withargs_a", rscript = myscript,
  schedule = "MINUTE", rscript_args = "productxyz 20160101")
taskscheduler_create(taskname = "myfancyscript_withargs_b", rscript = myscript,
  schedule = "MINUTE", rscript_args = c("productabc", "20150101"))
  
alltasks <- taskscheduler_ls()
subset(alltasks, TaskName \%in\% c("myfancyscript", "myfancyscriptdaily"))
# The field TaskName might have been different on Windows with non-english language locale

taskscheduler_delete(taskname = "myfancyscript")
taskscheduler_delete(taskname = "myfancyscriptdaily")
taskscheduler_delete(taskname = "myfancyscript_sun")
taskscheduler_delete(taskname = "myfancyscript_5min")
taskscheduler_delete(taskname = "myfancyscript_withargs_a")
taskscheduler_delete(taskname = "myfancyscript_withargs_b")

## Have a look at the log
mylog <- system.file("extdata", "helloworld.log", package = "taskscheduleR")
cat(readLines(mylog), sep = "\n")
}
}
