#' @title
#' pin_convert
#' 
#' @description
#' Converts one pin to standard format
#' 
#' @param pin A character element of length one.
#' @param format Which format should be converted. See \link{as.pin}.
#'  
#' @return
#' Character element on standard format.
#' 
#'

pin_convert <- function(pin, format){
  if(length(pin) == 0) return(pin)
  switch(EXPR = format,
         "1" = pin,
         "2" = paste(as.character(substr(pin, 1, 8)), as.character(substr(pin, 10, 13)) ,sep=""),
         "3" = paste(ifelse(substr(pin,start=7,7) == "-",
                             as.character(pin_century(pin)),
                             as.character(pin_century(pin)-1)), 
                      substr(pin, 1, 6),
                      substr(pin, 8, 11), sep=""),
         "4" = paste(as.character(pin_century(pin)), pin ,sep=""))
}

#' @title
#' pin_century
#' 
#' @description
#' Assuming that the person is less than 100 years old, calculate the century of birth.
#' 
#' @param pin_short Vector with swedish personal identity numbers on standard format.
#' 
#' @return
#' Century vector in numeric format
#' 
pin_century <- function(pin_short){
  pin_date <- as.Date(paste(paste(substr(Sys.Date(),1,2), substr(pin_short, 1,2), sep=""),
        substr(pin_short, 3,4), substr(pin_short, 5,6), sep="-"))
  ifelse(pin_date > Sys.Date(), 
         as.numeric(substr(Sys.Date(),1,2))-1,
         as.numeric(substr(Sys.Date(),1,2)))
}

#' @title
#' pin_coordn_correct
#' 
#' @description
#' Remove the change of day in coordination numbers (to enable age calculation).
#' 
#' @param pin Vector of pins at format atandard fromat 'YYYYMMDDNNNC'. See \link{as.pin}.
#' 
#' @return
#' Character vector with pin, corrected for coordination numbers.
#' 
pin_coordn_correct <- function(pin){
  coordn <- pin_coordn(pin)
  pin <- ifelse(coordn,
                paste(
                  substr(pin,1,6), 
                  as.character(as.numeric(substr(pin,7,7)) - 6),
                  substr(pin, 8, 12), sep=""), pin)
  return(pin)
}



#' @title
#' pin_birthplace_internal
#' 
#' @description
#' Internal computation of birthplace (one for each pin)
#' 
#' @param pin Character element with pin at standard format 'YYYYMMDDNNNC'. See \link{as.pin}.
#' @param birth_vector Vector mapping birth number to birthplace. See \link{pin_birthplace}.
#' 
#' @return
#' Character element containing birthplace
pin_birthplace_internal <- function(pin, birth_vector){
  if(is.na(pin)) return(pin)
  born <- as.numeric(substr(pin, 1, 4))
  if(born >= 1990){
    res <- "Born after 31 december 1989"
  } else {
    birth_number <- as.numeric(substr(pin, 9, 10))
    if(born >= 1968 & birth_number <= 13) {
      res <- birth_vector[11]
    } else {
      res <- birth_vector[as.numeric(substr(pin, 9, 10)) + 1]
    }
  }
  return(res)
}



#' @title
#' luhn_algo
#' 
#' @description
#' Calculates the control number.
#' 
#' @param id Element with swedish personal identity number.
#' @param multiplier What should each element in id be multiplied with
#' 
#' @references 
#' https://www.skatteverket.se/download/18.8dcbbe4142d38302d74be9/1387372677724/717B06.pdf
#' https://www.skatteverket.se/download/18.1e6d5f87115319ffba380001857/1285595720207/70408.pdf
#' 
#' @return
#' Logical element indicating if a pin is correct (TRUE) or not (FALSE)
#'
luhn_algo <- function(id, multiplier = c(0, 0, 2, 1, 2, 1, 2, 1, 2, 1, 2, 0)){
  ret <- as.numeric(unlist(strsplit(id, split="")))
  calc <- ret * multiplier
  sumValue <- sum(calc %% 10) + sum(calc %/% 10)
  output <- as.integer((10 - sumValue %% 10) %% 10)
  return(output)
}


#' @title
#' pin_add_zero
#' 
#' @description
#' Adds zeroes in front of pin that are converted from numeric
#' 
#' @param pin Personal identity number
#' 
#' @return
#' Corrected pin
#' 
pin_add_zero <- function(pin){
  elem <- 12 - nchar(pin)
  if(elem > 2) {
    paste(paste(rep("0", elem - 2),collapse=""), pin, sep="")
  } else {
    pin
  }
}
