% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/eval_design_custom_mc.R
\name{eval_design_custom_mc}
\alias{eval_design_custom_mc}
\title{Monte Carlo power evaluation for experimental designs with user-supplied libraries}
\usage{
eval_design_custom_mc(RunMatrix, model, alpha, nsim, rfunction, fitfunction,
  pvalfunction, anticoef, effectsize = 2, contrasts = contr.sum,
  coef_function = coef, parameternames = NULL, parallel = FALSE,
  parallelpackages = NULL)
}
\arguments{
\item{RunMatrix}{The run matrix of the design. Internally, \code{eval_design_custom_mc} rescales each numeric column
to the range [-1, 1].}

\item{model}{The statistical model used to fit the data.}

\item{alpha}{The type-I error.}

\item{nsim}{The number of simulations.}

\item{rfunction}{Random number generator function. Should be a function of the form f(X,b), where X is the
model matrix and b are the anticipated coefficients.}

\item{fitfunction}{Function used to fit the data. Should be of the form f(formula, X, contrasts)
where X is the model matrix. If contrasts do not need to be specified for the user supplied
library, that argument can be ignored.}

\item{pvalfunction}{Function that returns a vector of p-values from the object returned from the fitfunction.}

\item{anticoef}{The anticipated coefficients for calculating the power. If missing, coefficients will be
automatically generated based on \code{effectsize}.}

\item{effectsize}{The signal-to-noise ratio. Default 2. For a gaussian model, and for
continuous factors, this specifies the difference in response between the highest
and lowest levels of a factor (which are +1 and -1 after normalization).
More precisely: If you do not specify \code{anticoef}, the anticipated coefficients will be
half of \code{effectsize}. If you do specify \code{anticoef}, \code{effectsize} will be ignored.}

\item{contrasts}{Default \code{contr.sum}. Function used to generate the contrasts encoding for categorical variables. If the user has specified their own contrasts
for a categorical factor using the contrasts function, those will be used. Otherwise, skpr will use contr.sum.}

\item{coef_function}{Function that, when applied to a fitfunction return object, returns the estimated coefficients.}

\item{parameternames}{Vector of parameter names if the coefficients do not correspond simply to the columns in the model matrix
(e.g. coefficients from an MLE fit).}

\item{parallel}{If TRUE, uses all cores available to speed up computation of power. Default FALSE.}

\item{parallelpackages}{A vector of strings listing the external packages to be input into the parallel package.}
}
\value{
A data frame consisting of the parameters and their powers. The parameter estimates from the simulations are
stored in the 'estimates' attribute.
}
\description{
Evaluates the power of an experimental design, given its run matrix and the
statistical model to be fit to the data, using monte carlo simulation. Simulated data is fit using a
user-supplied fitting library and power is estimated by the fraction of times a parameter is significant. Returns
a data frame of parameter powers.
}
\examples{
#To demonstrate how a user can use their own libraries for Monte Carlo power generation,
#We will recreate eval_design_survival_mc using the eval_design_custom_mc framework.

#To begin, first let us generate the same design and random generation function shown in the
#eval_design_survival_mc examples:

basicdesign = expand.grid(a=c(-1,1))
design = gen_design(candidateset=basicdesign,model=~a,trials=100,
                         optimality="D",repeats=100)

#Random number generating function

rsurvival = function(X,b) {
 Y = rexp(n=nrow(X),rate=exp(-(X \%*\% b)))
 censored = Y > 1
 Y[censored] = 1
 return(survival::Surv(time=Y,event=!censored,type="right"))
}

#We now need to tell the package how we want to fit our data,
#given the formula and the model matrix X (and, if needed, the list of contrasts).
#If the contrasts aren't required, "contrastslist" should be set to NULL.
#This should return some type of fit object.

fitsurv = function(formula, X, contrastslist=NULL) {
 return(survival::survreg(formula, data=X,dist="exponential"))
}


#We now need to tell the package how to extract the p-values from the fit object returned
#from the fit function. This is how to extract the p-values from the survreg fit object:

pvalsurv = function(fit) {
 return(summary(fit)$table[,4])
}

#And now we evaluate the design, passing the fitting function and p-value extracting function
#in along with the standard inputs for eval_design_mc.

d=eval_design_custom_mc(RunMatrix=design,model=~a,alpha=0.05,nsim=100,
                     fitfunction=fitsurv, pvalfunction=pvalsurv, rfunction=rsurvival, effectsize=1)

#This has the exact same behavior as eval_design_survival_mc for the exponential distribution.
}
