% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/icc.R
\name{icc}
\alias{icc}
\alias{icc.stanreg}
\alias{icc.brmsfit}
\title{Intraclass-Correlation Coefficient}
\usage{
icc(x, ...)

\method{icc}{stanreg}(x, re.form = NULL, typical = "mean", prob = 0.89,
  ppd = FALSE, ...)

\method{icc}{brmsfit}(x, re.form = NULL, typical = "mean", prob = 0.89,
  ppd = FALSE, ...)
}
\arguments{
\item{x}{Fitted mixed effects model (of class \code{merMod}, \code{glmmTMB},
\code{stanreg} or \code{brmsfit}).}

\item{...}{Currently not used.}

\item{re.form}{Formula containing group-level effects to be considered in
the prediction. If \code{NULL} (default), include all group-level effects.
Else, for instance for nested models, name a specific group-level effect
to calculate the ICC for this group-level. Only applies if \code{ppd = TRUE}.}

\item{typical}{Character vector, naming the function that will be used as
measure of central tendency for the ICC. The default is "mean". See
\code{typical_value} for options.}

\item{prob}{Vector of scalars between 0 and 1, indicating the mass within
the credible interval that is to be estimated. See \code{\link{hdi}}.}

\item{ppd}{Logical, if \code{TRUE}, variance decomposition is based on the
posterior predictive distribution, which is the correct way for Bayesian
non-Gaussian models.}
}
\value{
A numeric vector with all random intercept intraclass-correlation-coefficients.
   Furthermore, between- and within-group variances as well as random-slope
   variance are returned as attributes.
   \cr \cr
   For \code{stanreg} or \code{brmsfit} objects, the HDI for each statistic
   is also  included as attribute.
}
\description{
This function calculates the intraclass-correlation
   (icc) - sometimes also called \emph{variance partition coefficient}
   (vpc) - for random intercepts of mixed effects models. Currently,
   \code{\link[lme4]{merMod}}, \code{\link[glmmTMB]{glmmTMB}},
   \code{stanreg} and \code{\link[brms]{brmsfit}} objects are supported.
}
\details{
The ICC is calculated by dividing the between-group-variance (random
  intercept variance) by the total variance (i.e. sum of between-group-variance
  and within-group (residual) variance). \cr \cr
  The calculation of the ICC for generalized linear mixed models with binary outcome is based on
  \cite{Wu et al. (2012)}. For Poisson multilevel models, please refer to \cite{Stryhn et al. (2006)}.
  \cite{Aly et al. (2014)} describe computation of ICC for negative binomial models.
  \cr \cr
  \strong{Caution:} For models with random slopes and random intercepts,
  the ICC would differ at each unit of the predictors. Hence, the ICC for these
  kind of models cannot be understood simply as proportion of variance
  (see \cite{Goldstein et al. 2010}). For convenience reasons, as the
  \code{icc()} function also extracts the different random effects
  variances, the ICC for random-slope-intercept-models is reported
  nonetheless, but it is usually no meaningful summary of the
  proportion of variances.
  \cr \cr
  The random effect variances indicate the between- and within-group
  variances as well as random-slope variance and random-slope-intercept
  correlation. The components are denoted as following:
  \itemize{
    \item Within-group (residual) variance: sigma_2
    \item Between-group-variance: tau.00 (variation between individual intercepts and average intercept)
    \item Random-slope-variance: tau.11 (variation between individual slopes and average slope)
    \item Random-Intercept-Slope-covariance: tau.01
    \item Random-Intercept-Slope-correlation: rho.01
  }
  If \code{ppd = TRUE}, \code{icc()} calculates a variance decomposition based on
  the posterior predictive distribution. In this case, first, the draws from
  the posterior predictive distribution \emph{not conditioned} on group-level
  terms (\code{posterior_predict(..., re.form = NA)}) are calculated as well
  as draws from this distribution \emph{conditioned} on \emph{all random effects}
  (by default, unless specified else in \code{re.form}) are taken. Then, second,
  the variances for each of these draws are calculated. The "ICC" is then the
  ratio between these two variances. This is the recommended way to
  analyse random-effect-variances for non-Gaussian models. It is then possible
  to compare variances accross models, also by specifying different group-level
  terms via the \code{re.form}-argument.
  \cr \cr
  Sometimes, when the variance of the posterior predictive distribution is
  very large, the variance ratio in the output makes no sense, e.g. because
  it is negative. In such cases, it might help to use a more robust measure
  to calculate the central tendency of the variances. For example, use
  \code{typical = "median"}.
}
\note{
Some notes on why the ICC is useful, based on \cite{Grace-Martin}:
      \itemize{
       \item It can help you determine whether or not a linear mixed model is even necessary. If you find that the correlation is zero, that means the observations within clusters are no more similar than observations from different clusters. Go ahead and use a simpler analysis technique.
       \item It can be theoretically meaningful to understand how much of the overall variation in the response is explained simply by clustering. For example, in a repeated measures psychological study you can tell to what extent mood is a trait (varies among people, but not within a person on different occasions) or state (varies little on average among people, but varies a lot across occasions).
       \item It can also be meaningful to see how the ICC (as well as the between and within cluster variances) changes as variable are added to the model.
      }
      In short, the ICC can be interpreted as \dQuote{the proportion of the variance
      explained by the grouping structure in the population} \cite{(Hox 2002: 15)}.
      \cr \cr
      Usually, the ICC is calculated for the null model ("unconditional model").
      However, according to \cite{Raudenbush and Bryk (2002)} or
      \cite{Rabe-Hesketh and Skrondal (2012)} it is also feasible to compute the ICC
      for full models with covariates ("conditional models") and compare how
      much a level-2 variable explains the portion of variation in the grouping
      structure (random intercept).
      \cr \cr
      \strong{Caution:} For three-level-models, depending on the nested structure
      of the model, the ICC only reports the proportion of variance explained
      for each grouping level. However, the proportion of variance for specific
      levels related to each other (e.g., similarity of level-1-units within
      level-2-units or level-2-units within level-3-units) must be computed
      manually. Use \code{\link{get_re_var}} to get the between-group-variances
      and residual variance of the model, and calculate the ICC for the various level
      correlations.
      \cr \cr
      For example, for the ICC between level 1 and 2: \cr
      \code{sum(get_re_var(fit)) / (sum(get_re_var(fit)) + get_re_var(fit, "sigma_2"))}
      \cr \cr
      or for the ICC between level 2 and 3: \cr
      \code{get_re_var(fit)[2] / sum(get_re_var(fit))}
}
\examples{
library(lme4)
fit0 <- lmer(Reaction ~ 1 + (1 | Subject), sleepstudy)
icc(fit0)

# note: ICC for random-slope-intercept model usually not
# meaningful - see 'Note'.
fit1 <- lmer(Reaction ~ Days + (Days | Subject), sleepstudy)
icc(fit1)

sleepstudy$mygrp <- sample(1:45, size = 180, replace = TRUE)
fit2 <- lmer(Reaction ~ Days + (1 | mygrp) + (1 | Subject), sleepstudy)
icc(fit2)

icc1 <- icc(fit1)
icc2 <- icc(fit2)

print(icc1, comp = "var")
print(icc2, comp = "var")

\dontrun{
# compute ICC for Bayesian mixed model, with an ICC for each
# sample of the posterior. The print()-method then shows
# the median ICC as well as 89\% HDI for the ICC.
# Change interval with print-method:
# print(icc(m, posterior = TRUE), prob = .5)

if (requireNamespace("brms", quietly = TRUE)) {
  library(dplyr)
  sleepstudy$mygrp <- sample(1:5, size = 180, replace = TRUE)
  sleepstudy <- sleepstudy \%>\%
    group_by(mygrp) \%>\%
    mutate(mysubgrp = sample(1:30, size = n(), replace = TRUE))
  m <- brms::brm(
    Reaction ~ Days + (1 | mygrp / mysubgrp) + (1 | Subject),
    data = sleepstudy
  )

  # by default, 89\% interval
  icc(m)

  # show 50\% interval
  icc(m, prob = .5)

  # variances based on posterior predictive distribution
  icc(m, ppd = TRUE)
}}

}
\references{
\itemize{
              \item Aguinis H, Gottfredson RK, Culpepper SA. 2013. Best-Practice Recommendations for Estimating Cross-Level Interaction Effects Using Multilevel Modeling. Journal of Management 39(6): 1490–1528 (\doi{10.1177/0149206313478188})
              \item Aly SS, Zhao J, Li B, Jiang J. 2014. Reliability of environmental sampling culture results using the negative binomial intraclass correlation coefficient. Springerplus 14(3) (\doi{10.1186/2193-1801-3-40})
              \item Goldstein H, Browne W, Rasbash J. 2010. Partitioning Variation in Multilevel Models. Understanding Statistics, 1:4, 223-231 (\doi{10.1207/S15328031US0104_02})
              \item Grace-Martion K. The Intraclass Correlation Coefficient in Mixed Models, \href{http://www.theanalysisfactor.com/the-intraclass-correlation-coefficient-in-mixed-models/}{web}
              \item Hox J. 2002. Multilevel analysis: techniques and applications. Mahwah, NJ: Erlbaum
              \item Rabe-Hesketh S, Skrondal A. 2012. Multilevel and longitudinal modeling using Stata. 3rd ed. College Station, Tex: Stata Press Publication
              \item Raudenbush SW, Bryk AS. 2002. Hierarchical linear models: applications and data analysis methods. 2nd ed. Thousand Oaks: Sage Publications
              \item Stryhn H, Sanchez J, Morley P, Booker C, Dohoo IR. 2006. Interpretation of variance parameters in multilevel Poisson regression models. Proceedings of the 11th International Symposium on Veterinary Epidemiology and Economics, 2006 Available at \url{http://www.sciquest.org.nz/node/64294}
              \item Wu S, Crespi CM, Wong WK. 2012. Comparison of methods for estimating the intraclass correlation coefficient for binary responses in cancer prevention cluster randomized trials. Contempory Clinical Trials 33: 869-880 (\doi{10.1016/j.cct.2012.05.004})
            }
            Further helpful online-ressources:
            \itemize{
              \item \href{http://stats.stackexchange.com/questions/18088/intraclass-correlation-icc-for-an-interaction/28100#28100}{CrossValidated (2012) \emph{Intraclass correlation (ICC) for an interaction?}}
              \item \href{http://stats.stackexchange.com/questions/113577/interpreting-the-random-effect-in-a-mixed-effect-model/113825#113825}{CrossValidated (2014) \emph{Interpreting the random effect in a mixed-effect model}}
              \item \href{http://stats.stackexchange.com/questions/67247/how-to-partition-the-variance-explained-at-group-level-and-individual-level/67356#67356}{CrossValidated (2014) \emph{how to partition the variance explained at group level and individual level}}
            }
}
\seealso{
\code{\link{re_var}}
}
