% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sits_get_data.R
\name{sits_get_data}
\alias{sits_get_data}
\alias{sits_get_data.default}
\alias{sits_get_data.csv}
\alias{sits_get_data.shp}
\alias{sits_get_data.sf}
\alias{sits_get_data.sits}
\alias{sits_get_data.data.frame}
\title{Get time series from data cubes and cloud services}
\usage{
sits_get_data(
  cube,
  samples,
  ...,
  start_date = as.Date(sits_timeline(cube)[1]),
  end_date = as.Date(sits_timeline(cube)[length(sits_timeline(cube))]),
  label = "NoClass",
  bands = sits_bands(cube),
  crs = 4326,
  impute_fn = sits_impute_linear(),
  label_attr = NULL,
  n_sam_pol = 30,
  pol_avg = FALSE,
  pol_id = NULL,
  multicores = 2,
  progress = FALSE
)

\method{sits_get_data}{default}(cube, samples, ...)

\method{sits_get_data}{csv}(
  cube,
  samples,
  ...,
  bands = sits_bands(cube),
  crs = 4326,
  impute_fn = sits_impute_linear(),
  multicores = 2,
  progress = FALSE
)

\method{sits_get_data}{shp}(
  cube,
  samples,
  ...,
  label = "NoClass",
  start_date = as.Date(sits_timeline(cube)[1]),
  end_date = as.Date(sits_timeline(cube)[length(sits_timeline(cube))]),
  bands = sits_bands(cube),
  impute_fn = sits_impute_linear(),
  label_attr = NULL,
  n_sam_pol = 30,
  pol_avg = FALSE,
  pol_id = NULL,
  multicores = 2,
  progress = FALSE
)

\method{sits_get_data}{sf}(
  cube,
  samples,
  ...,
  bands = sits_bands(cube),
  start_date = as.Date(sits_timeline(cube)[1]),
  end_date = as.Date(sits_timeline(cube)[length(sits_timeline(cube))]),
  impute_fn = sits_impute_linear(),
  label = "NoClass",
  label_attr = NULL,
  n_sam_pol = 30,
  pol_avg = FALSE,
  pol_id = NULL,
  multicores = 2,
  progress = FALSE
)

\method{sits_get_data}{sits}(
  cube,
  samples,
  ...,
  bands = sits_bands(cube),
  impute_fn = sits_impute_linear(),
  multicores = 2,
  progress = FALSE
)

\method{sits_get_data}{data.frame}(
  cube,
  samples,
  ...,
  start_date = as.Date(sits_timeline(cube)[1]),
  end_date = as.Date(sits_timeline(cube)[length(sits_timeline(cube))]),
  label = "NoClass",
  bands = sits_bands(cube),
  crs = 4326,
  impute_fn = sits_impute_linear(),
  multicores = 2,
  progress = FALSE
)
}
\arguments{
\item{cube}{Data cube from where data is to be retrieved.}

\item{samples}{Samples location (sits, sf, or data.frame).}

\item{...}{Specific parameters for specific cases.}

\item{start_date}{Start of the interval for the time series
in "YYYY-MM-DD" format (optional).}

\item{end_date}{End of the interval for the time series in
"YYYY-MM-DD" format (optional).}

\item{label}{Label to be assigned to the time series (optional).}

\item{bands}{Bands to be retrieved (optional).}

\item{crs}{A coordinate reference system of samples.
The provided crs could be a character
(e.g, "EPSG:4326" or "WGS84" or a proj4string), or a
a numeric with the EPSG code (e.g. 4326).
This parameter only works for 'csv' or data.frame'
samples. Default is 4326.}

\item{impute_fn}{Imputation function for NA values.}

\item{label_attr}{Attribute in the shapefile or sf object to be used
as a polygon label.}

\item{n_sam_pol}{Number of samples per polygon to be read
(for POLYGON or MULTIPOLYGON shapefile).}

\item{pol_avg}{Summarize samples for each polygon?}

\item{pol_id}{ID attribute for polygons.}

\item{multicores}{Number of threads to process the time series.}

\item{progress}{A logical value indicating if a progress bar
should be shown. Default is \code{FALSE}.}
}
\value{
A tibble with the metadata and data for each time series
<longitude, latitude, start_date, end_date, label, cube, time_series>.
}
\description{
Retrieve a set of time series from a data cube or from
a time series service. Data cubes and puts it in a "sits tibble".
Sits tibbles are the main structures of sits package.
They contain both the satellite image time series and their metadata.
}
\note{
There are four ways of specifying data to be retrieved using the
"samples" parameter:
\itemize{
\item{CSV file: }{Provide a CSV file with columns
"longitude", "latitude", "start_date", "end_date" and "label" for
each sample}
\item{SHP file: }{Provide a shapefile in POINT or POLYGON geometry
containing the location of the samples and an attribute to be
used as label. Also, provide start and end date for the time series.}
\item{sits object: }{A sits tibble.}
\item{sf object: }{An "sf" object with POINT or POLYGON geometry.}
\item{data.frame: }{A data.frame with with mandatory columns
"longitude", "latitude".}
}

Please refer to the sits documentation available in
<https://e-sensing.github.io/sitsbook/> for detailed examples.
}
\examples{
if (sits_run_examples()) {
    # reading a lat/long from a local cube
    # create a cube from local files
    data_dir <- system.file("extdata/raster/mod13q1", package = "sits")
    raster_cube <- sits_cube(
        source = "BDC",
        collection = "MOD13Q1-6",
        data_dir = data_dir,
        delim = "_",
        parse_info = c("X1", "tile", "band", "date")
    )
    samples <- tibble::tibble(longitude = -55.66738, latitude = -11.76990)
    point_ndvi <- sits_get_data(raster_cube, samples)
    #
    # reading samples from a cube based on a  CSV file
    csv_file <- system.file("extdata/samples/samples_sinop_crop.csv",
        package = "sits"
    )
    points <- sits_get_data(cube = raster_cube, samples = csv_file)

    # reading a shapefile from BDC (Brazil Data Cube)
    # needs a BDC access key that can be obtained
    # for free by registering in the BDC website
    if (nchar(Sys.getenv("BDC_ACCESS_KEY")) > 0) {
        # create a data cube from the BDC
        bdc_cube <- sits_cube(
            source = "BDC",
            collection = "CB4_64_16D_STK-1",
            bands = c("NDVI", "EVI"),
            tiles = c("022024", "022025"),
            start_date = "2018-09-01",
            end_date = "2018-10-28"
        )
        # define a shapefile to be read from the cube
        shp_file <- system.file("extdata/shapefiles/bdc-test/samples.shp",
            package = "sits"
        )
        # get samples from the BDC based on the shapefile
        time_series_bdc <- sits_get_data(
            cube = bdc_cube,
            samples = shp_file)
    }
}

}
\author{
Gilberto Camara
}
