#:#
#:#  *Parameter set classes*
#:# 
#:#  This file is part of the R package rpact: 
#:#  Confirmatory Adaptive Clinical Trial Design and Analysis
#:# 
#:#  Author: Gernot Wassmer, PhD, and Friedrich Pahlke, PhD
#:#  Licensed under "GNU Lesser General Public License" version 3
#:#  License text can be found here: https://www.r-project.org/Licenses/LGPL-3
#:# 
#:#  RPACT company website: https://www.rpact.com
#:#  rpact package website: https://www.rpact.org
#:# 
#:#  Contact us for information about our services: info@rpact.com
#:# 
#:#  File version: $Revision: 4311 $
#:#  Last changed: $Date: 2021-02-03 11:38:47 +0100 (Mi, 03 Feb 2021) $
#:#  Last changed by: $Author: pahlke $
#:# 

#' @include f_core_constants.R
NULL

PlotSubTitleItem <- setRefClass("PlotSubTitleItem",
	fields = list(
		title = "character",
		subscript = "character",
		value = "numeric"
	),
	methods = list(
		toQuote = function() {
			if (!is.null(subscript) && length(subscript) == 1 && !is.na(subscript)) {
				return(bquote(' '*.(title)[.(subscript)] == .(value)))
			}
			
			return(bquote(' '*.(title) == .(value)))
		}
	)
)

PlotSubTitleItems <- setRefClass("PlotSubTitleItems",
	fields = list(
		title = "character",
		subTitle = "character",
		items = "list"
	),
	methods = list(
		initialize = function(...) {
			callSuper(...)
			items <<- list()
		},
		
		addItem = function(item) {
			items <<- c(items, item)
		},
		
		add = function(title, value, subscript = NA_character_) {
			titleTemp <- title
			if (length(items) == 0) {
				titleTemp <- .firstCharacterToUpperCase(titleTemp)
			}
			
			titleTemp <- paste0(' ', titleTemp)
			if (length(subscript) > 0 && !is.na(subscript)) {
				subscript <- paste0(as.character(subscript), ' ')
			} else {
				titleTemp <- paste0(titleTemp, ' ')
			}
			addItem(PlotSubTitleItem(title = titleTemp, subscript = subscript, value = value))
		},
		
		toQuote = function() {
			quotedItems <- .getQuotedItems()
			if (is.null(quotedItems)) {
				if (length(subTitle) > 0) {
					return(bquote(atop(bold(.(title)), 
						atop(.(subTitle)))))
				}
				
				return(title)
			}
			
			if (length(subTitle) > 0) {
				return(bquote(atop(bold(.(title)), 
							atop(.(subTitle)*','~.(quotedItems)))))
			}
			
			return(bquote(atop(bold(.(title)), 
				atop(.(quotedItems)))))
		},
		
		.getQuotedItems = function() {
			item1 <- NULL
			item2 <- NULL
			item3 <- NULL
			item4 <- NULL
			if (length(items) > 0) {
				item1 <- items[[1]]
			}
			if (length(items) > 1) {
				item2 <- items[[2]]
			}
			if (length(items) > 2) {
				item3 <- items[[3]]
			}
			if (length(items) > 3) {
				item4 <- items[[4]]
			}
			
			if (!is.null(item1) && !is.null(item2) && !is.null(item3) && !is.null(item4)) {
				if (length(item1$subscript) == 1 && !is.na(item1$subscript) &&
					length(item2$subscript) == 1 && !is.na(item2$subscript)) {
					return(bquote(' '*.(item1$title)[.(item1$subscript)] == .(item1$value)*','~.(item2$title)[.(item2$subscript)] == .(item2$value)*','~.(item3$title) == .(item3$value)*','~.(item4$title) == .(item4$value)*''))
				}
				
				if (length(item1$subscript) == 1 && !is.na(item1$subscript)) {
					return(bquote(' '*.(item1$title)[.(item1$subscript)] == .(item1$value)*','~.(item2$title) == .(item2$value)*','~.(item3$title) == .(item3$value)*','~.(item4$title) == .(item4$value)*''))
				}
				
				if (length(item2$subscript) == 1 && !is.na(item2$subscript)) {
					return(bquote(' '*.(item1$title) == .(item1$value)*','~.(item2$title)[.(item2$subscript)] == .(item2$value)*','~.(item3$title) == .(item3$value)*','~.(item4$title) == .(item4$value)*''))
				}
				
				return(bquote(' '*.(item1$title) == .(item1$value)*','~.(item2$title) == .(item2$value)*','~.(item3$title) == .(item3$value)*','~.(item4$title) == .(item4$value)*''))
			}
			
			if (!is.null(item1) && !is.null(item2) && !is.null(item3)) {
				if (length(item1$subscript) == 1 && !is.na(item1$subscript) &&
						length(item2$subscript) == 1 && !is.na(item2$subscript)) {
					return(bquote(' '*.(item1$title)[.(item1$subscript)] == .(item1$value)*','~.(item2$title)[.(item2$subscript)] == .(item2$value)*','~.(item3$title) == .(item3$value)*''))
				}
				
				if (length(item1$subscript) == 1 && !is.na(item1$subscript)) {
					return(bquote(' '*.(item1$title)[.(item1$subscript)] == .(item1$value)*','~.(item2$title) == .(item2$value)*','~.(item3$title) == .(item3$value)*''))
				}
				
				if (length(item2$subscript) == 1 && !is.na(item2$subscript)) {
					return(bquote(' '*.(item1$title) == .(item1$value)*','~.(item2$title)[.(item2$subscript)] == .(item2$value)*','~.(item3$title) == .(item3$value)*''))
				}
				
				return(bquote(' '*.(item1$title) == .(item1$value)*','~.(item2$title) == .(item2$value)*','~.(item3$title) == .(item3$value)*''))
			}
			
			if (!is.null(item1) && !is.null(item2)) {
				if (length(item1$subscript) == 1 && !is.na(item1$subscript) &&
						length(item2$subscript) == 1 && !is.na(item2$subscript)) {
					return(bquote(' '*.(item1$title)[.(item1$subscript)] == .(item1$value)*','~.(item2$title)[.(item2$subscript)] == .(item2$value)*''))
				}

				if (length(item1$subscript) == 1 && !is.na(item1$subscript)) {
					return(bquote(' '*.(item1$title)[.(item1$subscript)] == .(item1$value)*','~.(item2$title) == .(item2$value)*''))
				}
				
				if (length(item2$subscript) == 1 && !is.na(item2$subscript)) {
					return(bquote(' '*.(item1$title) == .(item1$value)*','~.(item2$title)[.(item2$subscript)] == .(item2$value)*''))
				}
				
				return(bquote(' '*.(item1$title) == .(item1$value)*','~.(item2$title) == .(item2$value)*''))
			}
			
			if (!is.null(item1)) {
				if (length(item1$subscript) == 1 && !is.na(item1$subscript)) {
					return(bquote(' '*.(item1$title)[.(item1$subscript)] == .(item1$value)*''))
				}
				
				return(bquote(' '*.(item1$title) == .(item1$value)*''))
			}
			
			return(NULL)
		}
	)
)

#' 
#' @name FieldSet
#' 
#' @title
#' Field Set
#' 
#' @description 
#' Basic class for field sets.
#' 
#' @details
#' The field set implements basic functions for a set of fields.
#' 
#' @include class_core_plot_settings.R
#' 
#' @keywords internal
#' 
#' @importFrom methods new
#' 
FieldSet <- setRefClass("FieldSet",
	fields = list(
		.parameterTypes = "list",
		.parameterNames = "list",
		.parameterFormatFunctions = "list",
		.showParameterTypeEnabled = "logical",
		.catLines = "character"
	),
	methods = list(
		.getFieldNames = function() {
			return(names(.self$getRefClass()$fields()))
		},
		
		.getVisibleFieldNames = function() {
			fieldNames <- names(.self$getRefClass()$fields())
			fieldNames <- fieldNames[!startsWith(fieldNames, ".")]
			return(fieldNames)
		},
		
		.resetCat = function() {
			.catLines <<- character(0)
		},
		
		.cat = function(..., file = "", sep = "", fill = FALSE, labels = NULL, 
				append = FALSE, heading = 0, tableColumns = 0, consoleOutputEnabled = TRUE) {
			
			if (consoleOutputEnabled) {
				cat(..., file = file, sep = sep, fill = fill, labels = labels, append = append)
				return(invisible())
			}
			
			args <- list(...)
			line <- ""
			if (length(args) > 0) {
				if (tableColumns > 0) {
					values <- unlist(args, use.names = FALSE)
					values <- values[values != "\n"]
					for (i in 1:length(values)) {
						values[i] <- gsub("\n", "", values[i])
					}
					line <- paste0(values, collapse = "| ")
					if (trimws(line) != "" && !grepl("\\| *$", line)) {
						line <- paste0(line, "|")
					}
					line <- paste0("| ", line)
					extraCells <- tableColumns - length(values)
					if (extraCells > 0 && trimws(line) != "") {
						line <- paste0(line, paste0(rep(" |", extraCells), collapse = ""))
					}
					line <- paste0(line, "\n")
				} else {
					line <- paste0(args, collapse = sep)
					listItemEnabled <- grepl("^  ", line)
					if (heading > 0) {
						headingCmd <- paste0(rep("#", heading + 1), collapse = "")
						line <- paste0(headingCmd, " ", sub(": *", "", line))
					} else {
						parts <- strsplit(line, " *: ")[[1]]
						if (length(parts) == 2) {
							line <- paste0("*", trimws(parts[1]), "*: ", parts[2])
						}
					}
					if (listItemEnabled) {
						if (grepl("^  ", line)) {
							line <- sub("^  ", "* ", line)
						} else {
							line <- paste0("* ", line)
						}
					}
				}
			}
			if (length(.catLines) == 0) {
				.catLines <<- line
			} else {
				.catLines <<- c(.catLines, line)
			}
			return(invisible())
		},
		
		.getFields = function(values) {
			flds = names(.self$getRefClass()$fields())
			if (!missing(values)) {
				flds = flds[flds %in% values]
			}
			result = setNames(vector("list", length(flds)), flds)
			for (fld in flds) {
				result[[fld]] = .self[[fld]]
			}
			return(result)
		}
	)
)

#' 
#' @name ParameterSet
#' 
#' @title
#' Parameter Set
#' 
#' @description 
#' Basic class for parameter sets.
#' 
#' @details
#' The parameter set implements basic functions for a set of parameters.
#' 
#' @include f_core_constants.R
#' 
#' @keywords internal
#' 
#' @importFrom methods new
#' 
ParameterSet <- setRefClass("ParameterSet",
	contains = "FieldSet",
	fields = list(
		.parameterTypes = "list",
		.parameterNames = "list",
		.parameterFormatFunctions = "list",
		.showParameterTypeEnabled = "logical",
		.catLines = "character"
	),
	methods = list(
		initialize = function(...,
			.showParameterTypeEnabled = TRUE) {
			callSuper(..., 
				.showParameterTypeEnabled = .showParameterTypeEnabled)
			.parameterTypes <<- list()
			.parameterNames <<- list()
			.parameterFormatFunctions <<- list()
			.catLines <<- character(0)
		},
		
		.toString = function(startWithUpperCase = FALSE) {
			s <- .formatCamelCase(class(.self))
			return(ifelse(startWithUpperCase, .firstCharacterToUpperCase(s), s))
		},
		
		.initParameterTypes = function() {
			for (parameterName in names(.parameterNames)) {
				.parameterTypes[[parameterName]] <<- C_PARAM_TYPE_UNKNOWN
			}
		},
		
		.getParameterType = function(parameterName) {
			if (is.null(parameterName) || length(parameterName) == 0 || is.na(parameterName)) {
				stop(C_EXCEPTION_TYPE_ILLEGAL_ARGUMENT, 
					"'parameterName' must be a valid character with length > 0")
			}
			
			parameterType <- .parameterTypes[[parameterName]]
			if (is.null(parameterType)) {
				return(C_PARAM_TYPE_UNKNOWN)
			}
			
			return(parameterType[1])
		},
		
		.getParametersToShow = function() {
			return(.getVisibleFieldNames())
		},
		
		.setParameterType = function(parameterName, parameterType) {
			if (is.null(parameterName) || length(parameterName) == 0 || is.na(parameterName)) {
				stop(C_EXCEPTION_TYPE_ILLEGAL_ARGUMENT, 
					"'parameterName' must be a valid character with length > 0")
			}
			
			parameterType <- parameterType[1]
			
			if (!all(parameterType %in% c(C_PARAM_USER_DEFINED, C_PARAM_DEFAULT_VALUE, 
						C_PARAM_GENERATED, C_PARAM_DERIVED, C_PARAM_NOT_APPLICABLE))) {
				stop(C_EXCEPTION_TYPE_ILLEGAL_ARGUMENT, 
					"'parameterType' ('", parameterType, "') is invalid")
			}
			
			.parameterTypes[[parameterName]] <<- parameterType
			
			invisible(parameterType)
		},
		
		isUserDefinedParameter = function(parameterName) {
			return(.getParameterType(parameterName) == C_PARAM_USER_DEFINED)
		},
		
		isDefaultParameter = function(parameterName) {
			return(.getParameterType(parameterName) == C_PARAM_DEFAULT_VALUE)
		},
		
		isGeneratedParameter = function(parameterName) {
			return(.getParameterType(parameterName) == C_PARAM_GENERATED)
		},
		
		isDerivedParameter = function(parameterName) {
			return(.getParameterType(parameterName) == C_PARAM_DERIVED)
		},
		
		isUndefinedParameter = function(parameterName) {
			return(.getParameterType(parameterName) == C_PARAM_TYPE_UNKNOWN)
		},
		
		.getInputParameters = function() {
			params <- .getParametersOfOneGroup(c(C_PARAM_USER_DEFINED, C_PARAM_DEFAULT_VALUE))
			return(params)
		},
		
		.getUserDefinedParameters = function() {
			return(.getParametersOfOneGroup(C_PARAM_USER_DEFINED))
		},
		
		.getDefaultParameters = function() {
			return(.getParametersOfOneGroup(C_PARAM_DEFAULT_VALUE))
		},
		
		.getGeneratedParameters = function() {
			return(.getParametersOfOneGroup(C_PARAM_GENERATED))
		},
		
		.getDerivedParameters = function() {
			return(.getParametersOfOneGroup(C_PARAM_DERIVED))
		},
		
		.getUndefinedParameters = function() {
			return(.getParametersOfOneGroup(C_PARAM_TYPE_UNKNOWN))
		},
		
		.getParameterValueIfUserDefinedOrDefault = function(parameterName) {
			if (isUserDefinedParameter(parameterName) || isDefaultParameter(parameterName)) {
				return(.self[[parameterName]])
			}
			
			parameterType <- .self$getRefClass()$fields()[[parameterName]]
			if (parameterType == "numeric") {
				return(NA_real_)
			}
			
			if (parameterType == "integer") {
				return(NA_integer_)
			}
			
			if (parameterType == "character") {
				return(NA_character_)
			}
			
			return(NA)
		},
		
		.getParametersOfOneGroup = function(parameterType) {
			if (length(parameterType) == 1) {
				parameterNames <- names(.parameterTypes[.parameterTypes == parameterType])
			} else {
				parameterNames <- names(.parameterTypes[which(.parameterTypes %in% parameterType)])
			}
			parametersToShow <- .getParametersToShow()
			if (is.null(parametersToShow) || length(parametersToShow) == 0) {
				return(parameterNames)
			}
			
			return(parametersToShow[parametersToShow %in% parameterNames])
		},
		
		.showParameterType = function(parameterName) {
			if (!.showParameterTypeEnabled) {
				return("  ")
			}
			
			return(paste0("[", .getParameterType(parameterName), "]"))
		},
		
		.isMatrix = function(param) {
			if (missing(param) || is.null(param) || is.list(param)) {
				return(FALSE)
			}
			
			return(is.matrix(param))
		},
		
		.isArray = function(param) {
			if (missing(param) || is.null(param) || is.list(param)) {
				return(FALSE)
			}
			
			return(is.array(param))
		},
		
		.isVector = function(param) {
			if (missing(param) || is.null(param) || is.list(param)) {
				return(FALSE)
			}
			
			return(length(param) > 1)
		},
		
		.showAllParameters = function(consoleOutputEnabled = TRUE) {
			parametersToShow <- .getVisibleFieldNamesOrdered()
			for (parameter in parametersToShow) {
				.showParameter(parameter, showParameterType = TRUE, 
					consoleOutputEnabled = consoleOutputEnabled)
			}
		},
		
		.getVisibleFieldNamesOrdered = function() {
			visibleFieldNames <- .getVisibleFieldNames()
			
			parametersToShowSorted <- .getParametersToShow()
			if (is.null(parametersToShowSorted) || length(parametersToShowSorted) == 0) {
				return(visibleFieldNames)
			}
			
			visibleFieldNames <- visibleFieldNames[!(visibleFieldNames %in% parametersToShowSorted)]
			visibleFieldNames <- c(parametersToShowSorted, visibleFieldNames)
			return(visibleFieldNames)
		},
		
		.show = function(..., consoleOutputEnabled = FALSE) {
			showType <- .getOptionalArgument("showType", ...)
			if (!is.null(showType) && showType == 2) {
				.cat("Technical developer summary of the ", .self$.toString(), " object (",
					methods::classLabel(class(.self)), "):\n\n", sep = "", heading = 1,
					consoleOutputEnabled = consoleOutputEnabled)
				.showAllParameters(consoleOutputEnabled = consoleOutputEnabled)
				.showParameterTypeDescription(consoleOutputEnabled = consoleOutputEnabled)
			} else {
				stop(C_EXCEPTION_TYPE_RUNTIME_ISSUE, 
					"method '.show()' is not implemented in class '", class(.self), "'")
			}
		},
		
		.catMarkdownText = function() {
			.show(consoleOutputEnabled = FALSE)
			if (length(.catLines) == 0) {
				return(invisible())
			}
			
			for (line in .catLines) {
				cat(line)
			}
		},
		
		.showParametersOfOneGroup = function(parameters, title, 
				orderByParameterName = TRUE, consoleOutputEnabled = TRUE) {
			output <- ""
			if (is.null(parameters) || length(parameters) == 0 || all(is.na(parameters))) {
				if (!missing(title) && !is.null(title) && !is.na(title) && consoleOutputEnabled) {
					output <- paste0(title, ": not available\n\n")
					.cat(output, heading = 2, consoleOutputEnabled = consoleOutputEnabled)
				}
				invisible(output)
			} else {
				if (orderByParameterName) {
					parameters <- sort(parameters)
				}
				
				if (!missing(title) && !is.null(title) && !is.na(title)) {
					output <- paste0(title, ":\n")
					.cat(output, heading = 2, consoleOutputEnabled = consoleOutputEnabled)
				}
				for (parameterName in parameters) {
					output <- paste0(output, .showParameter(parameterName, 
						consoleOutputEnabled = consoleOutputEnabled))
				}			
				.cat("\n", consoleOutputEnabled = consoleOutputEnabled)
				output <- paste0(output, "\n")
				invisible(output)				
			}
		},
		
		.showParameter = function(parameterName, showParameterType = FALSE, consoleOutputEnabled = TRUE) {
			tryCatch({
				param <- .getParameterValueFormatted(parameterName = parameterName)
				output <- ""
				if (!is.null(param)) {
					if (param$type == "array" && length(dim(param$paramValue)) == 3) {
						numberOfEntries <- dim(param$paramValue)[3]
						numberOfRows <- dim(param$paramValue)[1]
						index <- 1
						for (i in 1:numberOfEntries) {
							for (j in 1:numberOfRows) { 
								output <- paste0(output, .showParameterFormatted(paramName = param$paramName, 
										paramValue = param$paramValue[j, , i], 
										paramValueFormatted = param$paramValueFormatted[[index]], 
										showParameterType = showParameterType,
										category = i,
										matrixRow = ifelse(numberOfRows == 1, NA_integer_, j), 
										consoleOutputEnabled = consoleOutputEnabled,
										paramNameRaw = parameterName,
										numberOfCategories = numberOfEntries))
								index <- index + 1
							}
						}
					} else if (param$type %in% c("matrix", "array")) {
						n <- length(param$paramValueFormatted)
						for (i in 1:n) {
							paramValue <- param$paramValue 
							if (is.array(paramValue) && 
								length(dim(paramValue)) == 3 && 
								dim(paramValue)[3] == 1) {
								paramValue <- paramValue[i, , 1]
							}
							else if (dim(paramValue)[1] > 1 || dim(paramValue)[2] > 1) {
								paramValue <- paramValue[i, ]
							}
							
							output <- paste0(output, .showParameterFormatted(paramName = param$paramName, 
									paramValue = paramValue, 
									paramValueFormatted = param$paramValueFormatted[[i]], 
									showParameterType = showParameterType,
									matrixRow = ifelse(n == 1, NA_integer_, i), consoleOutputEnabled = consoleOutputEnabled,
									paramNameRaw = parameterName,
									numberOfCategories = n))
						}
					} else {
						output <- .showParameterFormatted(paramName = param$paramName, paramValue = param$paramValue, 
							paramValueFormatted = param$paramValueFormatted, showParameterType = showParameterType, 
							consoleOutputEnabled = consoleOutputEnabled, paramNameRaw = parameterName)
					}
				}
				return(invisible(output))
			}, error = function(e) {
				if (consoleOutputEnabled) {
					warning("Failed to show parameter '", parameterName, "': ", e$message)
				}
			})
		},
		
		.extractParameterNameAndValue = function(parameterName) {
			d <- regexpr(paste0("\\..+\\$"), parameterName)
			if (d[1] != 1) {
				return(list(parameterName = parameterName, paramValue = get(parameterName)))
			}
			
			index <- attr(d, "match.length")
			objectName <- substr(parameterName, 1, index - 1)
			parameterName <- substr(parameterName, index + 1, nchar(parameterName))	
			paramValue <- get(objectName)[[parameterName]]
			return(list(parameterName = parameterName, paramValue = paramValue))
		},
		
		.getMatrixFormatted = function(paramValueFormatted) {
			if (!is.matrix(paramValueFormatted)) {
				return(list(
					paramValueFormatted = matrix(as.character(paramValueFormatted), ncol = 1),
					type = "matrix"
				))
			} 
			
			matrixFormatted <- paramValueFormatted
			paramValueFormatted <- .arrayToString(matrixFormatted[1, ])
			type <- "vector"
			if (nrow(matrixFormatted) > 1 && ncol(matrixFormatted) > 0) {
				type <- "matrix"
				paramValueFormatted <- list(paramValueFormatted)
				for (i in 2:nrow(matrixFormatted)) {
					paramValueFormatted <- c(paramValueFormatted, 
						.arrayToString(matrixFormatted[i, ]))
				}
			}
			return(list(
				paramValueFormatted = paramValueFormatted,
				type = type
			))
		},
		
		.getParameterValueFormatted = function(parameterName) {
			tryCatch({
				result <- .extractParameterNameAndValue(parameterName)
				parameterName <- result$parameterName
				paramValue <- result$paramValue

				if (isS4(paramValue)) {
					return(NULL)
				}

				if (is.function(paramValue)) {
					valueStr <- ifelse(.getParameterType(parameterName) == C_PARAM_USER_DEFINED, "user defined", "default")
					return(list(
						paramName = parameterName,
						paramValue = valueStr,
						paramValueFormatted = valueStr,
						type = "function"
					))
				}
				
				paramValueFormatted <- paramValue
				
				if (.getParameterType(parameterName) == C_PARAM_USER_DEFINED &&
						(!is.numeric(paramValue) || identical(paramValue, round(paramValue)))) {
					if (inherits(.self, C_CLASS_NAME_TRIAL_DESIGN_INVERSE_NORMAL) && parameterName == "typeOfDesign") {
						paramValueFormatted <- C_TYPE_OF_DESIGN_LIST[[paramValue]]
					}
					if (inherits(.self, C_CLASS_NAME_TRIAL_DESIGN_INVERSE_NORMAL) && parameterName == "typeBetaSpending") {
						paramValueFormatted <- C_TYPE_OF_DESIGN_BS_LIST[[paramValue]]
					}
				} else {
					formatFunctionName <- .parameterFormatFunctions[[parameterName]]
					if (!is.null(formatFunctionName)) {
						paramValueFormatted <- eval(call(formatFunctionName, paramValueFormatted))
						if (.isArray(paramValue) && length(dim(paramValue)) == 2) {
							paramValueFormatted <- matrix(paramValueFormatted, ncol = ncol(paramValue))
						}
					}
					else if (inherits(.self, C_CLASS_NAME_TRIAL_DESIGN_INVERSE_NORMAL) && parameterName == "typeOfDesign") {
						paramValueFormatted <- C_TYPE_OF_DESIGN_LIST[[paramValue]] 
					}
					else if (inherits(.self, C_CLASS_NAME_TRIAL_DESIGN_INVERSE_NORMAL) && parameterName == "typeBetaSpending") {
						paramValueFormatted <- C_TYPE_OF_DESIGN_BS_LIST[[paramValue]]
					}
				}
				
				type <- "vector"
				if (.isArray(paramValue) && length(dim(paramValue)) == 3) {
					arrayFormatted <- paramValueFormatted
					numberOfEntries <- dim(arrayFormatted)[3]
					m <- .getMatrixFormatted(arrayFormatted[, , 1])
					paramValueFormatted <- m$paramValueFormatted
					type <- m$type
					if (numberOfEntries > 1) {
						type <- "array"
						for (i in 2:numberOfEntries) {
							m <- .getMatrixFormatted(arrayFormatted[, , i])
							paramValueFormatted <- c(paramValueFormatted, m$paramValueFormatted)
						}
					}
				}
				
				else if (.isMatrix(paramValue) || .isArray(paramValue)) {
					m <- .getMatrixFormatted(paramValueFormatted)
					paramValueFormatted <- m$paramValueFormatted
					type <- m$type
				}
				
				else if (.isVector(paramValue)) {
					paramValueFormatted <- .arrayToString(paramValueFormatted)
				}
				
				else if (parameterName == "sided") {
					paramValueFormatted <- ifelse(paramValue == 1, "one-sided", "two-sided")
				}
				
				return(list(
					paramName = parameterName,
					paramValue = paramValue,
					paramValueFormatted = paramValueFormatted,
					type = type
				))
			}, error = function(e) {
				.logError(paste0("Error in '.getParameterValueFormatted'. ",
					"Failed to show parameter '%s' (class '%s'): %s"), parameterName, class(.self), e)
			})
		
			return(NULL)
		},
		
		.showUnknownParameters = function(consoleOutputEnabled = TRUE) {
			params <- .getUndefinedParameters()
			if (length(params) > 0) {
				.showParametersOfOneGroup(params, "ISSUES (parameters with undefined type)",
					consoleOutputEnabled = consoleOutputEnabled)
			}
		},
		
		.showParameterFormatted = function(paramName, paramValue, ..., paramValueFormatted = NA_character_,
				showParameterType = FALSE, category = NA_integer_, matrixRow = NA_integer_, consoleOutputEnabled = TRUE,
				paramNameRaw = NA_character_, numberOfCategories = NA_integer_) {
			if (!is.na(paramNameRaw)) {
				paramCaption <- .parameterNames[[paramNameRaw]]
			}
			if (is.null(paramCaption)) {
				paramCaption <- .parameterNames[[paramName]]
			}
			if (is.null(paramCaption)) {
				paramCaption <- paste0("%", paramName, "%")
			}
			if (!is.na(category)) {
				if (inherits(.self, "SimulationResultsMultiArmSurvival") && 
						paramName == "singleNumberOfEventsPerStage") {
					if (!is.na(numberOfCategories) && numberOfCategories == category) {
						category <- "control"
					}
					paramCaption <- paste0(paramCaption, " {", category, "}")
				} else {
					paramCaption <- paste0(paramCaption, " (", category, ")")
				}
				if (!is.na(matrixRow)) {
					paramCaption <- paste0(paramCaption, " [", matrixRow, "]")
				}
			}
			else if (!is.na(matrixRow)) {
				if (.isMultiArmAnalysisResults(.self) && paramName %in% 
					c("conditionalErrorRate", "secondStagePValues", 
						"adjustedStageWisePValues", "overallAdjustedTestStatistics")) {
					treatments <- .closedTestResults$.getHypothesisTreatmentArmVariants()[matrixRow]
					paramCaption <- paste0("Treatment", ifelse(grepl(",", treatments), "s", ""), " ", treatments, " vs. control")
				}
    			else if (.isMultiArmAnalysisResults(.self) || grepl("StageResultsMultiArm", class(.self)) || 
						(inherits(.self, "SimulationResults") && paramName == "effectMatrix") ||
						(inherits(.self, "ClosedCombinationTestResults") && paramName %in% c("rejected", "separatePValues"))) {
					paramCaption <- paste0(paramCaption, " (", matrixRow, ")")
				} else {
					paramCaption <- paste0(paramCaption, " [", matrixRow, "]")
				}
			}
			if (is.null(paramValueFormatted) || length(paramValueFormatted) == 0 || 
					is.na(paramValueFormatted)) {
				paramValueFormatted <- paramValue
			}
			if (is.list(paramValueFormatted)) {
				paramValueFormatted <- .listToString(paramValueFormatted)
			}
			if (is.function(paramValue)) {
				paramValueFormatted <- ifelse(.getParameterType(paramName) == C_PARAM_USER_DEFINED, "user defined", "default")
			}
			prefix <- ifelse(showParameterType, .showParameterType(paramName), "")
			variableNameFormatted <- .getFormattedVariableName(name = paramCaption, n = .getNChar(), prefix = prefix)
			output <- paste(variableNameFormatted, paramValueFormatted, "\n")
			.cat(output, consoleOutputEnabled = consoleOutputEnabled)
			invisible(output)
		},
		
		.getNChar = function() {
			if (length(.parameterNames) == 0) {
				return(40)
			}
			
			return(min(40, max(nchar(.parameterNames))) + 4)
		},
		
		.showParameterTypeDescription = function(consoleOutputEnabled = consoleOutputEnabled) {
			.cat("\n", consoleOutputEnabled = consoleOutputEnabled)
			.cat("Legend:\n", heading = 2, consoleOutputEnabled = consoleOutputEnabled)
			.cat("  ", C_PARAM_USER_DEFINED, ": user defined\n", consoleOutputEnabled = consoleOutputEnabled)
			.cat("  ", C_PARAM_DERIVED, ": derived value\n", consoleOutputEnabled = consoleOutputEnabled)
			.cat("  ", C_PARAM_DEFAULT_VALUE, ": default value\n", consoleOutputEnabled = consoleOutputEnabled)
			.cat("  ", C_PARAM_GENERATED, ": generated/calculated value\n", consoleOutputEnabled = consoleOutputEnabled)
			.cat("  ", C_PARAM_NOT_APPLICABLE, ": not applicable or hidden\n", consoleOutputEnabled = consoleOutputEnabled)
		},
		
		.printAsDataFrame = function(parameterNames, niceColumnNamesEnabled = FALSE,
				includeAllParameters = FALSE, handleParameterNamesAsToBeExcluded = FALSE,
				lineBreakEnabled = FALSE) {
			
			if (.isTrialDesign(.self)) {
				tableColumnNames <- .getTableColumnNames(design = .self)	
			} else {
				tableColumnNames <- C_TABLE_COLUMN_NAMES
			}
			
			if (.isTrialDesignPlan(.self)) {
				parameterNames <- NULL
			}
					
			dataFrame <- .getAsDataFrame(parameterNames, niceColumnNamesEnabled = niceColumnNamesEnabled,
				includeAllParameters = includeAllParameters, 
				handleParameterNamesAsToBeExcluded = handleParameterNamesAsToBeExcluded,
				returnParametersAsCharacter = TRUE, tableColumnNames = tableColumnNames)
			result <- as.matrix(dataFrame)
			if (.isTrialDesignPlan(.self)) {
				dimnames(result)[[1]] <- paste("  ", c(1:nrow(dataFrame)))
			} else if (!is.null(dataFrame[["stages"]])) {
				dimnames(result)[[1]] <- paste("  Stage", dataFrame$stages)
			}
			
			print(result, quote = FALSE, right = FALSE)
		},
		
		.getNumberOfRows = function(parameterNames) {
			numberOfRows <- 1
			for (parameterName in parameterNames) {
				parameterValues <- .self[[parameterName]]
				if (is.vector(parameterValues) && length(parameterValues) > numberOfRows) {
					numberOfRows <- length(parameterValues)
				}
				else if (is.matrix(parameterValues) && (nrow(parameterValues) == 1 || ncol(parameterValues) == 1) && 
						length(parameterValues) > numberOfRows) {
					numberOfRows <- length(parameterValues)
				}
			}
			return(numberOfRows)
		},
		
		.containsMultidimensionalParameters = function(parameterNames) {
			for (parameterName in parameterNames) {
				parameterValues <- .self[[parameterName]]
				if (!is.null(parameterValues) && is.matrix(parameterValues) && 
						nrow(parameterValues) > 0 && ncol(parameterValues) > 0) {
					return(TRUE)
				}
			}
			return(FALSE)
		},
		
		.getMultidimensionalNumberOfVariants = function(parameterNames) {
			parameterNames <- parameterNames[!(parameterNames %in% c(
				"accrualTime", "accrualIntensity", 
				"plannedSubjects", "plannedEvents",
				"minNumberOfSubjectsPerStage", "maxNumberOfSubjectsPerStage",
				"minNumberOfEventsPerStage", "maxNumberOfEventsPerStage", 
				"piecewiseSurvivalTime", "lambda2", "adaptations"))]
			if (!is.null(.self[[".piecewiseSurvivalTime"]]) && .self$.piecewiseSurvivalTime$piecewiseSurvivalEnabled) {
				parameterNames <- parameterNames[!(parameterNames %in% c("lambda1"))]
			}
			
			n <- 1
			for (parameterName in parameterNames) {
				parameterValues <- .self[[parameterName]]
				if (!is.null(parameterValues) && !is.array(parameterValues)) {
					if (is.matrix(parameterValues)) {
						if (nrow(parameterValues) > 0 && ncol(parameterValues) > n) {
							n <- ncol(parameterValues)
						}
					}
					else if (length(parameterValues) > n) {
						n <- length(parameterValues)
					}
				}
			}
			return(n)
		},
		
		.getMultidimensionalNumberOfStages = function(parameterNames) {
			n <- 1
			for (parameterName in parameterNames) {
				parameterValues <- .self[[parameterName]]
				if (!is.null(parameterValues) && is.matrix(parameterValues) && 
						ncol(parameterValues) > 0 && nrow(parameterValues) > n) {
					n <- nrow(parameterValues)
				}
			}
			return(n)
		},
		
		.getVariedParameter = function(parameterNames, numberOfVariants) {
			for (parameterName in parameterNames) {
				parameterValues <- .self[[parameterName]]
				if (!is.null(parameterValues) && !is.matrix(parameterValues) &&
						length(parameterValues) == numberOfVariants &&
						parameterName %in% C_VARIABLE_DESIGN_PLAN_PARAMETERS &&
						.getParameterType(parameterName) == C_PARAM_USER_DEFINED) {
					return(parameterName)
				}
			}
			for (parameterName in parameterNames) {
				parameterValues <- .self[[parameterName]]
				if (!is.null(parameterValues) && !is.matrix(parameterValues) &&
						length(parameterValues) == numberOfVariants &&
						parameterName %in% C_VARIABLE_DESIGN_PLAN_PARAMETERS &&
						.getParameterType(parameterName) == C_PARAM_DEFAULT_VALUE) {
					return(parameterName)
				}
			}
			return(NULL)
		},
		
		.getDataFrameColumnValues = function(parameterName, numberOfVariants, 
				numberOfStages, includeAllParameters) {
				
			if (.getParameterType(parameterName) == C_PARAM_TYPE_UNKNOWN) {
				return(NULL)
			}
			
			if (!includeAllParameters && .getParameterType(parameterName) == C_PARAM_NOT_APPLICABLE) {
				return(NULL)
			}
			
			parameterValues <- .self[[parameterName]]
			if (is.null(parameterValues) || length(parameterValues) == 0) {
				return(NULL)
			}
			
			if (is.function(parameterValues) || 
					(is.array(parameterValues) && (parameterName != "effectMatrix" || !is.matrix(parameterValues)))) {
				return(NULL)
			}
			
			if (!is.matrix(parameterValues)) {
				if (length(parameterValues) == 1) {
					return(rep(parameterValues, numberOfVariants * numberOfStages))
				}
				
				if (length(parameterValues) == numberOfVariants) {
					return(rep(parameterValues, numberOfStages))
				}
				
				if (parameterName %in% c("accrualTime", "accrualIntensity", 
						"plannedEvents", "plannedSubjects", 
						"minNumberOfEventsPerStage", "maxNumberOfEventsPerStage", 
						"minNumberOfSubjectsPerStage", "maxNumberOfSubjectsPerStage", 
						"piecewiseSurvivalTime", "lambda2")) {
					return(NULL)
				}
				
				stop(C_EXCEPTION_TYPE_RUNTIME_ISSUE, 
					"parameter '", parameterName, "' has an invalid ", 
					"dimension (length is ", length(parameterValues), ")")
			} 
			else if (parameterName == "effectMatrix") {
				# return effect matrix row if 'effectMatrix' is user defined
				if (.self$.getParameterType("effectMatrix") == C_PARAM_USER_DEFINED) {
					return(1:ncol(parameterValues))
				}
				
				return(parameterValues[nrow(parameterValues), ])
			}
			
			if (grepl("futility|alpha0Vec", parameterName) && 
					nrow(parameterValues) == numberOfStages - 1) {
				parameterValues <- rbind(parameterValues, rep(NA_real_, ncol(parameterValues)))
			}
			
			if (nrow(parameterValues) == numberOfStages && ncol(parameterValues) == 1) {
				columnValues <- c()
				for (parameterValue in parameterValues) {
					columnValues <- c(columnValues, rep(parameterValue, numberOfVariants))
				}
				return(columnValues)
			}
			
			if (nrow(parameterValues) == numberOfStages && ncol(parameterValues) == numberOfVariants) {
				columnValues <- c()
				for (i in 1:nrow(parameterValues)) {
					for (j in 1:ncol(parameterValues)) {
						columnValues <- c(columnValues, parameterValues[i, j])
					}
				}
				return(columnValues)
			}
			
			stop(C_EXCEPTION_TYPE_RUNTIME_ISSUE, 
				"parameter '", parameterName, "' has an invalid ", 
				"dimension (", nrow(parameterValues), " x ", ncol(parameterValues), "); ",
				"expected was (", numberOfStages, " x ", numberOfVariants, ")")
		},
		
		.getAsDataFrameMultidimensional = function(parameterNames, niceColumnNamesEnabled, 
				includeAllParameters, returnParametersAsCharacter, tableColumnNames) {
				
			numberOfVariants <- .getMultidimensionalNumberOfVariants(parameterNames)
			numberOfStages <- .getMultidimensionalNumberOfStages(parameterNames)
			
			stagesCaption <- .getDataFrameColumnCaption("stages", 
				tableColumnNames, niceColumnNamesEnabled)
			
			dataFrame <- data.frame(
				stages = sort(rep(1:numberOfStages, numberOfVariants))
			)
			names(dataFrame) <- stagesCaption
			
			variedParameter <- .getVariedParameter(parameterNames, numberOfVariants)
			if (!is.null(variedParameter) && variedParameter != "stages") {
				variedParameterCaption <- .getDataFrameColumnCaption(variedParameter, 
					tableColumnNames, niceColumnNamesEnabled)
				dataFrame[[variedParameterCaption]] <- rep(.self[[variedParameter]], numberOfStages)
			}
			
			for (parameterName in parameterNames) {
				if (!(parameterName %in% c("stages", "adaptations")) && 
						(is.null(variedParameter) || parameterName != variedParameter)) {
					columnValues <- .getDataFrameColumnValues(parameterName, 
						numberOfVariants, numberOfStages, includeAllParameters) 
					if (!is.null(columnValues)) {
						columnCaption <- .getDataFrameColumnCaption(parameterName, 
							tableColumnNames, niceColumnNamesEnabled)
						dataFrame[[columnCaption]] <- columnValues
						if (returnParametersAsCharacter) {
							.formatDataFrameParametersAsCharacter(dataFrame, 
								parameterName, columnValues, columnCaption)
						}
					}
				}
			}
			
			return(dataFrame)
		},
		
		.getDataFrameColumnCaption = function(parameterName, tableColumnNames, niceColumnNamesEnabled) {
			if (length(parameterName) == 0 || parameterName == "") {
				stop(C_EXCEPTION_TYPE_ILLEGAL_ARGUMENT, "'parameterName' must be a valid parameter name")
			}
			
			tableColumnName <- tableColumnNames[[parameterName]]
			return(ifelse(niceColumnNamesEnabled && !is.null(tableColumnName), 
				tableColumnName, parameterName))
		},
		
		.getUnidimensionalNumberOfStages = function(parameterNames) {
			kMax <- .self[["kMax"]]
			if (is.null(kMax) && !is.null(.self[[".design"]])) {
				kMax <- .self[[".design"]][["kMax"]]
			}
			if (!is.null(kMax) && length(kMax) == 1 && is.integer(kMax)) {
				return(kMax)
			}
			
			n <- 1
			for (parameterName in parameterNames) {
				parameterValues <- .self[[parameterName]]
				if (!is.null(parameterValues) && !is.matrix(parameterValues) && 
						length(parameterValues) > n) {
					n <- length(parameterValues)
				}
			}
			return(n)
		},
		
		.formatDataFrameParametersAsCharacter = function(dataFrame, 
				parameterName, parameterValues, parameterCaption) {
			tryCatch({	
				formatFunctionName <- .parameterFormatFunctions[[parameterName]]
				if (!is.null(formatFunctionName)) {
					parameterValuesFormatted <- eval(call(formatFunctionName, parameterValues))
				} else {
					parameterValuesFormatted <- as.character(parameterValues)
				}
				
				if (parameterName == "sided") {
					parameterValuesFormatted <- ifelse(parameterValues == 1, 
						"one-sided", "two-sided")
				}
				
				if (!is.null(dataFrame[[parameterCaption]])) {
					parameterValuesFormatted[is.na(dataFrame[[parameterCaption]])] <- ""
				}
				parameterValuesFormatted[is.na(parameterValuesFormatted)] <- ""
				parameterValuesFormatted[parameterValuesFormatted == "NA"] <- ""
				if (is.null(dataFrame)) {
					dataFrame <- data.frame(x = parameterValuesFormatted)
					names(dataFrame) <- parameterCaption
				} else {
					dataFrame[[parameterCaption]] <- parameterValuesFormatted
				}
				
			}, error = function(e) {
				.logError(paste0("Error in '.getAsDataFrame'. Failed to show parameter '%s' ", 
						"(class '%s'): %s"), parameterName, class(.self), e)
			})
		},
		
		.getAsDataFrameUnidimensional = function(parameterNames, niceColumnNamesEnabled, 
				includeAllParameters, returnParametersAsCharacter, tableColumnNames) {
				
			numberOfStages <- .getUnidimensionalNumberOfStages(parameterNames)

			dataFrame <- NULL
			for (parameterName in parameterNames) {			
				tryCatch({	
					parameterCaption <- ifelse(niceColumnNamesEnabled 
							&& !is.null(tableColumnNames[[parameterName]]), 
						tableColumnNames[[parameterName]], parameterName)
					parameterValues <- .self[[parameterName]]
					if (parameterName == "futilityBounds") {
						parameterValues[parameterValues == C_FUTILITY_BOUNDS_DEFAULT] <- -Inf
					}
					if (length(parameterValues) == 1) {
						parameterValues <- rep(parameterValues, numberOfStages)
					} else {
						while (length(parameterValues) < numberOfStages) {
							parameterValues <- c(parameterValues, NA)
						}
					}
					if (includeAllParameters || (
							.getParameterType(parameterName) != C_PARAM_NOT_APPLICABLE && 
							sum(is.na(parameterValues)) < length(parameterValues))) {
						if (is.null(dataFrame)) {
							dataFrame <- data.frame(x = parameterValues)
							names(dataFrame) <- parameterCaption
						} else {
							dataFrame[[parameterCaption]] <- parameterValues
						}
					}
					if (returnParametersAsCharacter) {
						.formatDataFrameParametersAsCharacter(dataFrame, 
							parameterName, parameterValues, parameterCaption)
					}
				}, error = function(e) {
					.logError("Failed to add parameter '%s' to data.frame: %s", parameterName, e)
				})
			}
			
			return(dataFrame)
		},
		
		.getAsDataFrame = function(parameterNames, niceColumnNamesEnabled = TRUE, 
				includeAllParameters = FALSE, handleParameterNamesAsToBeExcluded = FALSE,
				returnParametersAsCharacter = FALSE, tableColumnNames = C_TABLE_COLUMN_NAMES) {
			
			parameterNamesToBeExcluded <- c()
			if (handleParameterNamesAsToBeExcluded) {
				parameterNamesToBeExcluded <- parameterNames 
				parameterNames <- .getVisibleFieldNamesOrdered()
				if (!is.null(parameterNamesToBeExcluded) && length(parameterNamesToBeExcluded) > 0) {
					parameterNames <- parameterNames[!(parameterNames %in% parameterNamesToBeExcluded)]
				}
			}
			else if (is.null(parameterNames)) {
				parameterNames <- .getVisibleFieldNamesOrdered()
			}
			parameterNames <- parameterNames[!grepl("^\\.", parameterNames)]
			
			if (!is.null(.self[[".piecewiseSurvivalTime"]]) && .self$.piecewiseSurvivalTime$piecewiseSurvivalEnabled) {
				parameterNames <- parameterNames[!(parameterNames %in% c("lambda1", "lambda2"))]
			} 
			
			if (.containsMultidimensionalParameters(parameterNames)) {
				return(.getAsDataFrameMultidimensional(parameterNames, niceColumnNamesEnabled, 
						includeAllParameters, returnParametersAsCharacter, tableColumnNames))
			} 
			
			# remove matrices
			for (parameterName in parameterNames) {
				parameterValues <- .self[[parameterName]]
				if (is.matrix(parameterValues) && nrow(parameterValues) != 1 && ncol(parameterValues) != 1) {
					parameterNames <- parameterNames[parameterNames != parameterName]
				}
			}
				
			if (length(parameterNames) == 0) {
				return(data.frame())
			}
			
			return(.getAsDataFrameUnidimensional(parameterNames, niceColumnNamesEnabled, 
				includeAllParameters, returnParametersAsCharacter, tableColumnNames))
		},
		
		# 
		# Returns a sub-list.
		# 
		# @param x A list from which you would like to get a sub-list.
		# @param listEntryNames A vector of names which specify the entries of the sub-list to return.
		# 
		.getSubListByNames = function(x, listEntryNames) {
			"Returns a sub-list."
			if (!is.list(x)) {
				stop(C_EXCEPTION_TYPE_RUNTIME_ISSUE, "'x' must be a list")
			}
			
			if (!is.character(listEntryNames)) {
				stop(C_EXCEPTION_TYPE_RUNTIME_ISSUE, "'listEntryNames' must be a character vector")
			}
			
			return(x[which(names(x) %in% listEntryNames)])
		}
	)
)

#'
#' @name FieldSet_names
#' 
#' @title
#' Names of a Field Set Object
#'
#' @description
#' Function to get the names of a \code{\link{FieldSet}} object.
#' 
#' @param x A \code{\link{FieldSet}} object.
#' 
#' @details
#' Returns the names of a field set that can be accessed by the user.
#' 
#' @template return_names
#'
#' @export
#' 
#' @keywords internal
#' 
names.FieldSet <- function(x) {
	return(x$.getVisibleFieldNames())
}

#'
#' @name FieldSet_print
#' 
#' @title
#' Print Field Set Values
#'
#' @description
#' \code{print} prints its \code{\link{FieldSet}} argument and returns it invisibly (via \code{invisible(x)}). 
#' 
#' @param x A \code{\link{FieldSet}} object.
#' @inheritParams param_three_dots
#' 
#' @details
#' Prints the field set.
#' 
#' @export
#' 
#' @keywords internal
#' 
print.FieldSet <- function(x, ...) {
	x$show()
	invisible(x)
}

#'
#' @name ParameterSet_as.data.frame
#' 
#' @title
#' Coerce Parameter Set to a Data Frame
#'
#' @description
#' Returns the \code{ParameterSet} as data frame.
#' 
#' @param x A \code{\link{FieldSet}} object.
#' @inheritParams param_niceColumnNamesEnabled
#' @inheritParams param_includeAllParameters
#' @inheritParams param_three_dots
#' 
#' @details
#' Coerces the parameter set to a data frame.
#' 
#' @template return_dataframe
#' 
#' @export
#' 
#' @keywords internal
#'  
as.data.frame.ParameterSet <- function(x, row.names = NULL, 
		optional = FALSE, niceColumnNamesEnabled = FALSE, includeAllParameters = FALSE, ...) {	
		
	.warnInCaseOfUnknownArguments(functionName = "as.data.frame", ...)
	
	return(x$.getAsDataFrame(parameterNames = NULL, 
			niceColumnNamesEnabled = niceColumnNamesEnabled, includeAllParameters = includeAllParameters))
}

#'
#' @name FrameSet_as.matrix
#' 
#' @title
#' Coerce Frame Set to a Matrix
#'
#' @description
#' Returns the \code{FrameSet} as matrix.
#' 
#' @param x A \code{\link{FieldSet}} object.
#' @param enforceRowNames If \code{TRUE}, row names will be created 
#'        depending on the object type, default is \code{TRUE}.
#' @inheritParams param_niceColumnNamesEnabled
#' @inheritParams param_three_dots
#' 
#' @details
#' Coerces the frame set to a matrix.
#' 
#' @template return_matrix
#' 
#' @export
#' 
#' @keywords internal
#' 
as.matrix.FieldSet <- function(x, ..., enforceRowNames = TRUE, niceColumnNamesEnabled = TRUE) {
	dataFrame <- as.data.frame(x, niceColumnNamesEnabled = niceColumnNamesEnabled)
	result <- as.matrix(dataFrame)
	
	if (nrow(result) == 0) {
		return(result)
	}
	
	# sample size or power object
	if (.isTrialDesignPlan(x)) {
		dimnames(result)[[1]] <- paste("  ", c(1:nrow(dataFrame)))
		return(result)
	} 
	
	if (inherits(x, "PowerAndAverageSampleNumberResult")) {
		dimnames(result)[[1]] <- rep("", nrow(dataFrame))
		return(result)
	} 
	
	if (inherits(x, "AnalysisResults")) {
		dfDesign <- as.data.frame(x$.design, niceColumnNamesEnabled = niceColumnNamesEnabled)
		dfStageResults <- as.data.frame(x$.stageResults, niceColumnNamesEnabled = niceColumnNamesEnabled)
		dfStageResults <- dfStageResults[!is.na(dfStageResults[, grep("(test statistic)|(testStatistics)", colnames(dfStageResults))]), ]
		if (length(intersect(names(dfDesign), names(dfStageResults))) == 1) {
			dfTemp <- merge(dfDesign, dfStageResults)
			if (length(intersect(names(dfTemp), names(dataFrame))) >= 1) {
				dataFrame <- merge(dfTemp, dataFrame, all.x = FALSE, all.y = TRUE)
				result <- as.matrix(dataFrame)
			}
		} else if (length(intersect(names(dfStageResults), names(dataFrame))) >= 1) {
			dataFrame <- merge(dfStageResults, dataFrame, all.x = FALSE, all.y = TRUE)
			result <- as.matrix(dataFrame)
		}
	}
	
	if (is.na(enforceRowNames) || isTRUE(enforceRowNames)) {
		for (paramName in c("stage", "stages", "Stage", "Stages")) {
			paramNames <- colnames(result)
			if (paramName %in% paramNames) {
				stageNumbers <- result[, paramName]
				if (!is.null(stageNumbers) && length(stageNumbers) > 0) {
					dimnames(result)[[1]] <- rep("", nrow(result))
					result <- result[, c(paramName, paramNames[paramNames != paramName])]
					return(result)
				}
			}
		}
	}
	
	return(result)
}

#'
#' @name ParameterSet_summary
#' 
#' @title
#' Parameter Set Summary
#'
#' @description
#' Displays a summary of \code{\link{ParameterSet}} object.
#' 
#' @param object A \code{\link{ParameterSet}} object.
#' @inheritParams param_digits
#' @inheritParams param_three_dots
#' 
#' @details
#' Summarizes the parameters and results of a parameter set.
#' 
#' @template details_summary
#' 
#' @template return_object_summary
#' @template how_to_get_help_for_generics
#' 
#' @export
#' 
#' @keywords internal
#' 
summary.ParameterSet <- function(object, ..., type = 1, digits = NA_integer_) {
	
	.warnInCaseOfUnknownArguments(functionName = "summary", ...)
	
	if (type == 1 && inherits(object, "SummaryFactory")) {
		return(object)
	}
	
	if (type == 1 && (inherits(object, "TrialDesign") || inherits(object, "TrialDesignPlan") || 
			inherits(object, "SimulationResults") || inherits(object, "AnalysisResults") || 
			inherits(object, "TrialDesignCharacteristics"))) {
		return(.createSummary(object, digits = digits))	
	}
	
	object$show(showType = 2)
	object$.cat("\n")
	
	if (!is.null(object[[".piecewiseSurvivalTim"]])) {
		object$.piecewiseSurvivalTime$show()
		object$.cat("\n")
	}
	
	if (!is.null(object[[".accrualTime"]])) {
		object$.accrualTime$show()
		object$.cat("\n")
	}

	object$.cat(object$.toString(startWithUpperCase = TRUE), " table:\n", heading = 1)
	parametersToShow <- object$.getParametersToShow()
	for (parameter in parametersToShow) {
		if (length(object[[parameter]]) == 1) {
			parametersToShow <- parametersToShow[parametersToShow != parameter]
		}
	}
	object$.printAsDataFrame(parameterNames = parametersToShow, niceColumnNamesEnabled = TRUE)
	invisible(object)
}

#'
#' @name ParameterSet_print
#' 
#' @title
#' Print Parameter Set Values
#' 
#' @description
#' \code{print} prints its \code{ParameterSet} argument and returns it invisibly (via \code{invisible(x)}). 
#' 
#' @param x The \code{\link{ParameterSet}} object to print.
#' @param markdown If \code{TRUE}, the object \code{x} will be printed using markdown syntax; 
#'        normal representation will be used otherwise (default is \code{FALSE})
#' @inheritParams param_three_dots
#' 
#' @details
#' Prints the parameters and results of a parameter set.
#'
#' @export
#' 
#' @keywords internal
#' 
print.ParameterSet <- function(x, ..., markdown = FALSE) {
	if (markdown) {
		x$.catMarkdownText()
		return(invisible(x))
	}
	
	x$show()
	invisible(x)
}

#' 
#' @title
#' Parameter Set Plotting
#' 
#' @description
#' Plots an object that inherits from class \code{\link{ParameterSet}}.
#' 
#' @details
#' Generic function to plot a parameter set.
#' 
#' @param x The object that inherits from \code{\link{ParameterSet}}.
#' @param y Not available for this kind of plot (is only defined to be compatible to the generic plot function).
#' @param main The main title.
#' @param xlab The x-axis label.
#' @param ylab The y-axis label.
#' @param type The plot type (default = 1).
#' @inheritParams param_palette
#' @inheritParams param_showSource
#' @inheritParams param_legendPosition
#' @inheritParams param_three_dots_plot
#' 
#' @details
#' Generic function to plot a parameter set.
#' 
#' @template return_object_ggplot
#' 
#' @export
#' 
plot.ParameterSet = function(x, y, ..., main = NA_character_,
		xlab = NA_character_, ylab = NA_character_, type = 1L, palette = "Set1",
		legendPosition = NA_integer_, showSource = FALSE) {
	
	.assertGgplotIsInstalled()
	
	stop(C_EXCEPTION_TYPE_RUNTIME_ISSUE, 
		"sorry, function 'plot' is not implemented yet for class '", class(x), "'")
}
