% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sim.R
\name{sim}
\alias{sim}
\title{Mechanistic Metapopulation Simulator}
\usage{
sim(
  obj,
  time,
  burn = 0,
  return_mu = FALSE,
  cl = NULL,
  progress_bar = TRUE,
  quiet = FALSE
)
}
\arguments{
\item{obj}{\code{sim_data} object created by \code{\link{initialise}} containing all
simulation parameters and necessary data}

\item{time}{positive integer vector of length 1; number of time steps
simulated}

\item{burn}{positive integer vector of length 1; the number of burn-in time
steps that are discarded from the output}

\item{return_mu}{logical vector of length 1; if \code{TRUE} demographic
process return expected values; if \code{FALSE} the \code{\link[stats:Poisson]{rpois}} function
should be used}

\item{cl}{an optional cluster object created by
\code{\link[parallel:makeCluster]{makeCluster}} needed for parallel calculations}

\item{progress_bar}{logical vector of length 1 determines if progress bar
for simulation should be displayed}

\item{quiet}{logical vector of length 1; determines if warnings should
be displayed}
}
\value{
This function returns an object of class \code{sim_results} which is
a list containing the following components:
\itemize{
\item \code{extinction} - \code{TRUE} if population is extinct or \code{FALSE} otherwise
\item \code{simulated_time} - number of simulated time steps without
the burn-in ones
\item \code{N_map} - 3-dimensional array representing spatio-temporal
variability in population numbers. The first two dimensions correspond to
the spatial aspect of the output and the third dimension represents time.
}

In case of a total extinction, a simulation is stopped before reaching
the specified number of time steps. If the population died out before reaching
the \code{burn} threshold, then nothing can be returned and an error occurs.
}
\description{
This function simulates population growth and dispersal providing a given
environmental scenario. All parameters for the simulation must be set
in advance using \code{\link{initialise}}.
}
\details{
This is the main simulation module. It takes the \code{sim_data} object prepared
by \code{\link{initialise}} and runs simulation for a given number of time steps.
The initial (specified by the \code{burn} parameter) time steps are skipped
and discarded from the output. Computations can be done in parallel if
the name of a cluster created by \code{\link[parallel:makeCluster]{makeCluster}}
is provided.

Generally, at each time step, simulation consists of two phases: local
dynamics and dispersal.

Local dynamics (which connects habitat patches in time) is defined by
the function \code{\link{growth}}. This parameter is specified while creating
the \code{obj} using \code{\link{initialise}}, but can be later modified by using
the \code{\link{update}} function. Population growth can be either exponential
or density-dependent, and the regulation is implemented by the use of
Gompertz or Ricker models (with a possibility of providing any other,
user defined function). For every cell, the expected population density
during the next time step is calculated from the corresponding number
of individuals currently present in this cell, and the actual number
of individuals is set by drawing a random number from the Poisson
distribution using this expected value. This procedure introduces a realistic
randomness, however additional levels of random variability can be
incorporated by providing parameters of both demographic and environmental
stochasticity while specifying the \code{sim_data} object using the \code{\link{initialise}}
function (parameters \code{r_sd} and \code{K_sd}, respectively).

To simulate dispersal (which connects habitat patches in space), for each
individual in a given cell, a dispersal distance is randomly drawn from
the dispersal kernel density function. Then, each individual is translocated
to a randomly chosen cell at this distance apart from the current location.
For more details, see the \code{\link{disp}} function.
}
\examples{
\donttest{

# data preparation
library(terra)

n1_small <- rast(system.file("input_maps/n1_small.tif", package = "rangr"))
K_small <- rast(system.file("input_maps/K_small.tif", package = "rangr"))

sim_data <- initialise(
  n1_map = n1_small,
  K_map = K_small,
  r = log(2),
  rate = 1 / 1e3
)

# simulation
sim_1 <- sim(obj = sim_data, time = 20)

# simulation with burned time steps
sim_2 <- sim(obj = sim_data, time = 20, burn = 10)

# example with parallelization
library(parallel)
cl <- makeCluster(2)

# parallelized simulation
sim_3 <- sim(obj = sim_data, time = 20, cl = cl)
stopCluster(cl)


# visualisation
plot(
  sim_1,
  time_points = 20,
  template = sim_data$K_map
)

plot(
  sim_1,
  time_points = c(1, 5, 10, 20),
  template = sim_data$K_map
)

plot(
  sim_1,
  template = sim_data$K_map
)

}

}
\references{
Solymos P, Zawadzki Z (2023). pbapply: Adding Progress Bar to '*apply' Functions. R
package version 1.7-2, \url{https://CRAN.R-project.org/package=pbapply}.
}
\seealso{
\link[=get_observations]{get_observations}
}
