\name{cmaOptimDP}
\alias{cmaOptimDP}
\title{Perform a CMA-ES optimization with constraints (DP).}
\usage{
  cmaOptimDP(cma, fitFunc, isFeasible = function(x) {
        TRUE }, maxDimPrint = 5, iterPrint = 10,
    verbose = 2)
}
\arguments{
  \item{cma}{CMA-ES Java object, already initialized with
  \code{\link{cmaInit}}}

  \item{fitFunc}{a function to be minimized. Signature:
  accepts a vector \code{x}, returns a \code{double}.}

  \item{isFeasible}{[\code{function(x)\{TRUE\}}] a Boolean
  function checking the feasibility of the vector \code{x}.
  The default is to return always \code{TRUE}.}

  \item{maxDimPrint}{[5] how many dimensions of vector
  \code{x} to print in diagnostic output}

  \item{iterPrint}{[10] after how many iterations should
  diagnostic output be printed?}

  \item{verbose}{[2] possible values are 0 (no output), 1,
  2 (much output)}
}
\value{
  \code{res}, a list with diagnostic output from the
  optimization run: \item{\code{sMsg}}{ a string vector
  with all console output from the optimization run.  To
  print it, use: \code{ cat(sMsg) } or \code{ for (x in
  sMsg) cat(x) } } \item{\code{bestX}}{ vector of length
  \code{dimension} with the best-ever solution X }
  \item{\code{bestFitness}}{ the corresponding best-ever
  fitness } \item{\code{bestEvalNum}}{ the fitness function
  evaluation number which gave this best-ever result }
  \item{\code{nIter}}{ number of iterations }
  \item{\code{fitnessVec}}{ vector of length \code{nIter}:
  the best fitness after each iteration }
  \item{\code{xMat}}{ (\code{nIter x dimension})-matrix:
  \code{xMat[i,]} is the best solution X after iteration
  \code{i} } \item{\code{cfe}}{ number of constraint
  function evaluations (\code{isFeasible}) }
  \item{\code{ffe}}{ number of fitness function evaluations
  (\code{fitFunc}) }
}
\description{
  The optimization uses DP (death penalty) for handling
  constraint violations: Each time an infeasible individual
  is encountered, it is thrown away and a new individual is
  resampled from the CMA distribution.
}
\details{
  This functions loops through iterations (generations)
  until a \link[=cmaSetStopFitness]{stop condition} is met:
  In each iteration, a population is sampled (infeasible
  individuals are replaced via Java function
  \code{resampleSingle}), its fitness vector is evaluated
  and the CMA distribution is updated according to this
  fitness vector.
}
\note{
  If your fitness function depends on other parameters
  besides \code{x}, then encapsulate it in a new function
  \code{fitFunc} at a place where the other parameters are
  accessible and rely on R's mechanism to locate the other
  parameters in the environment surrounding \code{fitFunc}:
  \preformatted{ par1 <- someObject; } \preformatted{
  fitFunc <- function(x) { myFuncWithOtherPars(x,par1); } }
}
\examples{
## demo/demoCMA2.R: demo usage of package rCMA.
##
## After doing the unconstrained sphere (as in demoCMA1.r, for later reference in plot), 
## the constrained sphere problem TR2 is solved. 
##
## Note that in this second case, the optimimum lies exactly at the boundary 
## of the feasible region: res2$bestX=c(1,1).
##
## This script does exactly the same as class CMAExampleConstr in cma_jAll.jar,
## but it allows to define the functions fitFunc and isFeasible on the R side. 
## They can be replaced by arbitrary other R functions, which may depend on other 
## R variables as well. 
## 
fitFunc <- function(x) {  sum(x*x); }
isFeasible <- function(x) {  (sum(x) - length(x)) > 0;  }
dimension = 2;

cma <- cmaNew(propFile="CMAEvolutionStrategy.properties");
cmaInit(cma,seed=42,dimension=dimension,initialX=1.5, initialStandardDeviations=0.2);
res1 = cmaOptimDP(cma,fitFunc,iterPrint=30);

cma <- cmaNew(propFile="CMAEvolutionStrategy.properties");
cmaInit(cma,seed=42,dimension=dimension,initialX=1.5, initialStandardDeviations=0.2);
res2 = cmaOptimDP(cma,fitFunc,isFeasible,iterPrint=30);

fTarget1 = 0;         
fTarget2 = dimension; 
plot(res1$fitnessVec-fTarget1,type="l",log="y",xlim=c(1,max(res1$nIter,res2$nIter))
    ,xlab="Iteration",ylab="Distance to target fitness");
lines(res2$fitnessVec-fTarget2,col="red");
str(res2);



}
\author{
  Wolfgang Konen, FHK, 2013
}
\seealso{
  \code{\link{cmaNew}}, \code{\link{cmaInit}}
}

