% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ggplot_snpasso.R
\name{ggplot_snpasso}
\alias{ggplot_snpasso}
\title{Plot SNP associations}
\usage{
ggplot_snpasso(
  scan1output,
  snpinfo,
  genes = NULL,
  lodcolumn = 1,
  show_all_snps = TRUE,
  drop_hilit = NA,
  col_hilit = "violetred",
  col = "darkslateblue",
  ylim = NULL,
  gap = 25,
  minlod = 0,
  bgcolor = "gray90",
  altbgcolor = "gray85",
  ...
)
}
\arguments{
\item{scan1output}{Output of \code{\link[qtl2]{scan1}}.  Should
contain an attribute, \code{"snpinfo"}, as when
\code{\link[qtl2]{scan1}} are run with SNP probabilities
produced by \code{\link[qtl2]{genoprob_to_snpprob}}.}

\item{snpinfo}{Data frame with SNP information with the following
    columns (the last three are generally derived from with
    \code{\link[qtl2]{index_snps}}):
\itemize{
\item \code{chr} - Character string or factor with chromosome
\item \code{pos} - Position (in same units as in the \code{"map"}
    attribute in \code{genoprobs}.
\item \code{sdp} - Strain distribution pattern: an integer, between
    1 and \eqn{2^n - 2} where \eqn{n} is the number of strains, whose
    binary encoding indicates the founder genotypes
\item \code{snp} - Character string with SNP identifier (if
    missing, the rownames are used).
\item \code{index} - Indices that indicate equivalent
    groups of SNPs.
\item \code{intervals} - Indexes that indicate which marker
    intervals the SNPs reside.
\item \code{on_map} - Indicate whether SNP coincides with a marker
    in the \code{genoprobs}
}}

\item{genes}{Optional data frame containing gene information for
the region, with columns `start` and `stop` in Mbp, `strand`
(as `"-"`, `"+"`, or `NA`), and `Name`. If included, a
two-panel plot is produced, with SNP associations above and
gene locations below.}

\item{lodcolumn}{LOD score column to plot (a numeric index, or a
character string for a column name). One or more value(s) allowed.}

\item{show_all_snps}{If TRUE, expand to show all SNPs.}

\item{drop_hilit}{SNPs with LOD score within this amount of the maximum SNP association will be highlighted.}

\item{col_hilit}{Color of highlighted points}

\item{col}{Color of other points}

\item{ylim}{y-axis limits}

\item{gap}{Gap between chromosomes.}

\item{minlod}{Minimum LOD to display. (Mostly for GWAS, in which
case using `minlod=1` will greatly increase the plotting speed,
since the vast majority of points would be omittted.}

\item{bgcolor}{Background color for the plot.}

\item{altbgcolor}{Background color for alternate chromosomes.}

\item{...}{Additional graphics parameters.}
}
\value{
object of class \code{\link[ggplot2]{ggplot}}.
}
\description{
Plot SNP associations, with possible expansion from distinct snps to all snps.
}
\section{Hidden graphics parameters}{

A number of graphics parameters can be passed via `...`. For
example, `bgcolor` to control the background color and `altbgcolor`
to control the background color on alternate chromosomes.
`cex` for character expansion for the points (default 0.5),
`pch` for the plotting character for the points (default 16),
and `ylim` for y-axis limits.
}

\examples{
\dontrun{
# load example DO data from web
library(qtl2)
file <- paste0("https://raw.githubusercontent.com/rqtl/",
               "qtl2data/master/DOex/DOex.zip")
DOex <- read_cross2(file)

# subset to chr 2
DOex <- DOex[,"2"]

# calculate genotype probabilities and convert to allele probabilities
pr <- calc_genoprob(DOex, error_prob=0.002)
apr <- genoprob_to_alleleprob(pr)

# query function for grabbing info about variants in region
snp_dbfile <- system.file("extdata", "cc_variants_small.sqlite", package="qtl2")
query_variants <- create_variant_query_func(snp_dbfile)

# SNP association scan
out_snps <- scan1snps(apr, DOex$pmap, DOex$pheno, query_func=query_variants,
                      chr=2, start=97, end=98, keep_all_snps=TRUE)

# plot results
ggplot_snpasso(out_snps, snpinfo)

# can also just type autoplot() if ggplot2 attached
library(ggplot2)
autoplot(out_snps, snpinfo)

# plot just subset of distinct SNPs
autoplot(out_snps, snpinfo, show_all_snps=FALSE)

# highlight the top snps (with LOD within 1.5 of max)
autoplot(out_snps, snpinfo, drop_hilit=1.5)

# highlight SDP patterns in SNPs; connect with lines.
autoplot(out_snps, snpinfo, patterns="all",drop_hilit=4)

# highlight top SDP patterns in SNPs; connect with lines.
autoplot(out_snps, snpinfo, patterns="hilit",drop_hilit=4)

# query function for finding genes in region
gene_dbfile <- system.file("extdata", "mouse_genes_small.sqlite", package="qtl2")
query_genes <- create_gene_query_func(gene_dbfile)
genes <- query_genes(2, 97, 98)

# plot SNP association results with gene locations
autoplot(out_snps$lod, out_snps$snpinfo, drop_hilit=1.5, genes=genes)
}

}
\seealso{
\code{\link{ggplot_scan1}}, \code{\link{ggplot_coef}}
}
