% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gradient.R
\name{GenD}
\alias{GenD}
\title{Numerical derivative matrices with parallel capabilities}
\usage{
GenD(
  FUN,
  x,
  elementwise = NA,
  vectorised = NA,
  multivalued = NA,
  deriv.order = 1L,
  side = 0,
  acc.order = 2L,
  h = NULL,
  zero.tol = sqrt(.Machine$double.eps),
  h0 = NULL,
  control = list(),
  f0 = NULL,
  cores = 1,
  preschedule = TRUE,
  cl = NULL,
  func = NULL,
  method = NULL,
  method.args = list(),
  report = 1L,
  ...
)
}
\arguments{
\item{FUN}{A function returning a numeric scalar or a vector whose derivatives are to be
computed. If the function returns a vector, the output will be a Jacobian.}

\item{x}{Numeric vector or scalar: the point(s) at which the derivative is estimated.
\code{FUN(x)} must be finite.}

\item{elementwise}{Logical: is the domain effectively 1D, i.e. is this a mapping
\eqn{\mathbb{R} \mapsto \mathbb{R}}{R -> R} or
\eqn{\mathbb{R}^n \mapsto \mathbb{R}^n}{R^n -> R^n}. If \code{NA},
compares the output length ot the input length.}

\item{vectorised}{Logical: if \code{TRUE}, the function
is assumed to be vectorised: it will accept a vector of parameters and return
a vector of values of the same length. Use \code{FALSE} or \code{"no"}  for
functions that take vector arguments and return outputs of arbitrary length (for
\eqn{\mathbb{R}^n \mapsto \mathbb{R}^k}{R^n -> R^k} functions). If \code{NA},
checks the output length and assumes vectorisation if it matches the input length;
this check is necessary and potentially slow.}

\item{multivalued}{Logical: if \code{TRUE}, the function is assumed to return vectors longer
than 1. Use \code{FALSE} for element-wise functions. If \code{NA}, attempts inferring it from
the function output.}

\item{deriv.order}{Integer or vector of integers indicating the desired derivative order,
\eqn{\mathrm{d}^m / \mathrm{d}x^m}{d^m/dx^m}, for each element of \code{x}.}

\item{side}{Integer scalar or vector indicating the type of finite difference:
\code{0} for central, \code{1} for forward, and \code{-1} for backward differences.
Central differences are recommended unless computational cost is prohibitive.}

\item{acc.order}{Integer or vector of integers specifying the desired accuracy order
for each element of \code{x}.
The final error will be of the order \eqn{O(h^{\mathrm{acc.order}})}{O(h^acc.order)}.}

\item{h}{Numeric or character specifying the step size(s) for the numerical
difference or a method of automatic step determination (\code{"CR"}, \code{"CRm"},
\code{"DV"}, or \code{"SW"} to be used in \code{\link[=gradstep]{gradstep()}}). The default value is
described in \code{?GenD}.}

\item{zero.tol}{Small positive integer: if \code{abs(x) >= zero.tol}, then, the automatically
guessed step size is relative (\code{x} multiplied by the step), unless an auto-selection
procedure is requested; otherwise, it is absolute.}

\item{h0}{Numeric scalar of vector: initial step size for automatic search with
\code{gradstep()}.}

\item{control}{A named list of tuning parameters passed to \code{gradstep()}.}

\item{f0}{Optional numeric: if provided, used to determine the vectorisation type
to save time. If FUN(x) must be evaluated (e.g. second derivatives), saves one evaluation.}

\item{cores}{Integer specifying the number of CPU cores used for parallel computation.
Recommended to be set to the number of physical cores on the machine minus one.}

\item{preschedule}{Logical: if \code{TRUE}, disables pre-scheduling for \code{mclapply()}
or enables load balancing with \code{parLapplyLB()}. Recommended for functions that
take less than 0.1 s per evaluation.}

\item{cl}{An optional user-supplied \code{cluster} object  (created by \code{makeCluster}
or similar functions). If not \code{NULL}, the code uses \code{parLapply()} (if \code{preschedule}
is \code{TRUE}) or \code{parLapplyLB()} on that cluster on Windows, and \code{mclapply}
(fork cluster) on everything else.}

\item{func}{For compatibility with \code{numDeriv::grad()} only. If instead of
\code{FUN}, \code{func} is used, it will be reassigned to \code{FUN} with a warning.}

\item{method}{For compatibility with \code{numDeriv::grad()} only. Supported values:
\code{"simple"} and \code{"Richardson"}. Non-null values result in a warning.}

\item{method.args}{For compatibility with \code{numDeriv::grad()} only. Check
\code{?numDeriv::grad} for a list of values. Non-empty lists result in a warning.}

\item{report}{Integer for the level of detail in the output. If \code{0},
returns a gradient without any attributes; if \code{1},
attaches the step size and its selection method: \code{2} or higher attaches the full
diagnostic output as an attribute.}

\item{...}{Additional arguments passed to \code{FUN}. Important! Since R does partial
matching for argument names, using arguments like \code{f(x, a)} and passing \code{a = 1}
to \code{\link[=GenD]{GenD()}}, \code{\link[=Grad]{Grad()}}, or \code{\link[=Jacobian]{Jacobian()}} will result in the interpretation \code{acc.order = 1}
because the latter is a named argument of these functions. Either supply the full names of all
similar-looking arguments or name arguments differently (e.g. \code{a0} instead of \code{a}).}
}
\value{
A vector or matrix containing the computed derivatives, structured according
to the dimensionality of \code{x} and \code{FUN}. If \code{FUN} is scalar-valued,
returns a gradient vector. If \code{FUN} is vector-valued, returns a Jacobian matrix.
}
\description{
Computes numerical derivatives of a scalar or vector function using finite-difference methods.
This function serves as a backbone for \code{\link[=Grad]{Grad()}} and \code{\link[=Jacobian]{Jacobian()}}, allowing for detailed control
over the derivative computation process, including order of derivatives, accuracy, and step size.
\code{GenD} is fully vectorised over different coordinates of the function argument,
allowing arbitrary accuracies, sides, and derivative orders for different coordinates.
}
\details{
For computing gradients and Jacobians, use convenience wrappers \code{Jacobian} and \code{Grad}.

If the step size is too large, the slope of the secant poorly estimates the derivative;
if it is too small, it leads to numerical instability due to the function value rounding.

The optimal step size for one-sided differences typically approaches Mach.eps^(1/2)
to balance the Taylor series truncation error with the rounding error due to storing
function values with limited precision. For two-sided differences, it is proportional
to Mach.eps^(1/3). However, selecting the best step size typically requires knowledge
of higher-order derivatives, which may not be readily available. Luckily,
using \code{step = "SW"} invokes a reliable automatic data-driven step-size selection.
Other options include \code{"DV"}, \code{"CR"}, and \code{"CRm"}.
The step size  defaults to \code{1.5e-8} (the square root of machine epsilon)
for \code{x} less than \code{1.5e-8},
\code{(2.2e-16)^(1/(deriv.order + acc.order)) * x} for \code{x > 1}, and interpolates
exponentially between these values for \code{1.5e-8 < x < 1}.

The use of \code{f0} can reduce computation time similar to the use of \code{f.lower}
and \code{f.upper} in \code{uniroot()}.

For convenience, \code{report = 2} overrides \code{diagnostics = FALSE} in the
\code{control}) list.

Unlike \code{numDeriv::grad()} and \code{numDeriv::jacobian()}, this function
fully preserves the names of \code{x} and \code{FUN(x)}.
}
\examples{

# Case 1: Vector argument, vector output
f1 <- sin
g1 <- GenD(FUN = f1, x = 1:100)
g1.true <- cos(1:100)
plot(1:100, g1 - g1.true, main = "Approximation error of d/dx sin(x)")

# Case 2: Vector argument, scalar result
f2 <- function(x) sum(sin(x))
g2    <- GenD(FUN = f2, x = 1:4)
g2.h2 <- Grad(FUN = f2, x = 1:4, h = 7e-6)
g2 - g2.h2  # Tiny differences due to different step sizes
g2.auto <- Grad(FUN = f2, x = 1:4, h = "SW")
g2.full <- Grad(FUN = f2, x = 1:4, h = "SW", report = 2)
print(attr(g2.full, "step.search")$exitcode)  # Success

# Case 3: vector input, vector argument of different length
f3 <- function(x) c(sum(sin(x)), prod(cos(x)))
x3 <- 1:3
j3 <- GenD(f3, x3, multivalued = TRUE)
print(j3)

# Gradients for vectorised functions -- e.g. leaky ReLU
LReLU <- function(x) ifelse(x > 0, x, 0.01*x)
system.time(replicate(10, suppressMessages(GenD(LReLU, runif(30, -1, 1)))))
system.time(replicate(10, suppressMessages(GenD(LReLU, runif(30, -1, 1)))))

# Saving time for slow functions by using pre-computed values
x <- 1:4
finner <- function(x) sin(x*log(abs(x)+1))
fouter <- function(x) integrate(finner, 0, x, rel.tol = 1e-12, abs.tol = 0)$value
# The outer function is non-vectorised
fslow <- function(x) {Sys.sleep(0.01); fouter(x)}
f0 <- sapply(x, fouter)
system.time(GenD(fslow, x, side = 1, acc.order = 2, f0 = f0))
# Now, with extra checks, it will be slower
system.time(GenD(fslow, x, side = 1, acc.order = 2))
}
\seealso{
\code{\link[=gradstep]{gradstep()}} for automatic step-size selection.
}
