% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/analyze_objects.R
\name{analyze_objects}
\alias{analyze_objects}
\alias{plot.anal_obj}
\title{Analyzes objects in an image}
\usage{
analyze_objects(
  img,
  foreground = NULL,
  background = NULL,
  pattern = NULL,
  img_pattern = NULL,
  parallel = FALSE,
  workers = NULL,
  watershed = TRUE,
  resize = FALSE,
  trim = FALSE,
  fill_hull = FALSE,
  filter = FALSE,
  invert = FALSE,
  object_size = "medium",
  index = "NB",
  my_index = NULL,
  object_index = NULL,
  threshold = "Otsu",
  tolerance = NULL,
  extension = NULL,
  lower_size = NULL,
  upper_size = NULL,
  topn_lower = NULL,
  topn_upper = NULL,
  lower_eccent = NULL,
  upper_eccent = NULL,
  randomize = TRUE,
  nrows = 2000,
  show_image = TRUE,
  show_original = TRUE,
  show_chull = FALSE,
  show_contour = TRUE,
  contour_col = "red",
  contour_size = 1,
  show_background = TRUE,
  show_segmentation = FALSE,
  col_foreground = NULL,
  col_background = NULL,
  marker = FALSE,
  marker_col = NULL,
  marker_size = NULL,
  save_image = FALSE,
  prefix = "proc_",
  dir_original = NULL,
  dir_processed = NULL,
  verbose = TRUE
)

\method{plot}{anal_obj}(x, facet = FALSE, ...)
}
\arguments{
\item{img}{The image to be analyzed.}

\item{foreground}{A color palette of the foreground (optional).}

\item{background}{A color palette of the background (optional).}

\item{pattern}{A pattern of file name used to identify images to be imported.
For example, if \code{pattern = "im"} all images in the current working
directory that the name matches the pattern (e.g., img1.-, image1.-, im2.-)
will be imported as a list. Providing any number as pattern (e.g., \code{pattern = "1"}) will select images that are named as 1.-, 2.-, and so on. An error
will be returned if the pattern matches any file that is not supported
(e.g., img1.pdf).}

\item{img_pattern}{Deprecated. Use \code{pattern} instead.}

\item{parallel}{If \code{TRUE} processes the images asynchronously (in parallel)
in separate R sessions running in the background on the same machine. It
may speed up the processing time, especially when \code{pattern} is used is
informed. When \code{object_index} is informed, multiple sections will be used
to extract the RGB values for each object in the image. This may
significantly speed up processing time when an image has lots of objects
(say >1000).}

\item{workers}{A positive numeric scalar or a function specifying the number
of parallel processes that can be active at the same time. By default, the
number of sections is set up to 50\% of available cores.}

\item{watershed}{If \code{TRUE} (default) performs watershed-based object
detection. This will detect objects even when they are touching one other.
If \code{FALSE}, all pixels for each connected set of foreground pixels are set
to a unique object. This is faster but is not able to segment touching
objects.}

\item{resize}{Resize the image before processing? Defaults to \code{FALSE}. Use a
numeric value of range 0-100 (proportion of the size of the original
image).}

\item{trim}{Number of pixels removed from edges in the analysis. The edges of
images are often shaded, which can affect image analysis. The edges of
images can be removed by specifying the number of pixels. Defaults to
\code{FALSE} (no trimmed edges).}

\item{fill_hull}{Fill holes in the binary image? Defaults to \code{FALSE}. This is
useful to fill holes in objects that have portions with a color similar to
the background. IMPORTANT: Objects touching each other can be combined into
one single object, which may underestimate the number of objects in an
image.}

\item{filter}{Performs median filtering after image processing? defaults to
\code{FALSE}. See more at \code{\link[=image_filter]{image_filter()}}.}

\item{invert}{Inverts the binary image, if desired. This is useful to process
images with black background. Defaults to \code{FALSE}.}

\item{object_size}{The size of the object. Used to automatically set up
\code{tolerance} and \code{extension} parameters. One of the following. \code{"small"}
(e.g, wheat grains), \code{"medium"} (e.g, soybean grains), \code{"large"}(e.g,
peanut grains), and \code{"elarge"} (e.g, soybean pods)`.}

\item{index, my_index}{A character value specifying the target mode for
conversion to binary image when \code{foreground} and \code{background} are not
declared. Defaults to \code{"NB"} (normalized blue). See \code{\link[=image_index]{image_index()}} for
more details.}

\item{object_index}{Defaults to \code{FALSE}. If an index is informed, the average
value for each object is returned. It can be the R, G, and B values or any
operation involving them, e.g., \code{object_index = "R/B"}. In this case, it
will return for each object in the image, the average value of the R/B
ratio. Use \code{\link[=pliman_indexes_eq]{pliman_indexes_eq()}} to see the equations of available indexes.}

\item{threshold}{A numeric value for the segmentation threshold.  By default,
a threshold value based on Otsu's method is used to reduce the grayscale
image to a binary image.}

\item{tolerance}{The minimum height of the object in the units of image
intensity between its highest point (seed) and the point where it contacts
another object (checked for every contact pixel). If the height is smaller
than the tolerance, the object will be combined with one of its neighbors,
which is the highest.}

\item{extension}{Radius of the neighborhood in pixels for the detection of
neighboring objects. Higher value smooths out small objects.}

\item{lower_size, upper_size}{Lower and upper limits for size for the image
analysis. Plant images often contain dirt and dust. To prevent dust from
affecting the image analysis, objects with lesser than 10\% of the mean of
all objects are removed. Upper limit is set to \code{NULL}, i.e., no upper
limit used. One can set a known area or use \code{lower_limit = 0} to select all
objects (not advised). Objects that matches the size of a given range of
sizes can be selected by setting up the two arguments. For example, if
\code{lower_size = 120} and \code{upper_size = 140}, objects with size greater than
or equal 120 and less than or equal 140 will be considered.}

\item{topn_lower, topn_upper}{Select the top \code{n} objects based on its area.
\code{topn_lower} selects the \code{n} elements with the smallest area whereas
\code{topn_upper} selects the \code{n} objects with the largest area.}

\item{lower_eccent, upper_eccent}{Lower and upper limit for object
eccentricity for the image analysis. Users may use these arguments to
remove objects such as square papers for scale (low eccentricity) or cut
petioles (high eccentricity) from the images. Defaults to \code{NULL} (i.e., no
lower and upper limits).}

\item{randomize}{Randomize the lines before training the model?}

\item{nrows}{The number of lines to be used in training step. Defaults to
2000.}

\item{show_image}{Show image after processing?}

\item{show_original}{Show the count objects in the original image?}

\item{show_chull}{Show the convex hull around the objects? Defaults to
\code{FALSE}.}

\item{show_contour}{Show a contour line around the objects? Defaults
to \code{TRUE}.}

\item{contour_col, contour_size}{The color and size for the contour line
around objects. Defaults to \code{contour_col = "red"} and \code{contour_size = 1}.}

\item{show_background}{Show the background? Defaults to \code{TRUE}. A white
background is shown by default when \code{show_original = FALSE}.}

\item{show_segmentation}{Shows the object segmentation colored with random
permutations. Defaults to \code{FALSE}.}

\item{col_foreground, col_background}{Foreground and background color after
image processing. Defaults to \code{NULL}, in which \code{"black"}, and \code{"white"} are
used, respectively.}

\item{marker, marker_col, marker_size}{The type, color and size of the object
marker. Defaults to \code{NULL}, which plots the object id. Use \code{marker = "point"} to show a point in each object or \code{marker = FALSE} to omit object
marker.}

\item{save_image}{Save the image after processing? The image is saved in the
current working directory named as \verb{proc_*} where \code{*} is the image name
given in \code{img}.}

\item{prefix}{The prefix to be included in the processed images. Defaults to
\code{"proc_"}.}

\item{dir_original, dir_processed}{The directory containing the original and
processed images. Defaults to \code{NULL}. In this case, the function will
search for the image \code{img} in the current working directory. After
processing, when \code{save_image = TRUE}, the processed image will be also
saved in such a directory. It can be either a full path, e.g.,
\code{"C:/Desktop/imgs"}, or a subfolder within the current working directory,
e.g., \code{"/imgs"}.}

\item{verbose}{If \code{TRUE} (default) a summary is shown in the console.}

\item{x}{An object of class \code{anal_obj}.}

\item{facet}{Create a facet plot for each object. Defaults to \code{FALSE}.}

\item{...}{Currently not used}
}
\value{
\code{analyze_objects()} returns a list with the following objects:
\itemize{
\item \code{results} A data frame with the following variables for each object in the
image:
\itemize{
\item \code{id}:  object identification.
\item \code{x},\code{y}:  x and y coordinates for the center of mass of the object.
\item \code{area}:  area of the object (in pixels).
\item \code{area_ch}:  the area of the convex hull around object (in pixels).
\item \code{perimeter}: perimeter (in pixels).
\item \code{radius_min}, \code{radius_mean}, and \code{radius_max}: The minimum, mean, and
maximum radius (in pixels), respectively.
\item \code{radius_sd}: standard deviation of the mean radius (in pixels).
\item \code{radius_ratio}: radius ratio given by \code{radius_max / radius_min}.
\item \code{diam_min}, \code{diam_mean}, and \code{diam_max}: The minimum, mean, and
maximum diameter (in pixels), respectively.
\item \code{major_axis}, \code{minor_axis}: elliptical fit for major and minor axes (in
pixels).
\item \code{eccentricity}: elliptical eccentricity defined by
sqrt(1-minoraxis^2/majoraxis^2). Circle eccentricity is 0 and straight
line eccentricity is 1.
\item \code{theta}: object angle (in radians).
\item \code{solidity}: object solidity given by \code{area / area_ch}.
\item \code{circularity}: the object circularity given by \eqn{4*pi *(area /
    perimeter^2)}.
}
\item \code{statistics}: A data frame with the summary statistics for the area of the
objects.
\item \code{count}: If \code{pattern} is used, shows the number of objects in each image.
\item \code{object_rgb}: If \code{object_index} is used, returns the R, G, and B values
for each pixel of each object.
\item \code{object_index}: If \code{object_index} is used, returns the index computed for
each object.
}

\code{plot.anal_obj()} returns a \code{trellis} object containing the
distribution of the pixels, optionally  for each object when \code{facet = TRUE}
is used.
}
\description{
\itemize{
\item \code{\link[=analyze_objects]{analyze_objects()}} provides tools for counting and extracting object
features (e.g., area, perimeter, radius, pixel intensity) in an image. See
more at \strong{Details} section.
\item \code{\link[=plot.anal_obj]{plot.anal_obj()}} Produces an histogram for the R, G, and B values when
argument \code{object_index} is used in the function \code{\link[=analyze_objects]{analyze_objects()}}.
}
}
\details{
A binary image is first generated to segment the foreground and background.
The argument \code{index} is useful to choose a proper index to segment the image
(see \code{\link[=image_binary]{image_binary()}} for more details). Then, the number of objects in the
foreground is counted. By setting up arguments such as \code{lower_size},
\code{upper_size} it is possible to set a threshold for lower and upper sizes of
the objects, respectively. The argument \code{object_size} can be used to set up
pre-defined values of \code{tolerance} and \code{extension} depending on the image
resolution. This will influence the watershed-based object segmentation. Users
can also tune-up \code{tolerance} and \code{extension} explicitly to a better precision
of watershed segmentation.

If \code{watershed = FALSE} is used, all pixels for each connected set of
foreground pixels in \code{img} are set to a unique object. This is faster
(specially for a large number of objects) but is not able to segment touching
objects.

If color palettes samples are provided, a general
linear model (binomial family) fitted to the RGB values is used to segment
fore- and background.

By using \code{pattern} it is possible to process several images with common
pattern names that are stored in the current working directory or in the
subdirectory informed in \code{dir_original}'. To speed up the computation time,
one can set \code{parallel = TRUE}.
}
\examples{
\donttest{
library(pliman)
img <- image_pliman("soybean_touch.jpg")
obj <- analyze_objects(img)
obj$statistics

# Enumerate the objects in the original image
# Return the top-5 grains with the largest area

top <-
 analyze_objects(img,
                 marker = "id",
                 topn_upper = 5)
top$results
}

\donttest{
library(pliman)

img <- image_pliman("soy_green.jpg")
# Segment the foreground (grains) using the normalized blue index (NB)
# Shows the average value of the blue index in each object

rgb <-
   analyze_objects(img,
                   marker = "id",
                   index = "NB", # default
                   object_index = "B")
plot(rgb)
plot(rgb, facet = TRUE)
}
}
\references{
Gupta, S., Rosenthal, D. M., Stinchcombe, J. R., & Baucom, R. S. (2020). The
remarkable morphological diversity of leaf shape in sweet potato (Ipomoea
batatas): the influence of genetics, environment, and G×E. New Phytologist,
225(5), 2183–2195. \doi{10.1111/NPH.16286}

Lee, Y., & Lim, W. (2017). Shoelace Formula: Connecting the Area of a Polygon
and the Vector Cross Product. The Mathematics Teacher, 110(8), 631–636.
\doi{10.5951/mathteacher.110.8.0631}
}
\author{
Tiago Olivoto \email{tiagoolivoto@gmail.com}
}
