% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/survpred_prclmm.R
\name{survpred_prclmm}
\alias{survpred_prclmm}
\title{Compute the predicted survival probabilities obtained
from the PRC models}
\usage{
survpred_prclmm(step1, step2, step3, times = 1, new.longdata = NULL,
  new.basecovs = NULL, keep.ranef = FALSE)
}
\arguments{
\item{step1}{the output of \code{\link{fit_lmms}} (step 1
of the estimation of PRC-LMM)}

\item{step2}{the output of \code{\link{summarize_lmms}} (step 2
of the estimation of PRC-LMM)}

\item{step3}{the output of \code{\link{fit_prclmm}} (step 3
of the estimation of PRC-LMM)}

\item{times}{numeric vector with the time points at which
to estimate the time-dependent AUC}

\item{new.longdata}{longitudinal data if you want to compute 
predictions for new subjects on which the model was not trained.
Default is \code{NULL}}

\item{new.basecovs}{a dataframe with baseline covariates for the
new subjects for which predictions are to be computed. Only needed 
if baseline covariates where included in step 3 and 
\code{new.longdata} is specified. Default is \code{NULL}}

\item{keep.ranef}{should a data frame with the predicted random 
effects be included in the output? Default is \code{FALSE}}
}
\value{
A list containing the function call (\code{call}),
a data frame with the predicted survival probabilities
computed at the supplied time points (\code{predicted_survival}),
and if \code{keep.ranef = TRUE} also the predicted random effects
\code{predicted_ranefs}.
}
\description{
This function computes the predictive survival probabilities 
for the for the PRC-LMM model proposed 
in Signorelli et al. (2021, in review)
}
\examples{
# generate example data
set.seed(1234)
p = 4 # number of longitudinal predictors
simdata = simulate_prclmm_data(n = 100, p = p, p.relev = 2, 
             seed = 123, t.values = c(0, 0.2, 0.5, 1, 1.5, 2))
             
# step 1 of PRC-LMM: estimate the LMMs
y.names = paste('marker', 1:p, sep = '')
step1 = fit_lmms(y.names = y.names, 
                 fixefs = ~ age, ranefs = ~ age | id, 
                 long.data = simdata$long.data, 
                 surv.data = simdata$surv.data,
                 t.from.base = t.from.base,
                 n.boots = 0)
                 
# step 2 of PRC-LMM: compute the summaries 
# of the longitudinal outcomes
step2 = summarize_lmms(object = step1)

# step 3 of PRC-LMM: fit the penalized Cox models
step3 = fit_prclmm(object = step2, surv.data = simdata$surv.data,
                   baseline.covs = ~ baseline.age,
                   penalty = 'ridge')
                   
# predict survival probabilities at times 1, 2, 3
surv.probs = survpred_prclmm(step1, step2, step3, times = 1:3)
head(surv.probs$predicted_survival)

# predict survival probabilities for new subjects:
temp = simulate_prclmm_data(n = 10, p = p, p.relev = 2, 
      seed = 321, t.values = c(0, 0.2, 0.5, 1, 1.5, 2))
new.longdata = temp$long.data
new.basecovs = temp$surv.data[ , 1:2]
surv.probs.new = survpred_prclmm(step1, step2, step3, 
                     times = 1:3,
                     new.longdata = new.longdata,
                     new.basecovs = new.basecovs)
head(surv.probs.new$predicted_survival)
}
\references{
Signorelli, M., Spitali, P., Al-Khalili Szigyarto, C, 
The MARK-MD Consortium, Tsonaka, R. (2021). 
Penalized regression calibration: a method for the prediction 
of survival outcomes using complex longitudinal and 
high-dimensional data. arXiv preprint: arXiv:2101.04426.
}
\seealso{
\code{\link{fit_lmms}} (step 1),
\code{\link{summarize_lmms}} (step 2) and 
\code{\link{fit_prclmm}} (step 3)
}
\author{
Mirko Signorelli
}
